(ns farbetter.utils
  (:refer-clojure :exclude [long])
  (:require
   #?(:cljs cljsjs.long)
   #?(:cljs [cljs.pprint :refer [pprint]])
   #?(:cljs [cljs-time.extend]) ;; to make = work as expected
   [#?(:clj clj-time.core :cljs cljs-time.core) :as t]
   [#?(:clj clj-time.format :cljs cljs-time.format) :as f]
   #?(:clj [clj-uuid :as clj-uuid])
   [#?(:clj clojure.core.async :cljs cljs.core.async)
    :refer [alts! take! timeout #?@(:clj [go <!!])]]
   [clojure.string :as string :refer [join]]
   #?(:clj [clojure.test :refer [is]] :cljs [cljs.test :as test])
   [cognitect.transit :as transit]
   #?(:cljs [murmur])
   #?(:clj [puget.printer :refer [cprint]])
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debug errorf infof]])
  #?(:cljs
     (:require-macros
      [cljs.core.async.macros :refer [go]]
      [cljs.test :refer [async is]]
      [farbetter.utils :refer [inspect sym-map throws]]))
  #?(:clj
     (:import [com.google.common.hash Hashing]
              [com.google.common.primitives UnsignedLong]
              [java.io ByteArrayInputStream ByteArrayOutputStream]
              [java.nio.charset StandardCharsets]
              [java.util UUID]
              [org.joda.time DateTime])
     :cljs
     (:import [goog.date Date DateTime UtcDateTime])))


;; TODO: Add schemas to all functions

#?(:cljs (def class type))

(s/defn throw-far-error :- nil
  "Throws an ex-info error with the specified description, type, subtype,
  and error map. Includes the error map in the description thrown."
  [desc :- s/Str
   error-type :- s/Keyword
   error-subtype :- s/Keyword
   error-map :- {s/Any s/Any}]
  (let [error-map (merge error-map
                         {:type error-type :subtype error-subtype})
        desc (cond-> desc
               (not= "." (str (last desc))) (str "."))]
    (throw (ex-info (str desc " error-map: " error-map)
                    error-map))))

;;;;;;;;;;;;;;;;;;;; Longs ;;;;;;;;;;;;;;;;;;;;

#?(:cljs (def Long js/Long))

(s/defn long? :- s/Bool
  [x :- s/Any]
  (if x
    (boolean (= Long (class x)))
    false))

(s/defn long :- Long
  [x :- s/Any]
  (when-not (nil? x)
    #?(:clj (Long. x)
       :cljs (Long.fromValue x))))

(s/defn long= :- s/Bool
  [a :- s/Any
   b :- s/Any]
  #?(:clj (= a b)
     :cljs (.equals a b)))

#?(:cljs (extend-type Long
           IEquiv
           (-equiv [l other]
             (long= l other))))

(defn throw-long->int-err [l]
  (throw-far-error (str "Cannot convert long `" l "` to int.")
                   :illegal-argument :not-an-integer
                   {:input l
                    :class-of-input (#?(:clj class :cljs type) l)}))

(s/defn long->int :- s/Int
  [l :- Long]
  #?(:clj (if (and (<= l 2147483647) (>= l -2147483648))
            (.intValue l)
            (throw-long->int-err l))
     :cljs (if (and (.lte l 2147483647) (.gte l -2147483648))
             (.toInt l)
             (throw-long->int-err l))))

(s/defn long->str :- s/Str
  [l :- Long]
  #?(:clj
     (Long/toString l)
     :cljs
     (.toString l)))

(s/defn long->float :- s/Num
  [l :- Long]
  #?(:clj
     (float l)
     :cljs
     (.toNumber l)))

(s/defn long->double :- s/Num
  [l :- Long]
  #?(:clj
     (double l)
     :cljs
     (.toNumber l)))

(s/defn long->hex-str :- s/Str
  [l :- Long]
  #?(:clj
     (Long/toHexString l)
     :cljs
     (-> (.toUnsigned l)
         (.toString 16))))

(s/defn str->long :- Long
  [s :- s/Str]
  #?(:clj
     (Long/decode s)
     :cljs
     (.fromString Long s)))

(s/defn hex-str->long :- Long
  [hex-str :- s/Str]
  #?(:clj
     (.longValue (UnsignedLong/valueOf hex-str 16))
     :cljs
     (.fromString Long hex-str 16)))

(def FarNum (s/if long?
              Long
              s/Num))

;;;;;;;;;;;;;;;; Handy utilities

;; dissoc-in function was stolen from:
;; https://github.com/clojure/core.incubator/blob/master/src
;; /main/clojure/clojure/core/incubator.clj
(defn dissoc-in
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))

;;;;;;;;;;; Exception handling and testing

(s/defn negative? :- s/Bool
  [n :- FarNum]
  #?(:clj (neg? n)
     :cljs (if (long? n)
             (.isNegative n)
             (neg? n))))

(s/defn invert
  [n :- FarNum]
  #?(:clj (- n)
     :cljs (if (long? n)
             (.subtract (.-ZERO js/Long) n)
             (- n))))

(s/defn abs :- FarNum
  "Returns the absolute value of n"
  [n :- FarNum]
  (if (negative? n)
    (invert n)
    n))

(s/defn within?
  "Tests if the actual value is within error-margin of the
  expected value. Actual and expected parameters must be numeric."
  [error-margin :- FarNum
   expected :- FarNum
   actual :- FarNum]
  (let [neg-inf #?(:clj
                   (Double/NEGATIVE_INFINITY)
                   :cljs
                   (.-NEGATIVE_INFINITY js/Number))
        pos-inf #?(:clj
                   (Double/POSITIVE_INFINITY)
                   :cljs
                   (.-POSITIVE_INFINITY js/Number))
        eq? (fn [a b]
              #?(:clj (= a b)
                 :cljs (if (long? a)
                         (.eq a b)
                         (if (long? b)
                           (.eq b a)
                           (= a b)))))
        lte? (fn [a b]
               #?(:clj (<= a b)
                  :cljs (if (long? a)
                          (.lte a b)
                          (if (long? b)
                            (.lte b a)
                            (<= a b)))))
        sub (fn [a b]
              #?(:clj (- a b)
                 :cljs (if (long? a)
                         (.subtract a b)
                         (if (long? b)
                           (.subtract b a)
                           (- a b)))))]
    (cond
      (eq? expected neg-inf) (= actual neg-inf)
      (eq? expected pos-inf) (= actual pos-inf)
      :else (let [diff (sub actual expected)
                  abs-diff (abs diff)]
              (lte? abs-diff error-margin)))))

;;;;;;;;;;; Environment variables (Clojure only, no CLJS)

#?(:clj
   (defn construct-java-type-from-str [klass s]
     (.newInstance
      (.getConstructor klass (into-array java.lang.Class [java.lang.String]))
      (object-array [s]))))
#?(:clj
   (defn construct-from-str [var-type var-value]
     (if (= clojure.lang.Keyword var-type)
       (keyword var-value)
       (construct-java-type-from-str var-type var-value))))

#?(:clj
   (defn get-env-var
     ([var-name var-type default-value]
      (if-let [var-value-str (System/getenv var-name)]
        (construct-from-str var-type var-value-str)
        default-value))

     ([var-name var-type]
      (if-let  [var-value (get-env-var var-name var-type nil)]
        var-value
        (throw-far-error "Environment variable not found"
                         :environment-error :env-var-not-found
                         {:var-name var-name :var-type var-type})))))

;;;;;;;;;;; Time functions

(s/defn get-current-time-ms :- s/Num
  []
  #?(:clj (System/currentTimeMillis)
     :cljs (.getTime (js/Date.))))

(s/defn str->dt :- DateTime
  " Converts an ISO8601-formatted string to a date-time object"
  [s :- s/Str]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/parse formatter s)))

(s/defn dt->str :- s/Str
  "Converts a date-time object to an ISO8601-formatted string"
  [dt :- DateTime]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/unparse formatter dt)))

(s/defn add-days :- s/Str
  "Adds the specifed number of days to an ISO8601-formatted string
   Returns an ISO8601-formatted string"
  [dt-str :- s/Str
   days :- s/Num]
  (-> dt-str
      str->dt
      (t/plus (t/days days))
      dt->str))

(s/defn day-at :- DateTime
  "Takes a date-time, an hour, and a minute
   Returns a new date-time with the same date, but the
   specified hour and minute."
  ([dt :- DateTime] (day-at dt 0 0))
  ([dt :- DateTime h :- s/Num] (day-at dt h 0))
  ([dt :- DateTime h :- s/Num m :- s/Num]
   (t/date-time (t/year dt) (t/month dt) (t/day dt) h m)))

(s/defn get-num-midnights :- s/Num
  "Return the number of midnights (actually 23:59:59s) between two date-times"
  [begin :- DateTime
   end :- DateTime]
  (loop [night (t/date-time (t/year begin) (t/month begin) (t/day begin)
                            23 59 59)
         count 0]
    (if (t/within? begin end night)
      (recur (t/plus night (t/days 1)) (inc count))
      count)))

(s/defn dt+days :- s/Int
  "Takes a date-time, a number of days, and optionally hour and minute.
   Returns a new date-time offset by the number of days, with hour and
   minute set if they were provided."
  ([dt :- DateTime days :- s/Num]
   (dt+days dt days 0 0))
  ([dt :- DateTime days :- s/Num hours :- s/Num]
   (dt+days dt days hours 0))
  ([dt :- DateTime days :- s/Num hours :- s/Num minutes :- s/Num]
   (day-at (t/plus dt (t/days days)) hours minutes)))

(s/defn hrs-diff :- s/Num
  "Return the number of hours between two date-times."
  [dt1 :- DateTime
   dt2 :- DateTime]
  (/ (t/in-seconds (t/interval dt1 dt2)) 3600.0))

(s/defn local->utc :- DateTime
  "Takes a local date-time, hours, and (optionally) minutes offset from UTC.
  Returns the corresponding date-time in UTC."
  ([dt :- DateTime offset-hrs :- s/Num]
   (local->utc dt offset-hrs 0))
  ([dt :- DateTime offset-hrs :- s/Num offset-mins :- s/Num]
   (t/minus dt (t/hours offset-hrs) (t/minutes offset-mins))))

(s/defn same-day? :- s/Bool
  "Returns true if the two given date-times are on the same day,
   returns false otherwise."
  [dt1 :- DateTime
   dt2 :- DateTime]
  (= [(t/year dt1) (t/month dt1) (t/day dt1)]
     [(t/year dt2) (t/month dt2) (t/day dt2)]))

;;;;;;;;;;; Distance
(defn get-distance-mi [{lat1 :lat long1 :long}
                       {lat2 :lat long2 :long}]
  (let [earth-radius-mi 3959
        acos #?(:clj #(Math/acos %) :cljs #(.acos js/Math %))
        cos #?(:clj #(Math/cos %) :cljs #(.cos js/Math %))
        sin #?(:clj #(Math/sin %) :cljs #(.sin js/Math %))
        pi #?(:clj Math/PI :cljs (.-PI js/Math))
        to-radians #(* % (/ pi 180))]
    (* earth-radius-mi
       (acos (+ (* (sin (to-radians lat1))
                   (sin (to-radians lat2)))
                (* (cos (to-radians lat1))
                   (cos (to-radians lat2))
                   (cos (- (to-radians long1)
                           (to-radians long2)))))))))

;;;;;;;;;;;;;;; Randomness

;; From https://github.com/sjl/roul/blob/master/src/roul/random.clj
;; Copyright © 2012 Steve Losh and Contributors
;; MIT/X11 Licensed
(defn weighted-rand
  "Return a random element from the weighted collection.
  A weighted collection can be any seq of [choice, weight] elements.  The
  weights can be arbitrary numbers -- they do not need to add up to anything
  specific.
  Examples:
  (rand-nth-weighted [[:a 0.50] [:b 0.20] [:c 0.30]])
  (rand-nth-weighted {:a 10 :b 200})
  "
  [coll]
  (let [total (reduce + (map second coll))]
    (loop [i (rand total)
           [[choice weight] & remaining] (seq coll)]
      (if (>= weight i)
        choice
        (recur (- i weight) remaining)))))

(defn rand-from-vec [v]
  (v (rand-int (count v))))

(defn rand-from-set [s]
  (rand-nth (seq s)))

;;;;;;;;; Serialization / Deserialization ;;;;;;;;;

(def initial-transit-buffer-size 4096)
(def date-time-formatter (f/formatters :date-hour-minute-second-ms))
(def date-time-transit-tag "dt")

(def date-time-writer
  (transit/write-handler
   (constantly date-time-transit-tag)
   #(f/unparse date-time-formatter %)))

(def date-time-reader
  (transit/read-handler
   #(f/parse date-time-formatter %)))

(defn edn->transit [edn]
  #?(:clj
     (let [out (ByteArrayOutputStream. initial-transit-buffer-size)
           writer (transit/writer
                   out :json
                   {:handlers {DateTime date-time-writer}})]
       (transit/write writer edn)
       (.toString ^ByteArrayOutputStream out "UTF-8"))
     :cljs
     (transit/write (transit/writer
                     :json
                     {:handlers {UtcDateTime date-time-writer}})
                    edn)))

(defn transit->edn [^String transit-str]
  (when transit-str
    #?(:clj
       (let [bytes (.getBytes transit-str "UTF-8")
             in (ByteArrayInputStream. bytes)
             reader (transit/reader
                     in :json
                     {:handlers {date-time-transit-tag date-time-reader}})]
         (transit/read reader))
       :cljs
       (transit/read (transit/reader
                      :json
                      {:handlers {date-time-transit-tag date-time-reader}})
                     transit-str))))

;;;;;;;;;;;;;;;;;;;; Hashing ;;;;;;;;;;;;;;;;;;;;

(s/defn murmur-hash :- s/Int
  [s :- s/Str]
  #?(:clj
     (-> (Hashing/murmur3_32)
         (.hashBytes (.getBytes ^String s) 0 (count s))
         (.asInt))
     :cljs
     (murmur/hashBytes s (count s) 0)))

;;;;;;;;;;;;;;;;;;;; byte-array for cljs ;;;;;;;;;;;;;;;;;;;;

(def byte-array-type
  #?(:clj
     (class (byte-array []))
     :cljs
     js/Int8Array))

#?(:cljs
   (defn byte-array
     ([size-or-seq]
      (if (sequential? size-or-seq)
        (byte-array (count size-or-seq) size-or-seq)
        (byte-array size-or-seq 0)))
     ([size init-val-or-seq]
      (let [ba (byte-array-type. size)]
        (if (sequential? init-val-or-seq)
          (.set ba (clj->js init-val-or-seq))
          (.fill ba init-val-or-seq))
        ba))))

(s/defn byte-array? :- s/Bool
  [x :- s/Any]
  (when-not (nil? x)
    (boolean (= byte-array-type (class x)))))

(defn equivalent-byte-arrays? [a b]
  (let [cmp (fn [acc i]
              (and acc
                   (= (aget #^bytes a i)
                      (aget #^bytes b i))))
        result (and (= (count a) (count b))
                    (reduce cmp true (range (count a))))]
    result))

;; Make cljs byte-arrays countable
#?(:cljs (extend-protocol ICounted
           byte-array-type
           (-count [this] (.-length this))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn inspect-helper [& exprs]
  (doseq [[expr-name expr-val] exprs]
    (println (str expr-name ":"))
    (#?(:clj cprint :cljs pprint) expr-val)))

(defn throws-helper
  [error-type error-subtype body]
  (try
    (body)
    (throw-far-error "Did not throw" :execution-error :did-not-throw {})
    (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
      (let [{:keys [type subtype]} (ex-data e)]
        (is (= error-type type))
        (is (= error-subtype subtype))))))

(defmacro throws
  [error-type error-subtype & body]
  `(throws-helper ~error-type ~error-subtype #(do ~@body)))

(defmacro sym-map
  "Builds a map from local symbols.
  Symbol names are turned into keywords and become the map's keys
  Symbol values become the map's values
  (let [a 1
        b 2]
    (sym-map a b))  =>  {:a 1 :b 2}"
  [& syms]
  (zipmap (map keyword syms) syms))

(defmacro inspect
  "Print some symbols for debugging, using pprint/cprint.
  (inspect foo bar) => foo:
                       {:a 1}
                       bar:
                       [:a :vector]"
  [& syms]
  (let [exprs (map #(vector (name %) %) syms)]
    `(inspect-helper ~@exprs)))

;;;;;;;;;;;;;;;;;;;; Macro-writing utils ;;;;;;;;;;;;;;;;;;;;

;; From: http://blog.nberger.com.ar/blog/2015/09/18/more-portable-complex-macro-musing/
(defn- cljs-env?
  "Take the &env from a macro, and return whether we are expanding into cljs."
  [env]
  (boolean (:ns env)))

(defmacro if-cljs
  "Return then if we are generating cljs code and else for Clojure code.
  https://groups.google.com/d/msg/clojurescript/iBY5HaQda4A/w1lAQi9_AwsJ"
  [then else]
  (if (cljs-env? &env) then else))


;;;;;;;;;;;;;;;;;;;; core.async utils ;;;;;;;;;;;;;;;;;;;;

(defn log-exception [e]
  ;; TODO: Enhance this to intelligently handle throw-far-error/ex-data
  (errorf (str "\nException:\n"
               #?(:clj (.getMessage ^Exception e) :cljs (.-message e))
               "\nStacktrace:\n"
               #?(:clj (join "\n" (map str
                                       (.getStackTrace ^Exception e)))
                  :cljs (.-stack e)))))

(defmacro go-safe [& body]
  `(if-cljs
    (cljs.core.async.macros/go
      (try
        ~@body
        (catch :default e#
          (log-exception e#)
          [:failure :exception-thrown])))
    (clojure.core.async/go
      (try
        ~@body
        (catch Exception e#
          (log-exception e#)
          [:failure :exception-thrown])))))

;;;;;;;;; Async test helper fns ;;;;;;;;;
;; Taken from
;; http://stackoverflow.com/questions/30766215/how-do-i-unit-test-clojure-core-async-go-macros/30781278#30781278

(defn test-async
  "Asynchronous test awaiting ch to produce a value or close."
  [ch]
  #?(:clj
     (<!! ch)
     :cljs
     (async done
            (take! ch (fn [_] (done))))))

(defn test-within-ms
  "Asserts that ch does not close or produce a value within ms. Returns a
  channel from which the value can be taken."
  [ms ch]
  (go (let [t (timeout ms)
            [v ch] (alts! [ch t])]
        (is (not= ch t)
            (str "Test should have finished within " ms "ms."))
        v)))

;;;;;;;;;;;;;;;;;;;; Platform detection ;;;;;;;;;;;;;;

(defn jvm? []
  #?(:clj true
     :cljs false))

(defn browser? []
  #?(:clj false
     :cljs
     (exists? js/navigator)))

(defn node? []
  #?(:clj false
     :cljs (boolean (= "nodejs" cljs.core/*target*))))

(defn jsc-ios? []
  #?(:clj false
     :cljs
     (try
       (boolean (= "jsc-ios" js/JSEnv))
       (catch :default e
         false))))

(defn get-platform-kw []
  (cond
    (jvm?) :jvm
    (node?) :node
    (jsc-ios?) :jsc-ios
    (browser?) :browser
    :else :unknown))

;;;;;;;;;;;;;;;;;;;; UUIDs ;;;;;;;;;;;;;;;;;;;;

(def UUIDLongMap
  {:high-long Long
   :low-long Long})

(defprotocol IUUID
  (uuid->hex-str [this] "Return the UUID as a 36-character canonical
                         hex string.")
  (uuid->longs [this]   "Returns the UUID as two 64-bit longs.
                         High bits followed by low bits.")
  (uuid->long-map [this] "Returns the UUID as a map with two keys, :high-long
                          and :low-long, whose values are 64-bit longs."))

(extend-protocol IUUID
  #?(:clj java.util.UUID :cljs cljs.core/UUID)
  (uuid->hex-str [this]
    (.toString this))

  (uuid->longs [this]
    #?(:clj [(.getMostSignificantBits this) (.getLeastSignificantBits this)]
       :cljs (let [hex-str (-> (.toString this)
                               (string/replace #"-" ""))
                   high-str (subs hex-str 0 16)
                   low-str (subs hex-str 16 32)]
               [(hex-str->long high-str) (hex-str->long low-str)])))

  (uuid->long-map [this]
    (let [[high-long low-long] (uuid->longs this)]
      (sym-map high-long low-long))))

(defn canonicalize-hex-str [hex-str]
  (string/join "-" [(subs hex-str 0 8)
                    (subs hex-str 8 12)
                    (subs hex-str 12 16)
                    (subs hex-str 16 20)
                    (subs hex-str 20 32)]))

(s/defn longs->uuid :- UUID
  [high-long :- Long
   low-long :- Long]
  #?(:clj
     (UUID. high-long low-long)
     :cljs
     (let [high-str (long->hex-str high-long)
           low-str (long->hex-str low-long)
           hex-str (str high-str low-str)]
       (uuid (canonicalize-hex-str hex-str)))))

(s/defn long-map->uuid :- UUID
  [long-map :- UUIDLongMap]
  (longs->uuid (:high-long long-map) (:low-long long-map)))

#?(:cljs
   (defn set-multicast-bit [node-id-bytes]
     (let [length (.-length node-id-bytes)
           _ (when-not (= 6 (.-length node-id-bytes))
               (throw-far-error "Node id is not 6 bytes long (48 bits)"
                                :illegal-argument :bad-node-id-length
                                (sym-map length node-id-bytes)))]
       (aset node-id-bytes 0 (bit-or 0x01 (aget node-id-bytes 0)))
       node-id-bytes)))

#?(:cljs
   (defn hex-str->byte-array [hex-str]
     (byte-array (map #(js/parseInt (join %) 16)
                      (partition 2 hex-str)))))

#?(:cljs
   (defn get-node-id-bytes-node []
     (let [os (js/require "os")
           nis (js->clj (.networkInterfaces os))
           macs (for [k (keys nis)
                      addr (nis k)
                      :let [mac (addr "mac")]
                      :when (not= "00:00:00:00:00:00" mac)]
                  mac)
           mac-str (string/replace (first macs) ":" "")]
       (hex-str->byte-array mac-str))))

#?(:cljs
   (defn int->byte-array [i]
     (let [a32 (js/Uint32Array. (clj->js [i]))
           dv (js/DataView. (.-buffer a32))
           output (byte-array 4)]
       (dotimes [i 4]
         (aset output i (.getInt8 dv i) true))
       output)))

#?(:cljs
   (defn get-node-id-bytes-ios []
     (let [id-str (-> (js/identifierForVendor) ;; Returns a UUID-style string
                      (string/replace "-" ""))
           id-arr (hex-str->byte-array id-str)
           hash-arr (int->byte-array (murmur-hash id-str))
           node-id (byte-array 6)]
       ;; We need 6 bytes 16, so we concatenate the first 2
       ;; bytes of the id array with the 4 bytes from the hash of the id
       (.set node-id (.slice id-arr 0 2))
       (.set node-id hash-arr 2)
       node-id)))

#?(:cljs
   (defn get-node-id-bytes-browser []
     ;; TODO: Implement this using a hash of user agent string,
     ;; supported mimeTypes, and count of vars in global scope.
     ;; See the api.fingerprint method here:
     ;; https://github.com/ericelliott/cuid/blob/v1.3.8/dist/browser-cuid.js
     ;; Remember to set the multicast bit, since this is not a real
     ;; MAC address (call set-multicast-bit fn).
     (throw-far-error "get-node-id-bytes-browser is not implemented yet"
                      :execution-error :not-implmented {})))

#?(:cljs
   (defn get-node-id-bytes*
     "Return the 48-bit node id as a 12-char hex string."
     []
     (case (get-platform-kw)
       :node (get-node-id-bytes-node)
       :jsc-ios (get-node-id-bytes-ios)
       :browser (get-node-id-bytes-browser))))

#?(:cljs
   (def get-node-id-bytes (memoize get-node-id-bytes*)))

#?(:cljs
   (def uuid-gen-state
     (atom {:clock-seq (rand-int (bit-shift-left 1 14))
            :last-msecs 0
            :last-nsecs 0})))

#?(:cljs
   (defn make-uuid-str [time-low time-mid-high clock-seq node-id-bytes]
     (let [arr (js/Uint8Array. 16)]
       ;; time-low
       (aset arr 0 (bit-and (unsigned-bit-shift-right time-low 24) 0xff))
       (aset arr 1 (bit-and (unsigned-bit-shift-right time-low 16) 0xff))
       (aset arr 2 (bit-and (unsigned-bit-shift-right time-low 8) 0xff))
       (aset arr 3 (bit-and time-low 0xff))
       ;; time-mid
       (aset arr 4 (bit-and (unsigned-bit-shift-right time-mid-high 8) 0xff))
       (aset arr 5 (bit-and time-mid-high 0xff))
       ;; time-high-and-version
       (aset arr 6 (bit-or 0x10  ;; version
                           (bit-and (unsigned-bit-shift-right time-mid-high 24)
                                    0xf)))
       (aset arr 7 (bit-and (unsigned-bit-shift-right time-mid-high 16) 0xff))
       ;; clock-seq-hi-and-reserved (RFC 4122 4.2.2)
       (aset arr 8 (bit-or 0x80  ;; variant mask
                           (unsigned-bit-shift-right clock-seq 8)))
       (aset arr 9 (bit-and clock-seq 0xff))
       ;; node-id
       (.set arr node-id-bytes 10)
       (let [hex-str (.reduce arr (fn [pv cv i arr]
                                    (str pv (.toString cv 16))) "")]
         (canonicalize-hex-str hex-str)))))

;; The  cljs UUID v1 implementation is based heavily on this js
;; version: https://github.com/broofa/node-uuid/blob/master/uuid.js

;; UUID timestamps are 100 nano-second units since the Gregorian epoch,
;; (1582-10-15 00:00).  JSNumbers aren't precise enough for this, so
;; time is handled internally as 'msecs' (integer milliseconds) and 'nsecs'
;; (100-nanoseconds offset from msecs) since unix epoch, 1970-01-01 00:00.
#?(:cljs
   (defn make-v1-uuid-cljs []
     (let [{:keys [clock-seq last-msecs last-nsecs]} @uuid-gen-state
           msecs (get-current-time-ms)
           ;; Per 4.2.1.2, use count of uuid's generated during the current
           ;; clock cycle to simulate higher resolution clock
           nsecs (inc last-nsecs)
           ;; Time since last uuid creation (in msecs)
           dt (+ (- msecs last-msecs)
                 (/ (- nsecs last-nsecs) 10000))
           ;; Per 4.2.1.2, Bump clock-seq on clock regression
           clock-seq (if (neg? dt)
                       (inc clock-seq)
                       clock-seq)
           ;; Reset nsecs if clock regresses or we've moved onto a new
           ;; time interval
           nsecs (if (or (neg? dt)
                         (> msecs last-msecs))
                   0
                   nsecs)
           ;; Per 4.2.1.2 Throw error if too many uuids are requested
           _ (when (>= nsecs 10000)
               (throw-far-error "Can't create more than 10M uuids/sec"
                                :execution-error :uuid-creation-rate-exceeded
                                (sym-map nsecs)))
           gregorian-msecs (+ msecs 12219292800000)
           time-low (rem (+ nsecs
                            (* 10000 (bit-and gregorian-msecs 0xfffffff)))
                         0x100000000)
           time-mid-high (bit-and 0xfffffff
                                  (* 10000 (/ msecs 0x100000000)))
           node-id-bytes (get-node-id-bytes)
           new-state {:clock-seq clock-seq
                      :last-msecs msecs
                      :last-nsecs nsecs}]
       (reset! uuid-gen-state new-state)
       (uuid (make-uuid-str time-low time-mid-high clock-seq node-id-bytes)))))

(s/defn make-v1-uuid :- UUID
  []
  #?(:clj (clj-uuid/v1)
     :cljs (make-v1-uuid-cljs)))

(s/defn make-v4-uuid :- UUID
  []
  #?(:cljs (random-uuid)
     :clj (clj-uuid/v4)))

;;;;;;;;;;;;;;;;;;;; Logging helpers ;;;;;;;;;;;;;;;;;;;;

(defn short-log-output-fn [data]
  (let [{:keys [level msg_]} data
        formatter (f/formatters  :hour-minute-second-ms)
        timestamp (f/unparse formatter (t/now))]
    (str
     timestamp " "
     (clojure.string/upper-case (name level))  " "
     @msg_)))
