(ns farbetter.utils
  (:refer-clojure :exclude [byte-array long])
  (:require
   #?(:cljs cljsjs.long)
   #?(:cljs [cljs.pprint :refer [pprint]])
   #?(:cljs [cljs-http.client :as http])
   #?(:cljs [cljs-time.extend]) ;; to make = work as expected
   [#?(:clj clj-time.core :cljs cljs-time.core) :as t]
   [#?(:clj clj-time.format :cljs cljs-time.format) :as f]
   #?(:clj [clj-uuid :as clj-uuid])
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca
    :refer [alts! take! timeout #?@(:clj [go <!!])]]
   [#?(:clj clojure.core.async.impl.protocols
       :cljs cljs.core.async.impl.protocols) :as cap]
   [clojure.string :as string :refer [join]]
   #?(:clj [clojure.test :refer [is]] :cljs [cljs.test :as test])
   [cognitect.transit :as transit]
   #?(:cljs [murmur])
   #?(:clj [org.httpkit.client :as http])
   #?(:clj [puget.printer :refer [cprint]])
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [cljs.core.async.macros :refer [go]]
      [cljs.test :refer [async is]]
      [farbetter.utils :refer [go-safe inspect sym-map throws]]))
  #?(:clj
     (:import [com.google.common.hash Hashing]
              [com.google.common.primitives Bytes]
              [com.google.common.primitives UnsignedLong]
              [java.io ByteArrayInputStream ByteArrayOutputStream]
              [java.nio.charset StandardCharsets]
              [java.util Arrays UUID]
              [org.joda.time DateTime])
     :cljs
     (:import [goog.date Date DateTime UtcDateTime])))

(declare http-get-node)

#?(:cljs (def class type))

(s/defn throw-far-error :- nil
  "Throws an ex-info error with the specified description, type, subtype,
  and error map. Includes the error map in the description thrown."
  [desc :- s/Str
   error-type :- s/Keyword
   error-subtype :- s/Keyword
   error-map :- {s/Any s/Any}]
  (let [error-map (merge error-map
                         {:type error-type :subtype error-subtype})
        desc (cond-> desc
               (not= "." (str (last desc))) (str "."))]
    (throw (ex-info (str desc " error-map: " error-map)
                    error-map))))

(defmacro sym-map
  "Builds a map from local symbols.
  Symbol names are turned into keywords and become the map's keys
  Symbol values become the map's values
  (let [a 1
        b 2]
    (sym-map a b))  =>  {:a 1 :b 2}"
  [& syms]
  (zipmap (map keyword syms) syms))


;;;;;;;;;;;;;;;;;;;; Longs ;;;;;;;;;;;;;;;;;;;;

#?(:cljs (def Long js/Long))

(def LongIntMap
  {(s/required-key :high) s/Int
   (s/required-key :low) s/Int})

(s/defn long? :- s/Bool
  [x :- s/Any]
  (if x
    (boolean (= Long (class x)))
    false))

(s/defn long :- Long
  [x :- s/Any]
  (when-not (nil? x)
    #?(:clj (clojure.core/long x)
       :cljs (Long.fromValue x))))

(s/defn long= :- s/Bool
  [a :- s/Any
   b :- s/Any]
  #?(:clj (= a b)
     :cljs (.equals a b)))

#?(:cljs (extend-type Long
           IEquiv
           (-equiv [l other]
             (long= l other))))

#?(:cljs (extend-type js/Long
           IHash
           (-hash [l]
             (bit-xor (.getLowBits l) (.getHighBits l)))))

#?(:cljs (extend-type Long
           IComparable
           (-compare [l other]
             (.compare l other))))

(defn- throw-long->int-err [l]
  (throw-far-error (str "Cannot convert long `" l "` to int.")
                   :illegal-argument :not-an-integer
                   {:input l
                    :class-of-input (#?(:clj class :cljs type) l)}))

(s/defn long->int :- s/Int
  [l :- Long]
  #?(:clj (if (and (<= l 2147483647) (>= l -2147483648))
            (.intValue l)
            (throw-long->int-err l))
     :cljs (if (and (.lte l 2147483647) (.gte l -2147483648))
             (.toInt l)
             (throw-long->int-err l))))

(s/defn long->str :- s/Str
  [l :- Long]
  #?(:clj
     (Long/toString l)
     :cljs
     (.toString l)))

(s/defn long->float :- s/Num
  [l :- Long]
  #?(:clj
     (float l)
     :cljs
     (.toNumber l)))

(s/defn long->double :- s/Num
  [l :- Long]
  #?(:clj
     (double l)
     :cljs
     (.toNumber l)))

(s/defn long->hex-str :- s/Str
  [l :- Long]
  (let [pad (fn [s len]
              (loop [out s]
                (if (= len (count out))
                  out
                  (recur (str "0" out)))))
        s #?(:clj
             (Long/toHexString l)
             :cljs
             (-> (.toUnsigned l)
                 (.toString 16)))]
    (pad s 16)))

(s/defn long->ints :- (s/pair s/Int :high-int
                              s/Int :low-int)
  [l :- Long]
  (let [high (int #?(:clj (bit-shift-right l 32)
                     :cljs (.getHighBits l)))
        low (int #?(:clj (.intValue l)
                    :cljs (.getLowBits l)))]
    [high low]))

(s/defn long->int-map :- LongIntMap
  [l :- Long]
  (let [[high low] (long->ints l)]
    (sym-map high low)))

(s/defn str->long :- Long
  [s :- s/Str]
  #?(:clj
     (Long/decode s)
     :cljs
     (.fromString Long s)))

(s/defn hex-str->long :- Long
  [hex-str :- s/Str]
  #?(:clj
     (.longValue (UnsignedLong/valueOf hex-str 16))
     :cljs
     (.fromString Long hex-str 16)))

(s/defn ints->long :- Long
  [high :- s/Int
   low :- s/Int]
  #?(:clj (bit-or (bit-shift-left (long high) 32)
                  (bit-and low 0xFFFFFFFF))
     :cljs (.fromBits Long low high)))

(s/defn int-map->long :- Long
  [int-map :- LongIntMap]
  (let [{:keys [high low]} int-map]
    (ints->long high low)))

(def FarNum (s/if long?
              Long
              s/Num))

;;;;;;;;;;;;;;;;;;;; Core.async helpers ;;;;;;;;;;;;;;;;;;;;

(def Channel (s/protocol cap/Channel))

(s/defn channel? :- s/Bool
  [x :- s/Any]
  (satisfies? cap/Channel x))

;;;;;;;;;;;;;;;; Handy utilities

;; dissoc-in function was stolen from:
;; https://github.com/clojure/core.incubator/blob/master/src
;; /main/clojure/clojure/core/incubator.clj
(s/defn dissoc-in :- {s/Any s/Any}
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m :- {s/Any s/Any}
   [k & ks :as keys] :- [s/Any]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))

;;;;;;;;;;; Exception handling and testing

(s/defn negative? :- s/Bool
  [n :- FarNum]
  #?(:clj (neg? n)
     :cljs (if (long? n)
             (.isNegative n)
             (neg? n))))

(s/defn invert
  [n :- FarNum]
  #?(:clj (- n)
     :cljs (if (long? n)
             (.subtract (.-ZERO js/Long) n)
             (- n))))

(s/defn abs :- FarNum
  "Returns the absolute value of n"
  [n :- FarNum]
  (if (negative? n)
    (invert n)
    n))

(s/defn within?
  "Tests if the actual value is within error-margin of the
  expected value. Actual and expected parameters must be numeric."
  [error-margin :- FarNum
   expected :- FarNum
   actual :- FarNum]
  (let [neg-inf #?(:clj
                   (Double/NEGATIVE_INFINITY)
                   :cljs
                   (.-NEGATIVE_INFINITY js/Number))
        pos-inf #?(:clj
                   (Double/POSITIVE_INFINITY)
                   :cljs
                   (.-POSITIVE_INFINITY js/Number))
        eq? (fn [a b]
              #?(:clj (= a b)
                 :cljs (if (long? a)
                         (.eq a b)
                         (if (long? b)
                           (.eq b a)
                           (= a b)))))
        lte? (fn [a b]
               #?(:clj (<= a b)
                  :cljs (if (long? a)
                          (.lte a b)
                          (if (long? b)
                            (.lte b a)
                            (<= a b)))))
        sub (fn [a b]
              #?(:clj (- a b)
                 :cljs (if (long? a)
                         (.subtract a b)
                         (if (long? b)
                           (.subtract b a)
                           (- a b)))))]
    (cond
      (eq? expected neg-inf) (= actual neg-inf)
      (eq? expected pos-inf) (= actual pos-inf)
      :else (let [diff (sub actual expected)
                  abs-diff (abs diff)]
              (lte? abs-diff error-margin)))))

;;;;;;;;;;; Environment variables (Clojure only, no CLJS)

#?(:clj
   (s/defn construct-java-type-from-str :- s/Any
     [class :- java.lang.Class
      s :- s/Str]
     (.newInstance
      (.getConstructor class (into-array java.lang.Class [java.lang.String]))
      (object-array [s]))))

#?(:clj
   (s/defn construct-from-str :- s/Any
     [var-type :- java.lang.Class
      var-value :- s/Str]
     (if (= clojure.lang.Keyword var-type)
       (keyword var-value)
       (construct-java-type-from-str var-type var-value))))

#?(:clj
   (s/defn get-env-var :- s/Any
     ([var-name :- s/Str
       var-type :- java.lang.Class]
      (if-let  [var-value (get-env-var var-name var-type nil)]
        var-value
        (throw-far-error "Environment variable not found"
                         :environment-error :env-var-not-found
                         {:var-name var-name :var-type var-type})))
     ([var-name :- s/Str
       var-type :- java.lang.Class
       default-value :- s/Any]
      (if-let [var-value-str (System/getenv var-name)]
        (construct-from-str var-type var-value-str)
        default-value))))

;;;;;;;;;;; Time functions

(def DateMap {(s/required-key :year) s/Num
              (s/required-key :month) s/Num
              (s/required-key :day) s/Num})

(def TimeMap {(s/required-key :hour) s/Num
              (s/required-key :minute) s/Num
              (s/optional-key :second) s/Num
              (s/optional-key :offset-hours) s/Num
              (s/optional-key :offset-minutes) s/Num})

(def DateTimeMap {(s/required-key :date) DateMap
                  (s/required-key :time) TimeMap})

(def date-schema
  {:name :date
   :doc "A date represented as a record"
   :type :record
   :fields [{:name :year
             :type :int
             :doc "Four-digit year"
             :default 0}
            {:name :month
             :type :int
             :doc "Month as a number. 1=January, 12=December"
             :default 0}
            {:name :day
             :type :int
             :doc "Day of the month"
             :default 0}]})

(def date-default {:year 0 :month 1 :day 1})

(def time-schema
  {:name :time
   :doc "A time represented as a record"
   :type :record
   :fields [{:name :hour
             :type :int
             :doc "Hour in 24-hour format. 1=1 AM, 13=1 PM"
             :default 0}
            {:name :minute
             :type :int
             :doc "Minute"
             :default 0}
            {:name :second
             :type [:null :int]
             :doc "Second"
             :default nil}
            {:name :offset-hours
             :type [:null :int]
             :doc "Hours offset from UTC"
             :default nil}
            {:name :offset-minutes
             :type [:null :int]
             :doc "Minutes offset from UTC"
             :default nil}]})

(def time-default {:hour 0 :minute 0 :second 0})

(def date-time-schema
  {:name :date-time
   :doc "A date-time represented as a record"
   :type :record
   :fields [{:name :date
             :type date-schema
             :default date-default}
            {:name :time
             :type time-schema
             :default time-default}]})

(def date-time-default {:date date-default :time time-default})

(s/defn get-current-time-ms :- s/Num
  []
  #?(:clj (System/currentTimeMillis)
     :cljs (.getTime (js/Date.))))

(s/defn str->dt :- DateTime
  " Converts an ISO8601-formatted string to a date-time object"
  [s :- s/Str]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/parse formatter s)))

(s/defn dt->str :- s/Str
  "Converts a date-time object to an ISO8601-formatted string"
  [dt :- DateTime]
  (let [formatter (f/formatters :date-hour-minute)]
    (f/unparse formatter dt)))

(s/defn date-map->dt :- DateTime
  "Converts a DateMap to a date-time object."
  [m :- DateMap]
  (let [{:keys [year month day]} m]
    (t/date-time year month day)))

(s/defn dt->date-map :- DateMap
  "Converts a date-time object to a DateMap"
  [dt :- DateTime]
  {:year (t/year dt)
   :month (t/month dt)
   :day (t/day dt)})

;; TODO: Find a way to support arbitrary timezone conversions in cljs.
;; cljs-time only converts to and from utc and default timezone.
#?(:clj
   (s/defn date-time-map->dt :- DateTime
     "Converts a DateTimeMap to a date-time-object"
     [m :- DateTimeMap]
     (let [{:keys [year month day]} (:date m)
           {:keys [hour minute second offset-hours offset-minutes]
            :or {second 0 offset-hours 0 offset-minutes 0}} (:time m)]
       (let [base-dt (t/date-time year month day hour minute second)
             tz (t/time-zone-for-offset offset-hours offset-minutes)]
         (t/from-time-zone base-dt tz)))))

;; TODO: Find a way to support arbitrary timezone conversions in cljs.
;; cljs-time only converts to and from utc and default timezone.
#?(:clj
   (s/defn dt->date-time-map :- DateTimeMap
     "Converts a date-time object to a DateMap"
     [dt :- DateTime]
     (let [utc-dt (t/to-time-zone dt (t/time-zone-for-offset 0 0))
           date {:year (t/year utc-dt)
                 :month (t/month utc-dt)
                 :day (t/day utc-dt)}
           time {:hour (t/hour utc-dt)
                 :minute (t/minute utc-dt)
                 :second (t/second utc-dt)
                 :offset-hours 0
                 :offset-minutes 0}]
       (sym-map date time))))

(s/defn add-days :- s/Str
  "Adds the specifed number of days to an ISO8601-formatted string
   Returns an ISO8601-formatted string"
  [dt-str :- s/Str
   days :- s/Num]
  (-> dt-str
      str->dt
      (t/plus (t/days days))
      dt->str))

(s/defn day-at :- DateTime
  "Takes a date-time, an hour, and a minute
   Returns a new date-time with the same date, but the
   specified hour and minute."
  ([dt :- DateTime] (day-at dt 0 0))
  ([dt :- DateTime h :- s/Num] (day-at dt h 0))
  ([dt :- DateTime h :- s/Num m :- s/Num]
   (t/date-time (t/year dt) (t/month dt) (t/day dt) h m)))

(s/defn get-num-midnights :- s/Num
  "Return the number of midnights (actually 23:59:59s) between two date-times"
  [begin :- DateTime
   end :- DateTime]
  (loop [night (t/date-time (t/year begin) (t/month begin) (t/day begin)
                            23 59 59)
         count 0]
    (if (t/within? begin end night)
      (recur (t/plus night (t/days 1)) (inc count))
      count)))

(s/defn dt+days :- s/Int
  "Takes a date-time, a number of days, and optionally hour and minute.
   Returns a new date-time offset by the number of days, with hour and
   minute set if they were provided."
  ([dt :- DateTime days :- s/Num]
   (dt+days dt days 0 0))
  ([dt :- DateTime days :- s/Num hours :- s/Num]
   (dt+days dt days hours 0))
  ([dt :- DateTime days :- s/Num hours :- s/Num minutes :- s/Num]
   (day-at (t/plus dt (t/days days)) hours minutes)))

(s/defn hrs-diff :- s/Num
  "Return the number of hours between two date-times."
  [dt1 :- DateTime
   dt2 :- DateTime]
  (/ (t/in-seconds (t/interval dt1 dt2)) 3600.0))

(s/defn local->utc :- DateTime
  "Takes a local date-time, hours, and (optionally) minutes offset from UTC.
  Returns the corresponding date-time in UTC."
  ([dt :- DateTime offset-hrs :- s/Num]
   (local->utc dt offset-hrs 0))
  ([dt :- DateTime offset-hrs :- s/Num offset-mins :- s/Num]
   (t/minus dt (t/hours offset-hrs) (t/minutes offset-mins))))

(s/defn same-day? :- s/Bool
  "Returns true if the two given date-times are on the same day,
   returns false otherwise."
  [dt1 :- DateTime
   dt2 :- DateTime]
  (= [(t/year dt1) (t/month dt1) (t/day dt1)]
     [(t/year dt2) (t/month dt2) (t/day dt2)]))

;;;;;;;;;;; Distance

(def LatLongMap {(s/required-key :lat) s/Num
                 (s/required-key :long) s/Num})

(s/defn get-distance-mi :- s/Num
  [point1 :- LatLongMap
   point2 :- LatLongMap]
  (let [{lat1 :lat long1 :long} point1
        {lat2 :lat long2 :long} point2
        earth-radius-mi 3959
        acos #?(:clj #(Math/acos %) :cljs #(.acos js/Math %))
        cos #?(:clj #(Math/cos %) :cljs #(.cos js/Math %))
        sin #?(:clj #(Math/sin %) :cljs #(.sin js/Math %))
        pi #?(:clj Math/PI :cljs (.-PI js/Math))
        to-radians #(* % (/ pi 180))]
    (* earth-radius-mi
       (acos (+ (* (sin (to-radians lat1))
                   (sin (to-radians lat2)))
                (* (cos (to-radians lat1))
                   (cos (to-radians lat2))
                   (cos (- (to-radians long1)
                           (to-radians long2)))))))))

;;;;;;;;;;;;;;; Randomness

(def WeightedSeq (s/if map?
                   {s/Any s/Any}
                   [[(s/one s/Any "choice")
                     (s/one s/Num "weight")]]))

;; From https://github.com/sjl/roul/blob/master/src/roul/random.clj
;; Copyright © 2012 Steve Losh and Contributors
;; MIT/X11 Licensed
(s/defn weighted-rand :- s/Any
  "Return a random element from the weighted collection.
  A weighted collection can be any seq of [choice, weight] elements.  The
  weights can be arbitrary numbers -- they do not need to add up to anything
  specific.
  Examples:
  (weighted-rand [[:a 0.50] [:b 0.20] [:c 0.30]])
  (weighted-rand {:a 10 :b 200})
  "
  [coll :- WeightedSeq]
  (let [total (reduce + (map second coll))]
    (loop [i (rand total)
           [[choice weight] & remaining] (seq coll)]
      (if (>= weight i)
        choice
        (recur (- i weight) remaining)))))

(s/defn rand-from-vec :- s/Any
  [v :- [s/Any]]
  (v (rand-int (count v))))

(s/defn rand-from-set :- s/Any
  [s :- #{s/Any}]
  (rand-nth (seq s)))

(s/defn rand-range :- s/Num
  [lower :- s/Num
   upper :- s/Num]
  (let [range (- upper lower)
        n (rand)]
    (int (+ lower (* range n)))))

;;;;;;;;; Serialization / Deserialization ;;;;;;;;;

(def initial-transit-buffer-size 4096)
(def date-time-formatter (f/formatters :date-hour-minute-second-ms))
(def date-time-transit-tag "dt")

(def date-time-writer
  (transit/write-handler
   (constantly date-time-transit-tag)
   #(f/unparse date-time-formatter %)))

(def date-time-reader
  (transit/read-handler
   #(f/parse date-time-formatter %)))

(s/defn edn->transit :- (s/maybe s/Str)
  [edn :- s/Any]
  #?(:clj
     (let [out (ByteArrayOutputStream. initial-transit-buffer-size)
           writer (transit/writer
                   out :json
                   {:handlers {DateTime date-time-writer}})]
       (transit/write writer edn)
       (.toString ^ByteArrayOutputStream out "UTF-8"))
     :cljs
     (transit/write (transit/writer
                     :json
                     {:handlers {UtcDateTime date-time-writer}})
                    edn)))

(s/defn transit->edn :- s/Any
  [transit-str :- (s/maybe s/Str)]
  (when transit-str
    #?(:clj
       (let [bytes (.getBytes transit-str "UTF-8")
             in (ByteArrayInputStream. bytes)
             reader (transit/reader
                     in :json
                     {:handlers {date-time-transit-tag date-time-reader}})]
         (transit/read reader))
       :cljs
       (transit/read (transit/reader
                      :json
                      {:handlers {date-time-transit-tag date-time-reader}})
                     transit-str))))

;;;;;;;;;;;;;;;;;;;; Hashing ;;;;;;;;;;;;;;;;;;;;

(s/defn murmur-hash :- s/Int
  [s :- s/Str]
  #?(:clj
     (-> (Hashing/murmur3_32)
         (.hashBytes (.getBytes ^String s) 0 (count s))
         (.asInt))
     :cljs
     (murmur/hashBytes s (count s) 0)))

;;;;;;;;;;;;;;;;;;;; byte-arrays ;;;;;;;;;;;;;;;;;;;;

;; Schema
(def ByteArray
  #?(:clj
     (class (clojure.core/byte-array []))
     :cljs
     js/Int8Array))

(s/defn byte-array? :- s/Bool
  [x :- s/Any]
  (when-not (nil? x)
    (boolean (= ByteArray (class x)))))

(s/defn concat-byte-arrays :- ByteArray
  [arrays :- [ByteArray]]
  #?(:clj (Bytes/concat (into-array arrays))
     :cljs
     (let [lengths (map count arrays)
           new-array (js/Int8Array. (apply + lengths))
           offsets (loop [lens lengths
                          pos 0
                          positions [0]]
                     (if (= 1 (count lens))
                       positions
                       (let [[len & rest] lens
                             new-pos (+ pos len)]
                         (recur rest
                                new-pos
                                (conj positions new-pos)))))]
       (dotimes [i (count arrays)]
         (let [v (arrays i)
               offset (offsets i)]
           (.set new-array v offset)))
       new-array)))

(def SizeOrSeq
  (s/if integer?
    s/Num
    [s/Any]))

#?(:cljs
   (s/defn byte-array-cljs :- ByteArray
     ([size-or-seq :- SizeOrSeq]
      (if (sequential? size-or-seq)
        (byte-array-cljs (count size-or-seq) size-or-seq)
        (byte-array-cljs size-or-seq 0)))
     ([size init-val-or-seq]
      (let [ba (js/Int8Array. size)]
        (if (sequential? init-val-or-seq)
          (.set ba (clj->js init-val-or-seq))
          (.fill ba init-val-or-seq))
        ba))))

(s/defn byte-array :- ByteArray
  ([size-or-seq :- SizeOrSeq]
   (#?(:clj clojure.core/byte-array
       :cljs byte-array-cljs) size-or-seq))
  ([size :- SizeOrSeq
    init-val-or-seq :- (s/if sequential?
                         [s/Any]
                         s/Any)]
   (#?(:clj clojure.core/byte-array
       :cljs byte-array-cljs) size init-val-or-seq)))

(s/defn equivalent-byte-arrays? :- s/Bool
  [a :- ByteArray
   b :- ByteArray]
  (let [cmp (fn [acc i]
              (and acc
                   (= (aget #^bytes a i)
                      (aget #^bytes b i))))
        result (and (= (count a) (count b))
                    (reduce cmp true (range (count a))))]
    result))

;; Make cljs byte-arrays countable
#?(:cljs (extend-protocol ICounted
           js/Int8Array
           (-count [this] (.-length this))))

(s/defn slice :- ByteArray
  "Return a slice of the given byte array.
   Args:
        - array - Array to be sliced. Required.
        - start - Start index. Required.
        - end - End index. Optional. If not provided, the slice will extend
             to the end of the array. The returned slice will not contain
             the byte at the end index position, i.e.: the slice fn uses
             a half-open interval."
  ([array :- ByteArray
    start :- s/Num]
   (slice array start (count array)))
  ([array :- ByteArray
    start :- s/Num
    end :- s/Num]
   (when (> start end)
     (throw-far-error "Slice start is greater than end."
                      :illegal-argument :slice-start-is-greater-than-end
                      (sym-map start end)))
   (#?(:clj Arrays/copyOfRange
       :cljs .slice)
    array start end)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn inspect-helper ;; Must be public to work w/ macros
  [& exprs]
  (doseq [[expr-name expr-val] exprs]
    (println (str expr-name ":"))
    (#?(:clj cprint :cljs pprint) expr-val)))

(defn throws-helper ;; Must be public to work w/ macros
  [error-type error-subtype body]
  (try
    (body)
    (throw-far-error "Did not throw" :execution-error :did-not-throw {})
    (catch #?(:clj clojure.lang.ExceptionInfo :cljs cljs.core.ExceptionInfo) e
      (let [{:keys [type subtype]} (ex-data e)]
        (is (= error-type type))
        (is (= error-subtype subtype))))))

(defmacro throws
  [error-type error-subtype & body]
  `(throws-helper ~error-type ~error-subtype #(do ~@body)))

(defmacro inspect
  "Print some symbols for debugging, using pprint/cprint.
  (inspect foo bar) => foo:
                       {:a 1}
                       bar:
                       [:a :vector]"
  [& syms]
  (let [exprs (map #(vector (name %) %) syms)]
    `(inspect-helper ~@exprs)))

;;;;;;;;;;;;;;;;;;;; Macro-writing utils ;;;;;;;;;;;;;;;;;;;;

;; From: http://blog.nberger.com.ar/blog/2015/09/18/more-portable-complex-macro-musing/
(defn- cljs-env?
  "Take the &env from a macro, and return whether we are expanding into cljs."
  [env]
  (boolean (:ns env)))

(defmacro if-cljs
  "Return then if we are generating cljs code and else for Clojure code.
  https://groups.google.com/d/msg/clojurescript/iBY5HaQda4A/w1lAQi9_AwsJ"
  [then else]
  (if (cljs-env? &env) then else))


;;;;;;;;;;;;;;;;;;;; core.async utils ;;;;;;;;;;;;;;;;;;;;

(s/defn get-exception-msg-and-stacktrace :- s/Str
  [e]
  ;; TODO: Enhance this to intelligently handle
  ;; throw-far-error/ex-data and CLJS source maps for stack trace
  (str "\nException:\n"
       #?(:clj (.getMessage ^Exception e) :cljs (.-message e))
       "\nStacktrace:\n"
       #?(:clj (join "\n" (map str
                               (.getStackTrace ^Exception e)))
          :cljs (.-stack e))))

(defn log-exception [e]
  (errorf (get-exception-msg-and-stacktrace e)))

(defmacro go-safe [& body]
  `(if-cljs
    (cljs.core.async.macros/go
      (try
        ~@body
        (catch :default e#
          (log-exception e#)
          [:failure :exception-thrown])))
    (clojure.core.async/go
      (try
        ~@body
        (catch Exception e#
          (log-exception e#)
          [:failure :exception-thrown])))))

;;;;;;;;; Async test helper fns ;;;;;;;;;
;; Taken from
;; http://stackoverflow.com/questions/30766215/how-do-i-unit-test-clojure-core-async-go-macros/30781278#30781278

(s/defn test-async :- s/Any
  "Asynchronous test awaiting ch to produce a value or close."
  [ch :- Channel]
  #?(:clj
     (<!! ch)
     :cljs
     (async done
            (take! ch (fn [_] (done))))))

(s/defn test-within-ms :- Channel
  "Asserts that ch does not close or produce a value within ms. Returns a
  channel from which the value can be taken."
  [ms :- s/Num
   ch :- Channel]
  (go (let [t (timeout ms)
            [v ch] (alts! [ch t])]
        (is (not= ch t)
            (str "Test should have finished within " ms "ms."))
        v)))

;;;;;;;;;;;;;;;;;;;; Platform detection ;;;;;;;;;;;;;;

(s/defn jvm? :- s/Bool
  []
  #?(:clj true
     :cljs false))

(s/defn browser? :- s/Bool
  []
  #?(:clj false
     :cljs
     (exists? js/navigator)))

(s/defn node? :- s/Bool
  []
  #?(:clj false
     :cljs (boolean (= "nodejs" cljs.core/*target*))))

(s/defn jsc-ios? :- s/Bool
  []
  #?(:clj false
     :cljs
     (try
       (boolean (= "jsc-ios" js/JSEnv))
       (catch :default e
         false))))

(s/defn get-platform-kw :- s/Keyword
  []
  (cond
    (jvm?) :jvm
    (node?) :node
    (jsc-ios?) :jsc-ios
    (browser?) :browser
    :else :unknown))

;;;;;;;;;;;;;;;;;;;; UUIDs ;;;;;;;;;;;;;;;;;;;;

(def UUIDIntMap
  {:high s/Int
   :midh s/Int
   :midl s/Int
   :low s/Int})

(defprotocol IUUID
  (uuid->hex-str [this] "Return the UUID as a 36-character canonical
                         hex string.")
  (uuid->longs [this]   "Returns the UUID as two 64-bit longs.
                         High bits followed by low bits.")
  (uuid->int-map [this]
    "Returns the UUID as a map with four keys (:high, :midh, :midl, :low)
     whose values are 32-bit signed integers."))

(extend-protocol IUUID
  #?(:clj java.util.UUID :cljs cljs.core/UUID)
  (uuid->hex-str [this]
    (.toString this))

  (uuid->longs [this]
    #?(:clj [(.getMostSignificantBits this) (.getLeastSignificantBits this)]
       :cljs (let [hex-str (-> (.toString this)
                               (string/replace #"-" ""))
                   high-str (subs hex-str 0 16)
                   low-str (subs hex-str 16 32)]
               [(hex-str->long high-str) (hex-str->long low-str)])))

  (uuid->int-map [this]
    (let [[high-long low-long] (uuid->longs this)
          [high midh] (long->ints high-long)
          [midl low] (long->ints low-long)]
      (sym-map high midh midl low))))

(defn- canonicalize-hex-str [hex-str]
  (string/join "-" [(subs hex-str 0 8)
                    (subs hex-str 8 12)
                    (subs hex-str 12 16)
                    (subs hex-str 16 20)
                    (subs hex-str 20 32)]))

(s/defn longs->uuid :- UUID
  [high-long :- Long
   low-long :- Long]
  #?(:clj
     (UUID. high-long low-long)
     :cljs
     (let [high-str (long->hex-str high-long)
           low-str (long->hex-str low-long)
           hex-str (str high-str low-str)]
       (uuid (canonicalize-hex-str hex-str)))))

(s/defn int-map->uuid :- UUID
  [int-map :- UUIDIntMap]
  (let [{:keys [high midh midl low]} int-map
        high-long (ints->long high midh)
        low-long (ints->long midl low)]
    (longs->uuid high-long low-long)))

(s/defn hex-str->uuid :- UUID
  [hex-str :- s/Str]
  #?(:clj (UUID/fromString hex-str)
     :cljs (uuid hex-str)))

(s/defn int-map->hex-str :- s/Str
  [int-map :- UUIDIntMap]
  (-> int-map
      (int-map->uuid)
      (uuid->hex-str)))

(s/defn hex-str->int-map :- UUIDIntMap
  [hex-str :- s/Str]
  (-> hex-str
      (hex-str->uuid)
      (uuid->int-map)))

#?(:cljs
   (defn- set-multicast-bit [node-id-bytes]
     (let [length (.-length node-id-bytes)
           _ (when-not (= 6 (.-length node-id-bytes))
               (throw-far-error "Node id is not 6 bytes long (48 bits)"
                                :illegal-argument :bad-node-id-length
                                (sym-map length node-id-bytes)))]
       (aset node-id-bytes 0 (bit-or 0x01 (aget node-id-bytes 0)))
       node-id-bytes)))

#?(:cljs
   (defn- hex-str->byte-array [hex-str]
     (byte-array (map #(js/parseInt (join %) 16)
                      (partition 2 hex-str)))))

#?(:cljs
   (defn- get-node-id-bytes-node []
     (let [os (js/require "os")
           nis (js->clj (.networkInterfaces os))
           macs (for [k (keys nis)
                      addr (nis k)
                      :let [mac (addr "mac")]
                      :when (not= "00:00:00:00:00:00" mac)]
                  mac)
           mac-str (string/replace (first macs) ":" "")]
       (hex-str->byte-array mac-str))))

#?(:cljs
   (defn- int->byte-array [i]
     (let [a32 (js/Uint32Array. (clj->js [i]))
           dv (js/DataView. (.-buffer a32))
           output (byte-array 4)]
       (dotimes [i 4]
         (aset output i (.getInt8 dv i) true))
       output)))

#?(:cljs
   (defn- get-node-id-bytes-ios []
     (let [id-str (-> (js/identifierForVendor) ;; Returns a UUID-style string
                      (string/replace "-" ""))
           id-arr (hex-str->byte-array id-str)
           hash-arr (int->byte-array (murmur-hash id-str))
           node-id (byte-array 6)]
       ;; We need 6 bytes 16, so we concatenate the first 2
       ;; bytes of the id array with the 4 bytes from the hash of the id
       (.set node-id (.slice id-arr 0 2))
       (.set node-id hash-arr 2)
       node-id)))

#?(:cljs
   (defn- get-node-id-bytes-browser []
     ;; TODO: Implement this using a hash of user agent string,
     ;; supported mimeTypes, and count of vars in global scope.
     ;; See the api.fingerprint method here:
     ;; https://github.com/ericelliott/cuid/blob/v1.3.8/dist/browser-cuid.js
     ;; Remember to set the multicast bit, since this is not a real
     ;; MAC address (call set-multicast-bit fn).
     (throw-far-error "get-node-id-bytes-browser is not implemented yet"
                      :execution-error :not-implmented {})))

#?(:cljs
   (defn- get-node-id-bytes*
     "Return the 48-bit node id as a 12-char hex string."
     []
     (case (get-platform-kw)
       :node (get-node-id-bytes-node)
       :jsc-ios (get-node-id-bytes-ios)
       :browser (get-node-id-bytes-browser))))

#?(:cljs
   (def get-node-id-bytes (memoize get-node-id-bytes*)))

#?(:cljs
   (def uuid-gen-state
     (atom {:clock-seq (rand-int (bit-shift-left 1 14))
            :last-msecs 0
            :last-nsecs 0})))

#?(:cljs
   (defn- make-uuid-str [time-low time-mid-high clock-seq node-id-bytes]
     (let [arr (js/Uint8Array. 16)]
       ;; time-low
       (aset arr 0 (bit-and (unsigned-bit-shift-right time-low 24) 0xff))
       (aset arr 1 (bit-and (unsigned-bit-shift-right time-low 16) 0xff))
       (aset arr 2 (bit-and (unsigned-bit-shift-right time-low 8) 0xff))
       (aset arr 3 (bit-and time-low 0xff))
       ;; time-mid
       (aset arr 4 (bit-and (unsigned-bit-shift-right time-mid-high 8) 0xff))
       (aset arr 5 (bit-and time-mid-high 0xff))
       ;; time-high-and-version
       (aset arr 6 (bit-or 0x10  ;; version
                           (bit-and (unsigned-bit-shift-right time-mid-high 24)
                                    0xf)))
       (aset arr 7 (bit-and (unsigned-bit-shift-right time-mid-high 16) 0xff))
       ;; clock-seq-hi-and-reserved (RFC 4122 4.2.2)
       (aset arr 8 (bit-or 0x80  ;; variant mask
                           (unsigned-bit-shift-right clock-seq 8)))
       (aset arr 9 (bit-and clock-seq 0xff))
       ;; node-id
       (.set arr node-id-bytes 10)
       (let [hex-str (.reduce arr (fn [pv cv i arr]
                                    (let [hex (.toString cv 16)
                                          hex (if (= 1 (count hex))
                                                (str "0" hex)
                                                hex)]
                                      (str pv hex))) "")]
         (canonicalize-hex-str hex-str)))))

;; This cljs UUID v1 implementation is based heavily on this js
;; version: https://github.com/broofa/node-uuid/blob/master/uuid.js

;; UUID timestamps are 100 nano-second units since the Gregorian epoch,
;; (1582-10-15 00:00).  JSNumbers aren't precise enough for this, so
;; time is handled internally as 'msecs' (integer milliseconds) and 'nsecs'
;; (100-nanoseconds offset from msecs) since unix epoch, 1970-01-01 00:00.
#?(:cljs
   (s/defn make-v1-uuid-cljs :- UUID
     []
     (let [{:keys [clock-seq last-msecs last-nsecs]} @uuid-gen-state
           msecs (get-current-time-ms)
           ;; Per 4.2.1.2, use count of uuid's generated during the current
           ;; clock cycle to simulate higher resolution clock
           nsecs (inc last-nsecs)
           ;; Time since last uuid creation (in msecs)
           dt (+ (- msecs last-msecs)
                 (/ (- nsecs last-nsecs) 10000))
           ;; Per 4.2.1.2, Bump clock-seq on clock regression
           clock-seq (if (neg? dt)
                       (inc clock-seq)
                       clock-seq)
           ;; Reset nsecs if clock regresses or we've moved onto a new
           ;; time interval
           nsecs (if (or (neg? dt)
                         (> msecs last-msecs))
                   0
                   nsecs)
           ;; Per 4.2.1.2 Throw error if too many uuids are requested
           _ (when (>= nsecs 10000)
               (throw-far-error "Can't create more than 10M uuids/sec"
                                :execution-error :uuid-creation-rate-exceeded
                                (sym-map nsecs)))
           gregorian-msecs (+ msecs 12219292800000)
           time-low (rem (+ nsecs
                            (* 10000 (bit-and gregorian-msecs 0xfffffff)))
                         0x100000000)
           time-mid-high (bit-and 0xfffffff
                                  (* 10000 (/ msecs 0x100000000)))
           node-id-bytes (get-node-id-bytes)
           new-state {:clock-seq clock-seq
                      :last-msecs msecs
                      :last-nsecs nsecs}]
       (reset! uuid-gen-state new-state)
       (uuid (make-uuid-str time-low time-mid-high clock-seq node-id-bytes)))))

(s/defn make-v1-uuid :- UUID
  []
  #?(:clj (clj-uuid/v1)
     :cljs (make-v1-uuid-cljs)))

(s/defn make-v4-uuid :- UUID
  []
  #?(:clj (clj-uuid/v4)
     :cljs (random-uuid)))

;;;;;;;;;;;;;;;;;;;; Logging helpers ;;;;;;;;;;;;;;;;;;;;

;; This is for timbre logging
(s/defn short-log-output-fn :- s/Str
  [data :- {(s/required-key :level) s/Keyword
            s/Any s/Any}]
  (let [{:keys [level msg_]} data
        formatter (f/formatters  :hour-minute-second-ms)
        timestamp (f/unparse formatter (t/now))]
    (str
     timestamp " "
     (clojure.string/upper-case (name level))  " "
     @msg_)))

;;;;;;;;;;;;;;;;;;;; http utils ;;;;;;;;;;;;;;;;;;;;

(s/defn http-get :- Channel
  ([uri :- s/Str] (http-get uri {}))
  ([uri :- s/Str
    opts :- {}]
   (let [platform (get-platform-kw)]
     #?(:clj (go-safe
              @(http/get uri opts))
        :cljs (case platform
                :node (http-get-node uri opts)
                :browser (http/get uri opts)
                (throw-far-error
                 (str "http-get is not implemented for platform `"
                      platform "`.")
                 :execution-error :not-implmented (sym-map platform)))))))

#?(:cljs
   (defn- http-get-node [uri opts]
     (go-safe
      (let [result-chan (ca/chan 1)
            body (atom "")
            headers (atom nil)
            status (atom nil)
            error (atom nil)
            uri-pat #"(http[s]?://)?([^:/\s]+)(.*)"
            [uri protocol host path] (re-find uri-pat uri)
            http (js/require (case protocol
                               nil "http"
                               "http://" "http"
                               "https://" "https"
                               (throw-far-error
                                (str "Unknown protocol: " protocol)
                                :illegal-argument :unknown-protocol
                                (sym-map protocol))))
            opts (merge opts (sym-map host path))

            data-cb (fn [data]
                      (swap! body str data))
            end-cb (fn []
                     (ca/put! result-chan {:body @body
                                           :headers @headers
                                           :status @status
                                           :error @error}))
            cb (fn [resp]
                 (reset! headers (js->clj (aget resp "headers")
                                          :keywordize-keys true))
                 (reset! status (.-statusCode resp))
                 (.on resp "data" data-cb)
                 (.on resp "end" end-cb))
            error-cb (fn [err]
                       (reset! error err)
                       (end-cb))
            client (.get http (clj->js opts) cb)]
        (.on client "error" error-cb)
        (ca/<! result-chan)))))
