(ns farbetter.utils.attempt-strategies)

;; Forked from https://github.com/listora/again
;; This file is licensed under the Eclipse Public License
;; either version 1.0 or (at your option) any later version.

(defn make-integer [n]
  (#?(:clj int :cljs Math/round) n))

(defn constant-strategy
  "Generates a retry strategy with a constant delay (ms) between
  retries, ie the delay is the same for each retry."
  [delay-ms]
  {:pre [(>= delay-ms 0)]}
  (repeat delay-ms))

(defn immediate-strategy
  "Returns a retry strategy that retries without any delay."
  []
  (constant-strategy 0))

(defn additive-strategy
  "Returns a retry strategy where, after the `initial-delay-ms` (ms), the
  delay increases by `increment` (ms) after each retry. The single
  argument version uses the given increment as both the initial delay
  and the increment."
  ([increment]
   (additive-strategy increment increment))
  ([initial-delay-ms increment]
   {:pre [(>= initial-delay-ms 0)
          (>= increment 0)]}
   (iterate #(+ increment %) (make-integer initial-delay-ms))))

(defn stop-strategy
  "A no-retries policy."
  []
  nil)

(defn multiplicative-strategy
  "Returns a retry strategy with exponentially increasing delays, ie
  each previous delay is multiplied by delay-multiplier to generate
  the next delay."
  [initial-delay-ms delay-multiplier]
  {:pre [(<= 0 initial-delay-ms)
         (<= 0 delay-multiplier)]}
  (iterate #(* delay-multiplier %) (make-integer initial-delay-ms)))

(defn- randomize-delay
  "Returns a random delay from the range [`delay` - `delta`, `delay` + `delta`],
  where `delta` is (`rand-factor` * `delay`). Note: return values are
  rounded to whole numbers, so eg (randomize-delay 0.8 1) can return
  0, 1, or 2."
  [rand-factor delay-ms]
  {:pre [(< 0 rand-factor 1)]}
  (let [delta (* delay-ms rand-factor)
        min-delay-ms (- delay-ms delta)
        max-delay-ms (+ delay-ms delta)]
    ;; The inc is there so that if min-delay-ms is 1 and max-delay-ms is 3,
    ;; then we want a 1/3 chance for selecting 1, 2, or 3.
    ;; Cast the delay-ms to an int.
    (make-integer (+ min-delay-ms
                     (* (rand) (inc (- max-delay-ms min-delay-ms)))))))

(defn randomize-strategy
  "Returns a new strategy where all the delays have been scaled by a
  random number between [1 - rand-factor, 1 + rand-factor].
  Rand-factor must be greater than 0 and less than 1."
  [rand-factor retry-strategy]
  {:pre [(< 0 rand-factor 1)]}
  (map #(randomize-delay rand-factor %) retry-strategy))

(defn max-retries
  "Stop retrying after `n` retries."
  [n retry-strategy]
  {:pre [(>= n 0)]}
  (take n retry-strategy))

(defn max-attempts
  "Stop retrying after `n` attempts."
  [n timeout-strategy]
  {:pre [(>= n 0)]}
  (take n timeout-strategy))

(defn clamp-delay
  "Replace delays in the strategy that are larger than `delay-ms` with
  `delay-ms`."
  [delay-ms retry-strategy]
  {:pre [(>= delay-ms 0)]}
  (map #(min delay-ms %) retry-strategy))

(defn max-delay-ms
  "Stop retrying once the a delay is larger than `delay-ms`."
  [delay-ms retry-strategy]
  {:pre [(>= delay-ms 0)]}
  (take-while #(< % delay-ms) retry-strategy))

(defn max-duration
  "Limit the maximum wallclock time of the operation to `timeout` (ms)"
  [timeout retry-strategy]
  (when (and (pos? timeout) (seq retry-strategy))
    (let [[f & r] retry-strategy]
      (cons f
            (lazy-seq (max-duration (- timeout f) r))))))
