(ns farg.with-state
  (:require [farg.pmatch :refer [pmatch-loop pmatch-recur]]))

(def ^:dynamic body-throws)
  ;Boolean: tracks whether the code generated by the current with-state
  ;generates code to throw an exception.

(defn- insert-state [state expr]
  (pmatch-loop [expr expr, state state]
    (apply ~f ~@args)
      `(apply ~f ~state ~@args)
    (~f ~@args)
      `(~f ~state ~@args)
    ~x
      (pmatch-recur (list expr) state)))

(declare with-state-body)

(defn- make-doseq [state bindings body]
  (if (empty? bindings)
    (with-state-body state body)
    (let [[[bind-to bind-from] more] (split-at 2 bindings)]
      `(reduce (fn [~state ~bind-to]
                 ~(make-doseq state more body))
               ~state
               ~bind-from))))

(defn- with-state-body [state body]
  (pmatch-loop [body body, k identity]
    ()
      (k state)
    ((setq ~v ~expr) ~@more)  ;TODO Make this work so that you can set a
      (pmatch-recur more      ;variable inside an 'if' and access it afterward
        (fn [inside]
          (k `(let [[~state ~v] ~(insert-state state expr)]
                  ~inside))))
    ((bind ~v ~expr) ~@more)  ;TODO Same as for setq
      (pmatch-recur more
        (fn [inside]
          (k `(let [~v ~expr]
                  ~inside))))
    ((if ~c ~then ~else) ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(let [~state (if ~c
                                ~(with-state-body state (list then))
                                ~(with-state-body state (list else)))]
                ~inside))))
    ((when ~c ~@body) ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(let [~state (if ~c
                               ~(with-state-body state body)
                               ~state)]
                ~inside))))
    ((when-let [~v ~expr] ~@body) ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(let [~state (if-let [~v ~expr]
                             ~(with-state-body state body)
                             ~state)]
                ~inside))))
    ((do ~@body) ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(let [~state ~(with-state-body state body)]
                ~inside))))
    ((dotimes [~v ~expr] ~@body) ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(let [~state
                       (reduce
                         (fn [~state ~v]
                           ~(with-state-body state body))
                         ~state
                         (range ~expr))]
                 ~inside))))
    ((doseq ~bindings ~@body) ~@more)
      (pmatch-recur more
        (fn [inside]
          ;TODO Throw error if bindings doesn't have even num of elems
          (k `(let [~state ~(make-doseq state bindings body)]
                ~inside))))
    ((while ~test ~@body) ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(let [~state (loop [~state ~state]
                             (if ~test
                               (recur ~(with-state-body state body))
                               ~state))]
                 ~inside))))
    ((return ~expr) ~@more)
      (pmatch-recur more
        (do
          (set! body-throws true)
          (fn [inside]
            (k `(throw (ex-info "Early return from with-state"
                                {::return ~expr}))))))
    ((is ~expr) ~@more)  ;; 'is' doesn't insert state
      (pmatch-recur more
        (fn [inside]
          (k `(do
                (clojure.test/is ~expr)
                ~inside))))
    (-- ~expr ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(do (~@expr)
                  ~inside))))
    ((~f ~@args) ~@more)
      (pmatch-recur more
        (fn [inside]
          (k `(let [~state ~(insert-state state (list* f args))]
                  ~inside))))
    (~atom ~@more)
      (pmatch-recur `(~(list atom) ~@more) k)))

(defmacro with-state
  "Threads a state variable as the first argument through all the expressions
  in body (like ->) but the state variable is named and taken from bindvec,
  the state variable as rebound to the result of each expression (like as->),
  and some common Clojure functions are redefined inside with-state to
  enable many common operations that can't be done with ->. with-state
  returns the value of the state variable when the last expression in body
  terminates or the argument to a 'return' statement if reached (see below).

  For example, this expression:

    (with-state [state {:a 0, :b 0}]
      (doseq [x [1 2 3 4]]         ;doseq is rewritten as reduce
        (update :a #(+ % x))       ;'state' is implicit first argument
        (when (odd? x)             ;conditional execution
          (update :b #(+ % x)))))  ;'state' is implicit first argument

  returns {:a 10, :b 4}.

  The following are redefined inside with-state:

    (if c t f)
      Conditional execution. c is not rewritten. t and f are rewritten the
      same as any other with-state line.

    (when c exprs ...)
      Conditional execution. c is not rewritten. exprs are rewritten the same
      as any other with-state line.

    (when-let [v c] exprs ...)
      Conditional execution. c is not rewritten. If c is logical true, it
      is assigned to v and exprs are executed. exprs are rewritten the same
      as any other with-state line.

    (return x)
      Premature exit from with-state, returning x. If invoked inside
      conditionally or repeatedly executed code as in an 'if', 'when', or
      'doseq', exits the entire with-state.

    (apply args ...)
      The state variable is not inserted into an 'apply'. The state variable
      variable is rebound to the result of the 'apply'.

    (doseq bindings exprs ...)
      Loops variables through values as in ordinary 'doseq'. The expressions
      assigned in the bindings are not rewritten. exprs are rewritten the same
      as any other with-state line.

    (dotimes [i n] exprs ...)
      Executes exprs n times, successively assigning the numbers 0..n to i
      on each iteration of the loop. n is not rewritten. exprs are rewritten
      the same as any other with-state line.

    (while c exprs ...)
      Executes exprs repeatedly as long as c is logical true. c is not
      rewritten. exprs are rewritten the same as any other with-state line.

    (setq v expr)
      Rewrites expr as normal. expr is expected to return a pair [state x],
      where 'state' is the new value of the state variable and 'x' is the
      value to assign to v.

    (bind v expr)
      Sets v to the value of expr. Does not rewrite expr.

    -- expr
      expr is not rewritten and the state variable is not bound to its result.
      This makes it easy to insert println statements or other statements
      that you don't want to affect the state variable."
  [bindvec & body]
  (when (not (and (vector? bindvec)
                (= 2 (count bindvec))))
      (throw (IllegalArgumentException. (str "The first argument to with-state "
               "must be a binding vector containing a single binding pair. "
               "Instead, got: " bindvec ". Check to see if you have a "
               "with-state inside another with-state."))))
    (binding [body-throws false]
      (let [[state init-state] bindvec
            body (if (nil? body) () body)
            body (with-state-body state body)]
        (if body-throws
          `(try
             (let [~state ~init-state]
               ~body)
             (catch clojure.lang.ExceptionInfo e#
               (if (contains? (ex-data e#) :farg.with-state/return)
                 (get (ex-data e#) :farg.with-state/return)
                 (throw e#))))
          `(let [~state ~init-state]
             ~body)))))
