
(defmacro -> [x & forms]
  (loop [x x, forms forms]
    (if forms
      (let [form (first forms)
            threaded (if (seq? form)
                       `(~(first form) ~x ~@(next form))
                       (list form x))]
        (recur threaded (next forms)))
      x)))
(defmacro ->> [x & forms]
  (loop [x x, forms forms]
    (if forms
      (let [form (first forms)
            threaded (if (seq? form)
                       `(~(first form) ~@(next form)  ~x)
                       (list form x))]
        (recur threaded (next forms)))
      x)))
(defmacro defn [name & body]
  `(def ~name (fn ~@body)))
(defmacro fn [& sig]
  (let [name (if (symbol? (first sig)) (first sig) nil)
        body (if name (rest sig) sig)]
    (if (vector? (first body))
      (let [[args & body] body]
        (new-fir-fn :name name :args args :body body))
      ;; handle multi arity function
      (let [fns   (map (fn* [body]
                            (let [[args & body] body]
                              (new-fir-fn :args args :body body)))
                       body)
            arity (->> (map first body)
                       (map (fn* [args] (filter #(not (= % '&)) args)))
                       (map #(count %)))
            fns   (->> (interleave arity fns)
                       (partition 2)
                       (sort-by first))
            fns   (if (->> fns last second second      ;; last arity arguments
                           (take-last 2) first (= '&)) ;; check &
                    (let [switch        (drop-last 1 fns)
                          [[_ default]] (take-last 1 fns)]
                      `(fir-defn-arity ~switch ~default))
                    `(fir-defn-arity ~fns))]
        (new-fir-fn :escape false :name name :body [fns])))))
(defmacro cxx [& body]
  (let [body (apply str body)]
    `((fn [] ~body))))
(defmacro defnative [name args & form]
  (let [includes (->> (filter #(seq? (nth % 2)) form)
                      (map #(cons (nth % 1) (apply list (nth % 2))))
                      (map (fn [form]
                             (let [[guard & headers] form]
                               (str "\n#if " guard " \n"
                                    (apply str (map #(str "#include \"" % "\"\n") headers))
                                    "#endif\n"))))
                      (map #(list 'native-declare %)))
        enabled (-> (symbol-conversion name)
                    (str "_enabled")
                    .toUpperCase)
        body (->> (map #(vector (second %) (last %)) form)
                  (map #(str "\n#if " (first %) " \n"
                             "#define " enabled "\n"
                             (second %)
                             "\n#endif\n"))
                  (apply str))
        body (str body
                  "\n#if !defined " enabled " \n"
                  "# error " (symbol-conversion name)
                  " Not Supported on This Platform \n"
                  "#endif\n")
        pre-ample (->> (map #(vector (second %) (drop-last (drop 3 %))) form)
                       (remove #(empty? (second %)))
                       (map #(str "\n#if " (first %) " \n"
                                  (apply str (map (fn [line] (str line "\n")) (second %)))
                                  "\n#endif\n"))
                       (map #(list 'native-declare %)))]
    `(def ~name (fn ~args ~@includes ~@pre-ample  ~body))))
(defobject seekable_i "ferret/obj/seekable_i.h")
(defobject lambda_i "ferret/obj/lambda_i.h")
(defobject deref_i "ferret/obj/deref_i.h")
(defn deref [a]
  "return a.cast<deref_i>()->deref();")
(defobject boolean "ferret/obj/boolean_o.h")
(defobject pointer "ferret/obj/pointer_o.h")
(defobject value "ferret/obj/value_o.h")
(defobject number "ferret/obj/number_o.h")
(defobject empty_sequence "ferret/obj/empty_sequence_o.h")
(defobject sequence "ferret/obj/sequence_o.h")
(defobject lazy_sequence "ferret/obj/lazy_sequence_o.h")
(defn new-lazy-seq [f]
  "return obj<lazy_sequence>(f);")

(defmacro lazy-seq [& body]
  `(new-lazy-seq (fn [] ~@body)))
(defobject d-list "ferret/obj/d_list_o.h")

(defn new-d-list-aux [keys vals]
  "return obj<d_list>(keys, vals);")

(defmacro new-d-list [& args]
  (let [kvs (partition 2 args)
        keys (map first kvs)
        vals (map second kvs)]
    `(new-d-list-aux
      (list ~@keys) (list ~@vals))))

(defn assoc [m k v]
  "return m.cast<map_t>()->assoc(k,v);")

(defn dissoc [m k]
  "return m.cast<map_t>()->dissoc(k);")

(defn get [m & args]
  "return m.cast<map_t>()->val_at(args);")

(defn vals [m]
  "return m.cast<map_t>()->vals();")

(defn keys [m]
  "return m.cast<map_t>()->keys();")
(defobject keyword "ferret/obj/keyword_o.h")
(defobject string "ferret/obj/string_o.h")
(defn new-string
  ([& ss]
   ((fn [s] "return obj<string>(s);")
    (reduce (fn [h v] (concat h v)) ss))))
(defobject atomic "ferret/obj/atomic_o.h")
(defn atom [x]
  "return obj<atomic>(x)")

(defn swap! [a f & args]
  "return a.cast<atomic>()->swap(f,args);")

(defn reset! [a newval]
  "return a.cast<atomic>()->reset(newval);")
(defobject async "ferret/obj/async_o.h")

(defmacro future [& body]
  `(_future_ (fn [] ~@body)))

(defn _future_ [f] "return obj<async>(f);")

(defn future-done? [f] "return obj<boolean>(f.cast<async>()->is_ready());")
(defobject delayed "ferret/obj/delayed_o.h")

(defn _delay_ [f]
  "return obj<delayed>(f)")

(defmacro delay [& body]
  `(_delay_ (fn [] ~@body)))

(defn delay? [d]
  "return obj<boolean>(d.is_type(type_id<delayed>));")

(defn force [d] @d)
(defn list [& xs] "if (xs.is_nil())
                     return runtime::list();
                   else
                     return xs;")
(defn list? [x] "return obj<boolean>(x.is_type(type_id<sequence>));")
(defn empty? [x]
  (if (nil? x)
    true
    (= (list ) x)))
(defn cons [x seq] "return runtime::cons(x, seq);")
(defn first [x]
  "return runtime::first(x);")
(defn second [x]
  "return runtime::first(runtime::rest(x));")
(defn rest [x] "ref r = runtime::rest(x);
                if (r.is_nil())
                  return runtime::list();
                else 
                  return r;")
(defn nth [coll ^number_t index] "return runtime::nth(coll,index);")
(defn nthrest [coll ^number_t n]
  "return runtime::nthrest(coll,n);")
(defn reduce
  ([f xs]
   (reduce f (first xs) (rest xs)))
  ([f acc coll]
   "__result = acc;
    for_each(i, coll)
     __result = run(f, __result, i);"))
(defn apply [f & argv]
  "if (runtime::rest(argv) == runtime::list())
     return f.cast<lambda_i>()->invoke(runtime::first(argv));

   struct{
     var operator()(ref seq) const { 
       ref head = runtime::first(seq);

       if (head.cast<seekable_i>())
         return head;
       else
         return runtime::cons(head, (*this)(runtime::rest(seq)));
       }
   } spread;

   return f.cast<lambda_i>()->invoke(spread(argv));")
(defn conj [coll & xs]
  (reduce (fn[h v] (cons v h)) (if (nil? coll) (list) coll) xs))
(defn reverse [s]
  (reduce (fn[h v] (cons v h)) (list) s))
(defn true? [x]
  "if (x)
     return cached::true_t;
   return cached::false_t;")
(defn false? [x]
  "if (!x)
     return cached::true_t;
   return cached::false_t;")
(defn nil? [x] "return obj<boolean>(x.is_nil())")
(defn not [x]
  "if (x)
     return cached::false_t;
   return cached::true_t;")
(defn = [& args]
  "var curr = runtime::first(args);
   for_each(it, runtime::rest(args)){
    if (curr != it)
      return cached::false_t;
    curr = it;
   }
   return cached::true_t;")
(defmacro not= [& test]
  `(not (= ~@test)))
(defn identical? [x y]
  "if(x.get() == y.get())
    return cached::true_t;
   return cached::false_t;")
(defn < [& args]
  "var a = runtime::first(args);

   for_each(b, runtime::rest(args)){
    if (number::to<real_t>(a) >= number::to<real_t>(b))
      return cached::false_t;
    a = b;
   }

   return cached::true_t;")
(defn > [& args]
  "var a = runtime::first(args);

   for_each(b, runtime::rest(args)){
    if (number::to<real_t>(a) <= number::to<real_t>(b))
      return cached::false_t;
    a = b;
   }

   return cached::true_t;")
(defn >= [& args]
  "var a = runtime::first(args);

   for_each(b, runtime::rest(args)){
    if (number::to<real_t>(a) < number::to<real_t>(b))
      return cached::false_t;
    a = b;
   }

   return cached::true_t;")
(defn <= [& args]
  "var a = runtime::first(args);

   for_each(b, runtime::rest(args)){
    if (number::to<real_t>(a) > number::to<real_t>(b))
      return cached::false_t;
    a = b;
   }

   return cached::true_t;")
(defmacro and
  ([] true)
  ([x] x)
  ([x & next]
   `(if ~x (and ~@next) false)))
(defmacro or
  ([] nil)
  ([x] x)
  ([x & next]
   `(if ~x ~x (or ~@next))))
(defmacro when [test & body]
  `(if ~test (do ~@body)))
(defmacro cond [& clauses]
  (when clauses
    `(if ~(first clauses)
       ~(if (next clauses)
          (second clauses)
          (throw (IllegalArgumentException.
                  "cond requires an even number of forms")))
       (cond ~@(next (next clauses))))))
(defn _while_ [pred fn]
  "while(run(pred))
     run(fn);")

(defmacro while [test & body]
  `(_while_ (fn [] ~test) (fn [] ~@body)))
(defmacro forever [& body]
  `(while true ~@body))
(defmacro if-let
  ([bindings then]
   `(if-let ~bindings ~then nil))
  ([bindings then else & oldform]
   (let [form (bindings 0) tst (bindings 1)]
     `(let* [temp# ~tst]
        (if temp#
          (let* [~form temp#]
            ~then)
          ~else)))))
(defmacro when-let
  [bindings & body]
  (let [form (bindings 0) tst (bindings 1)]
    `(let* [temp# ~tst]
       (when temp#
         (let* [~form temp#]
           ~@body)))))
(defn zero? [x]
  (= x 0))
(defn pos? [x]
  (> x 0))
(defn neg? [x]
  (< x 0))
(defn + [& args]
  "real_t value(0.0);

   for_each(i, args)
    value = value + number::to<real_t>(i);

   __result = obj<number>(value);")
(defn - [& args]
  "__result = runtime::first(args);
   real_t value = number::to<real_t>(__result);
   size_t count = 1;

   for_each(i, runtime::rest(args)){
    value = (value - number::to<real_t>(i));
    count++;
   }

   if (count == 1)
    value = value * real_t(-1.0);

   __result = obj<number>(value);")
(defn * [& args]
  "real_t value(1.0);

   for_each(i, args)
    value = (value * number::to<real_t>(i));

   __result = obj<number>(value);")
(defn / [& args]
  "__result = runtime::first(args);
   real_t value = number::to<real_t>(__result);
   size_t count = 1;

   for_each(i, runtime::rest(args)){
    value = (value / number::to<real_t>(i));
    count++;
   }

   if (count == 1)
    value = 1.0 / value;

   __result = obj<number>(value);")
(defn inc [x]
  (+ x 1))
(defn dec [x]
  (- x 1))
(defn count [s] "return obj<number>(runtime::count(s))")
(defn min [& args]
  "__result = runtime::first(args);
   for_each(i, runtime::rest(args))
    if (number::to<real_t>(__result) > number::to<real_t>(i))
     __result = i;")
(defn max [& args]
  "__result = runtime::first(args);
   for_each(i, runtime::rest(args))
    if (number::to<real_t>(__result) < number::to<real_t>(i))
     __result = i;")
(defn rem [^number_t num ^number_t div]
  "return obj<number>((num % div));")
(defn mod [num div] 
  (let [m (rem num div)] 
    (if (or (zero? m) (= (pos? num) (pos? div)))
      m 
      (+ m div))))
(defn floor [^number_t x] "return obj<number>(x);")
(defn scale [x in-min in-max out-min out-max]
  (+ (/ (* (- x in-min) (- out-max out-min)) (- in-max in-min)) out-min))
(defn clamp [x min max]
  (cond
    (> x max) max
    (< x min) min
    true x))
(defn bit-and [^number_t x ^number_t y] "return obj<number>((x & y));")
(defn bit-not [^number_t x] "return obj<number>(~x);")
(defn bit-or [^number_t x ^number_t y] "return obj<number>((x | y ));")
(defn bit-xor [^number_t x ^number_t y] "return obj<number>((x ^ y ));")
(defn bit-shift-left [^number_t x ^number_t n] "return obj<number>((x << n ));")
(defn bit-shift-right [^number_t x ^number_t n] "return obj<number>((x >> n ));")
(defn bit-extract [^number_t x ^number_t p ^number_t k]
  "__result = obj<number>((x >> p) & ((1 << k) - 1));")
(defn bit-override [^number_t dst ^number_t src ^number_t pos ^number_t len]
  "number_t mask = (((number_t)1 << len) - 1) << pos;
   number_t num = (dst & ~mask) | (src & mask);
   return obj<number>(num);")
(defn encode-int16 [n] 
  "int16_t val = number::to<int16_t>(n);
   byte *p = (byte*)&val;
   __result = runtime::list();
   for (int i = (sizeof(int16_t) -1); i >= 0; i--)
     __result = runtime::cons(obj<number>((number_t)p[i]),__result);")
(defn decode-int16 [s]
  "int16_t val = 0;
   byte *p = (byte*)&val; 

   size_t index = 0;
   for_each(i, s){
     p[index] = number::to<byte>(i);
     index++;
   }

   return obj<number>(val);")
(defn encode-float [n] 
  "static_assert(sizeof(float) == 4 * sizeof(byte), \"\");
   float val = number::to<float>(n);
   byte *p = (byte*)&val;
   __result = runtime::list();
   for (int i = (sizeof(float) -1); i >= 0; i--)
     __result = runtime::cons(obj<number>(p[i]),__result);")
(defn decode-float [s]
  "union {
    float f;
    byte b[4];
   } u;
   static_assert(sizeof(float) == 4 * sizeof(byte), \"\");

   size_t index = 0;
   for_each(i, s){
     if (index > 3)
      break;
     u.b[index] = number::to<byte>(i);
     index++;
   }

   return obj<number>(u.f);")
(defn sqrt [^real_t s]
  "return obj<number>(::sqrt(s));")
(defn pow [^real_t b ^real_t e]
  "return obj<number>(::pow(b, e));")
(defn cos [^real_t s]
  "return obj<number>(::cos(s));")
(defn sin [^real_t s]
  "return obj<number>(::sin(s));")
(defn asin [^real_t x]
  "return obj<number>(::asin(x));")
(defn atan2 [^real_t x ^real_t y]
  "return obj<number>(::atan2(x,y));")
(defn log [^real_t x]
  "return obj<number>(::log(x));")
(defn log10 [^real_t x]
  "return obj<number>(::log10(x));")
(defn to-degrees [^real_t x]
  "return obj<number>((x * 180.0 / 1_pi));")
(defn to-radians [^real_t x]
  "return obj<number>((x * 1_pi / 180.0));")
(defn identity [x] x)
(defn thread [f]
  "return obj<async>(f);")
(defnative get-char []
  (on "defined FERRET_STD_LIB"
      "return obj<number>(getchar());"))
(defnative sleep [^number_t t]
  (on "defined FERRET_STD_LIB"
      "auto duration = ::std::chrono::milliseconds(t);
       ::std::this_thread::sleep_for(duration);")
  (on "defined FERRET_HARDWARE_ARDUINO"
      "::delay(t);"))
(defmacro doseq [binding & body]
  `(_doseq_ ~(second binding)
            (fn [~(first binding)] ~@body)))

(defn _doseq_ [seq f] "for_each(it, seq) run(f,it);")
(defmacro dotimes [binding & body]
  `(_dotimes_ ~(second binding)
              (fn [~(first binding)] ~@body)))

(defn _dotimes_ [^number_t t f] "for(number_t i = 0; i < t; i++) run(f,obj<number>(i));")
(defn map
  ([f col]
   (if (not (empty? col))
     (cons (f (first col))
           (lazy-seq (map f (rest col))))))
  ([f s1 s2]
   (lazy-seq
    (when (and (not (empty? s1))
               (not (empty? s2)))
      (cons (f (first s1) (first s2))
            (map f (rest s1) (rest s2)))))))
(defn range
  ([high]
   (range 0 high))
  ([low high]
   (if (< low high)
     (cons low (lazy-seq
                (range (inc low) high))))))
(defn take [n coll]
  (if (not (empty? coll))
    (if (> n 0)
      (cons (first coll)
            (lazy-seq (take (- n 1) (rest coll)))))))
(defn take-while [pred s]
  (if (and (not (empty? s))
           (pred (first s)))
    (cons (first s) (lazy-seq (take-while pred (rest s))))))
(defn drop [n coll]
  (if (and (pos? n)
           (not (empty? coll)))
    (drop (dec n) (rest coll))
    coll))
(defn drop-while-aux [p c]
  "__result = c;
   while(run(p,__result))
     __result = runtime::rest(__result);")

(defn drop-while [pred coll]
  (lazy-seq
   (drop-while-aux
       (fn [c]
         (and (not (empty? c))
              (pred (first c))))
     coll)))
(defn concat
  ([]
   (list))
  ([x]
   (if (not (empty? x))
     (cons (first x) (lazy-seq (concat (rest x))))))
  ([x y]
   (if (not (empty? x))
     (cons (first x) (lazy-seq (concat (rest x) y)))
     (concat y))))
(defn filter [pred coll]
  (if (not (empty? coll))
    (let [[f & r] coll]
      (if (pred f)
        (cons f (filter pred r))
        (filter pred r)))
    coll))
(defn repeatedly
  ([f] (cons (f) (lazy-seq (repeatedly f))))
  ([n f] (take n (repeatedly f))))
(defn partition
  ([n coll]
   (partition n n coll))
  ([n step coll]
   (lazy-seq
    (if (not (empty? coll))
      (let [p (take n coll)]
        (when (= n (count p))
          (cons p (partition n step (nthrest coll step))))))))
  ([n step pad coll]
   (lazy-seq
    (if (not (empty? coll))
      (let [p (take n coll)]
        (if (= n (count p))
          (cons p (partition n step pad (nthrest coll step)))
          (list (take n (concat p pad)))))))))
(defn every? [pred coll]
  "for_each(i, coll){
     if (!run(pred, i))
      return cached::false_t;
   }
   return cached::true_t;")
(defn interleave
  ([s1 s2]
   (lazy-seq
    (when (and (not (empty? s1))
               (not (empty? s2)))
      (cons (first s1) (cons (first s2) 
                             (interleave (rest s1) (rest s2))))))))    
(defnative rand-aux []
  (on "defined FERRET_STD_LIB"
      ("random")
      "::std::random_device ferret_random_device;
       ::std::mt19937_64 ferret_random_generator(ferret_random_device());
       ::std::uniform_real_distribution<ferret::real_t> ferret_random_distribution(0.0,1.0);"
      "return obj<number>(ferret_random_distribution(ferret_random_generator));"))

(defn rand
  ([]
   (rand-aux))
  ([x]
   (* x (rand-aux))))
(defn rand-int
  [x]
  (floor (rand x)))
(defnative millis []
  (on "defined FERRET_STD_LIB"
      "auto now = ::std::chrono::system_clock::now();
       auto epoch = now.time_since_epoch();
       auto time = ::std::chrono::duration_cast<::std::chrono::milliseconds>(epoch).count();
       return obj<number>(time);")
  (on "defined FERRET_HARDWARE_ARDUINO"
      "return obj<number>(::millis());"))
(defnative micros []
  (on "defined FERRET_STD_LIB"
      "auto now = ::std::chrono::high_resolution_clock::now();
       auto epoch = now.time_since_epoch();
       auto time = ::std::chrono::duration_cast<::std::chrono::microseconds>(epoch).count();
       return obj<number>(time);")
  (on "defined FERRET_HARDWARE_ARDUINO"
      "return obj<number>(::micros());"))
(defnative sleep-micros [^number_t t]
  (on "defined FERRET_STD_LIB"
      "auto duration = ::std::chrono::microseconds(t);
       ::std::this_thread::sleep_for(duration);")
  (on "defined FERRET_HARDWARE_ARDUINO"
      "::delayMicroseconds(t);"))
(defobject elapsed_micros "ferret/obj/elapsed_micros_o.h")

(defn new-elapsed-micros []
  "return obj<elapsed_micros>();")

(defn elapsed-micros? [t ^real_t r]
  "return obj<boolean>(t.cast<elapsed_micros>()->is_elapsed(r));")

(defn elapsed-micros-now [t]
  "return obj<number>(t.cast<elapsed_micros>()->elapsed());")

(defn elapsed-micros-reset [t]
  "t.cast<elapsed_micros>()->reset()")
(defn time-fn [f]
  (let [start (millis)]
    (f)
    (- (millis) start)))
(defn benchmark [f n]
  (let [values (map (fn [_] (time-fn f)) (range n))]
    (floor (/ (apply + values) n))))
(defn fn-throttler-aux-blocking [timer f rate]
  (fn [& args]
    (let [wait (- rate (elapsed-micros-now timer))]
      (elapsed-micros-reset timer)
      (sleep-micros wait)
      (apply f args))))

(defn fn-throttler-aux-non-blocking [timer f rate]
  (fn [& args]
    (when (elapsed-micros? timer rate)
      (elapsed-micros-reset timer)
      (apply f args))))

(defmacro fn-throttler [f rate unit policy]
  (let [unit->ms {:microsecond 1 :millisecond 1000
                  :second 1000000 :minute 60000000
                  :hour 3600000000 :day 86400000000
                  :month 2678400000000}
        rate (/ (unit->ms unit) rate)]
    (if (= policy :blocking)
      `(fn-throttler-aux-blocking     (new-elapsed-micros) ~f ~rate)
      `(fn-throttler-aux-non-blocking (new-elapsed-micros) ~f ~rate))))
(defn ston [str]
  "var seq = str;
   real_t factor = 1;
   real_t value = 0;

   if (number::to<byte>(runtime::first(seq)) == '-'){
     seq = runtime::rest(seq);
     factor = -1;
   }

   bool point_passed = false;
   for_each(i, seq){
     byte ch = number::to<byte>(i);

     if (ch == '.'){
       point_passed = true;
       continue;
     }

     number_t d = ch - '0';
     if (d >= 0 && d <= 9){
       if (point_passed)
         factor /= 10.0f;
       value = value * 10.0f + (real_t)d;
     }
   }
  
   return obj<number>(value * factor);")
(defn ntos [^real_t f]
  "number_t n = (number_t)f;
   number_t sign;

   if ((sign = n) < 0){
      n = -n;
      f = -f;
   }

   var s;

   f = (f - n) + 10;
   for (int i = number_precision; i >= 1; i--){
      number_t ch = ((number_t)(f * ::pow(10, i)) % 10) + '0';
      s = runtime::cons(obj<number>(ch), s);
   }

   s = runtime::cons(obj<number>('.'), s);

   do {
     s = runtime::cons(obj<number>(n % 10 + '0'), s);
    } while ((n /= 10) > 0);

   if (sign < 0)
     s = runtime::cons(obj<number>('-'), s);

   return obj<string>(s);")
(defmacro doto
  [x & forms]
  (let [gx (gensym)]
    `(let [~gx ~x]
       ~@(map (fn [f]
                (if (seq? f)
                  `(~(first f) ~gx ~@(next f))
                  `(~f ~gx)))
              forms)
       ~gx)))
(defnative print [& more]
  (on "!defined(FERRET_DISABLE_STD_OUT)"
      "if (more.is_nil())
         return nil();
       ref f = runtime::first(more);
       f.stream_console();
       ref r = runtime::rest(more);
       for_each(it, r){
        runtime::print(\" \");
        it.stream_console();
       }
       return nil();"))
(defnative newline []
  (on "!defined(FERRET_DISABLE_STD_OUT)"
      "runtime::print(\"\\n\");"))
(defn println [& more]
  (when more
    (apply print more))
  (newline))
(defn read-line []
  "char buf[FERRET_IO_STREAM_SIZE] = {0};
   runtime::read_line(buf, FERRET_IO_STREAM_SIZE);
   return obj<string>(buf);")
(defnative slurp [^std_string f]
  (on "defined FERRET_STD_LIB"
      ("fstream")
      "std::ifstream ifs(f.c_str(), std::ios::in | std::ios::binary | std::ios::ate);
 
       if (!ifs.good())
         return nil();
 
       std::ifstream::pos_type file_size = ifs.tellg();
       ifs.seekg(0, std::ios::beg);

       std::vector<char> bytes(file_size);
       ifs.read(bytes.data(), file_size);
       return obj<string>(std::string(bytes.data(), file_size));"))
(defnative sh [^std_string cmd]
  (on "defined FERRET_STD_LIB"
      ("memory")
      "::std::shared_ptr<FILE> pipe(popen(cmd.c_str(), \"r\"), pclose);
       if (!pipe) 
         return nil();
       char buffer[128];
       ::std::string result = \"\";
       while (!feof(pipe.get()))
        if (fgets(buffer, 128, pipe.get()) != NULL)
         result += buffer;
       return obj<string>(result);"))
(defnative lock-memory []
  (on "defined FERRET_STD_LIB"
      ("sys/mman.h")
      "mlockall(MCL_CURRENT | MCL_FUTURE);"))
(defn pr-object-sizes []
  (println "Object Sizes")
  (println "\tvar:\t\t\t" (cxx "return obj<number>(sizeof(var));"))
  (println "\tobject:\t\t\t" (cxx "return obj<number>(sizeof(object));"))
  (println "\tpointer:\t\t" (cxx "return obj<number>(sizeof(pointer));"))
  (println "\tnumber:\t\t\t" (cxx "return obj<number>(sizeof(number));"))
  (println "\tkeyword:\t\t" (cxx "return obj<number>(sizeof(keyword));"))
  (println "\tempty_sequence:\t\t" (cxx "return obj<number>(sizeof(empty_sequence));"))
  (println "\tsequence:\t\t" (cxx "return obj<number>(sizeof(sequence));"))
  (println "\tlazy_sequence:\t\t" (cxx "return obj<number>(sizeof(lazy_sequence));"))
  (println "\tstring:\t\t\t" (cxx "return obj<number>(sizeof(string));"))
  (println "\tboolean:\t\t" (cxx "return obj<number>(sizeof(boolean));"))
  (println "\tlambda_i:\t\t" (cxx "return obj<number>(sizeof(lambda_i));"))
  (println "\tatom:\t\t\t" (cxx "return obj<number>(sizeof(atomic));"))
  (println "\telapsed_micros:\t\t" (cxx "return obj<number>(sizeof(elapsed_micros));"))
  (println "\tpid_controller<real_t>:\t"
           (cxx "return obj<number>(sizeof(pid_controller<real_t>));")))
(defnative memory-pool-free-space []
  (on "defined FERRET_MEMORY_POOL_SIZE"
      "size_t acc = 0;
       for(size_t i = 0; i < FERRET_MEMORY_POOL_PAGE_COUNT; i++)
         if(memory::allocator::program_memory.used.get(i) == false)
           acc++;
       return obj<number>((acc*sizeof(FERRET_MEMORY_POOL_PAGE_TYPE)));"))
(defn system-exit [^number_t code]
  "::exit(code);")
(defn system-abort [code]
  "::abort();")
(defn byte-stream-encoder [write]
  (fn [bytes]
    (let [size (count bytes)
          CS   (reduce bit-xor size bytes)]
      (write 0X06)
      (write 0X85)
      (write size)
      (doseq [b bytes] 
        (write b))
      (write CS))))
(defn byte-stream-header-ready [read in-waiting]
  (and (>= (in-waiting) 3) (= (read) 0X06) (= (read) 0X85)))

(defn byte-stream-payload-ready [size in-waiting]
  (>= (in-waiting) (inc @size)))

(defn byte-stream-handle-payload [read ^number_t size handler]
  "number_t cs_calculated = size;
   var payload_rev;

   for(number_t i = 0 ; i < size; i++){
     ref v = run(read);
     cs_calculated ^= number::to<number_t>(v);
     payload_rev = runtime::cons(v,payload_rev);
   }

   number_t cs_read = number::to<number_t>(run(read));

   if (cs_calculated == cs_read){
     var payload;

     for_each(i, payload_rev)
       payload = runtime::cons(i,payload);
     
     run(handler,payload);
   }")

(defn byte-stream-decoder [read in-waiting handler]
  (let [size (atom nil)]
    (fsm 
     (states
      (sync-header)
      (read-size        (reset! size (read)))
      (wait-payload)
      (handle-payload   (byte-stream-handle-payload read @size handler)))
     (transitions
      (sync-header     #(byte-stream-header-ready read in-waiting)      read-size)
      (read-size                                                        wait-payload)
      (wait-payload    #(byte-stream-payload-ready size in-waiting)     handle-payload)
      (handle-payload                                                   sync-header)))))
(defobject fsm "ferret/obj/fsm_o.h")

(defn new-fsm [env transitions]
  "return obj<fsm>(env, transitions)")

(defmacro fsm [[_ & states] [_ & transitions]]
  (let [states (->> (reduce (fn [h v]
                              (let [[name & body] v]
                                (conj h name `(fn [] ~@body))))
                            [] states)
                    (apply flatland.ordered.map/ordered-map))
        fsm-state (gensym)
        switch (->> (reduce
                     (fn [h v]
                       (let [[state & conds] v
                             at-state `(= ~state ~fsm-state)
                             jmp (if (= (count conds) 1)
                                   (first conds)
                                   (->> (reduce
                                         (fn [h v]
                                           (let [[check state] v]
                                             (conj h `(~check) state)))
                                         ['cond] (partition 2 conds))
                                        (apply list)))]
                         (conj h at-state jmp)))
                     ['cond] transitions)
                    (apply list))]
    `(new-fsm (list ~@(vals states))
              (fn [[~@(keys states)]
                  ~fsm-state]
                ~switch))))
(defobject pid_controller "ferret/obj/pid_controller_o.h")
(defn new-pid-controller [kp ki kd i-min i-max o-min o-max cont sp]
  "return obj<pid_controller<real_t>>(kp, ki, kd, i_min, i_max, o_min, o_max, cont, sp);")

(defmacro pid-controller [& options]
  (let [defaults {:kp 0 :ki 0 :kd 0 :set-point 0 :bounds [-1 1 -1 1] :continuous false}
        options (merge defaults (apply hash-map options))
        {:keys [container kp ki kd set-point bounds continuous]} options
        [in-min in-max out-min out-max] bounds]
    `(new-pid-controller ~kp ~ki ~kd ~in-min ~in-max ~out-min ~out-max ~continuous ~set-point)))
(defobject moving_average_filter "ferret/obj/moving_average_filter_o.h")

(defn new-moving-average-filter [a]
  "return obj<moving_average_filter<real_t>>(a);")
(defn assert-aux [f msg]
  (when (not (f))
    (println "Assertion Failed =>" msg)
    (system-exit 1)))

(defn assert-aux-callback [f callback]
  (when (not (f)) (callback)))

(defmacro assert
  ([exp]
   `(assert-aux (fn [] ~exp) ~(-> exp pr-str (clojure.string/escape {\\ "\\\\"}))))
  ([exp callback]
   `(assert-aux-callback (fn [] ~exp) (fn [] ~callback))))
(defn is-aux-expect [ex-fb form-fn form-str]
  (let [expect (ex-fb)
        got  (form-fn)]
    (when (not=  expect got)
      (println "fail in" form-str "\n expected" expect "\n      got" got)
      (system-exit 1))))

(defn is-aux [f msg]
  (when (not (f))
    (println "fail" msg)
    (system-exit 1)))

(defmacro is [form]
  (let [check-op (first form)
        form-str (-> form pr-str (clojure.string/escape {\\ "\\\\"}))]

    (cond (= check-op '=)
          (let [[_ expected form] form]
            `(is-aux-expect (fn [] ~expected) (fn [] ~form) ~form-str))
          
          :default `(is-aux (fn [] ~form) ~form-str))))

(defmacro deftest [name & exprs]
  (defonce fir-unit-tests (atom []))
  (swap! fir-unit-tests conj name)
  `(def ~name (fn [] ~@exprs)))

(defmacro run-all-tests []
  (if (bound? #'fir-unit-tests)
    `(do ~@(map #(list %) @fir-unit-tests) (system-exit 0))
    `(do (system-exit 0))))
(defmacro configure-runtime! [& body]
  `(native-define ~(->> (partition 2 body)
                        (map #(str "#define " (first %) " " (second %) "\n"))
                        (list))))
(defmacro configure-ferret! [& body]
  `(native-define ~(str "// build-conf-begin\n"
                        "//" (str (apply hash-map body)) "\n"
                        "// build-conf-end\n")))
