;; -*- coding: utf-8 -*-
;;
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: David Creemer
;;
;; flipboard.base.config
;;
;; read Flipboard low-level static configuration
;;
;; it's just a map, though there are a few helper functions. The public
;; interface are the map fns

(ns flipboard.base.config
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [clojurewerkz.propertied.properties :as props]))

(defn- compute-config-file
  "return a file object of the first config path that exists"
  []
  (let [sysenv (System/getenv)
        paths  [(str (get sysenv "FL_CONFIG" "") "/services.config")
                (str (get sysenv "HOME") "/.flipboard/services.config")
                "/ebsa/config/services.config"]]
    (->> paths
         (map #(io/file %))
         (filter #(.exists %))
         (first))))

(defn- read-config
  "read the services.config file and return a list of [path, data-as-map]"
  []
  (let [config-file (compute-config-file)]
    (if (some? config-file)
      (let [config-path (.getAbsolutePath config-file)]
        (log/info "loading config from:" config-path)
        [config-path (props/properties->map (props/load-from config-file))])
      (do
        (log/error "no config file could be found!")
        [nil {}]))))

(defn create-config-map
  "calculates the static services and environment configuration map"
  []
  (let [sysenv               (into {} (System/getenv))
        os                   (case (System/getProperty "os.name")
                               "Mac OS X" "darwin"
                               "Linux"    "linux"
                               :else      "unknown")
        env                  (get sysenv "FL_ENV" "dev")
        pool                 (get sysenv "FL_POOL" "recluse")
        [config-path config] (read-config)
        dynconfig            (get sysenv "FL_DYNCONFIG" "/ebsa/dynconfig")]
    (into {:fl-env env
           :fl-pool pool
           :fl-config config-path
           :fl-dynconfig dynconfig
           :fl-os os
           :fl-role "flap"
           :fl-subclass "standard"} config)))

;;
;; public interface
;;

(defn dynpath
  "calculate the absolute dynconfig path from the config for the given path fragment"
  [cfg p]
  (let [frag (format "%s/%s" (get cfg :fl-dynconfig) p)]
    (.getAbsolutePath (io/file frag))))

(defn get-int
  "get integer config value from conf for property name.
   returns default (integer) if not found.
   throw NumberFormatException if config value can not be parse to int"
  ([conf name]
   (if-let [val (get conf name)]
     (Integer/parseInt val)))
  ([conf name default]
   (if-let [val (get-int conf name)]
     val default)))

