;; -*- coding: utf-8 -*-
;;
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: David Creemer
;;
;; flipboard.base.instance
;;
;; Flipboard AWS instance interrogation for services

(ns flipboard.base.instance
  (:require [clojure.string :as str]
            [flipboard.base.util :as fu]
            [clojure.tools.logging :as log]
            [amazonica.aws.ec2 :as ec2]
            [amazonica.aws.simpledb :as sdb]))

(defn- calc-instance-number
  "calculate the instance number in the group as an integer (e.g. flyf01-crawler.beta ->
  1). Returns 0 if unknown."
  [i]
  (let [s (-> (i :name)
              (fu/chop-prefix (i :role))
              (fu/chop-suffix (str "." (i :pool)))
              (fu/chop-suffix (str "-" (i :subclass))))]
    (try
      (Integer. s)
      (catch NumberFormatException _ 0))))

(defn- sdb-query-one
  "run an SDB query and return the first result"
  [cred qry]
  (let [rows (sdb/select cred :select-expression qry)]
    (first (:items rows))))

(defn lookup-instance
  "lookup the first host instance in SDB satisfying query k = v in domain dom"
  [cfg k v]
  (let [cred   {:access-key (get cfg "aws.accessKey")
                :secret-key (get cfg "aws.secretKey")}
        domain (format "flipboard.%s.instances" (get cfg :fl-env))
        rec    (sdb-query-one cred (format "select * from `%s` where `%s`=\"%s\"" domain k v))]
    (if (some? rec)
      (let [attrs (:attributes rec)
            host (zipmap (map #(keyword (:name %)) attrs) (map :value attrs))]
        (into host {:number (calc-instance-number host), :instance (:name rec)}))
      {})))

(defn my-instance
  "return a host map describing the instance matching my IP address"
  [cfg]
  (try
    (let [my-ip (.getHostAddress (java.net.InetAddress/getLocalHost))]
      (lookup-instance cfg "private" my-ip))
    (catch java.net.UnknownHostException e "unknown")))

(defn graphite-name
  "calculate the graphite metrics name for a given instance map"
  [i]
  (let [num (get i :number 0)]
    (format "%s.%s.%s.%02d" (:pool i "developer") (:role i "test") (:subclass i "standard") num)))
