;; -*- coding: utf-8 -*-
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: Howard Zhao
;; created: 1/5/15
;; flipboard.component.graphite
;; 
;; Purpose: report metrix to graphite.
;; Just include this component in system map,
;; the metrics will be collected and sent to service.config graphite.tcp.host
;;

(ns flipboard.component.graphite
  (:require [com.stuartsierra.component :as component]
            [metrics.reporters.graphite :as mg]
            [flipboard.base.instance :refer [graphite-name]]
            [clojure.tools.logging :as log]
            [clojure.string :as str]
            [flipboard.base.util :as util])
  (:import (com.codahale.metrics.graphite Graphite)))

(defn send-raw-metrics
  "send metrics directly to graphite server as configured
   metrics key: e.g. flock.task.upcoming-5days
   val-timestamp-list: list of map {:val val :timestamp timestamp}
    timestamp in epoch"
  [graphite-comp metrics-key val-timestamp-list]
  (let [{host :host port :port prefix :prefix}
        (:config graphite-comp)
        g (Graphite. host port)
        fullkey (str prefix "." metrics-key)]
    (log/debug "send metrix" fullkey "count:" (count val-timestamp-list))
    (.connect g)
    (doseq [{val :val timestamp :timestamp} val-timestamp-list]
      (.send g fullkey (str val) timestamp))
    (.close g)))

(defrecord GraphiteReporter [core]
  component/Lifecycle

  (start [this]
    (let [graphite_cfg (get-in this
                               [:core :config "graphite.tcp.host"]
                               "localhost:2003")
          [host port] (str/split graphite_cfg #"\:")
          port (util/to-int port)
          inst (get-in this [:core :instance])
          prefix (graphite-name inst)
          opts {:host host :port port :prefix prefix}
          reporter (mg/reporter opts)]
      (mg/start reporter 20)
      (log/info "Started graphite reporter on " opts)
      (assoc this :reporter reporter :config opts)))

  (stop [this]
    (log/info "Stopping graphite reporter")
    (mg/stop (:reporter this))))

(defn new-graphite-reporter
  []
  (map->GraphiteReporter {}))

