;; -*- coding: utf-8 -*-
;; (c)2014 Flipboard Inc, All Rights Reserved.
;; Author: Howard Zhao
;; created: 3/2/16
;; flipboard.component.hbase
;; 
;; Purpose: component for hbase connection with convenient methods get, put, delete, scan
;;

(ns flipboard.component.hbase
  (:require [com.stuartsierra.component :as component]
            [clojure.tools.logging :as log]
            [clojure-hbase.core :as hb]
            [clojure-hbase.util :as hu])
  (:import (org.apache.hadoop.hbase.client Result)
           (org.apache.hadoop.hbase KeyValue)))

(defn as-vecmap
  "Extracts the contents of the Result object and sticks them into a
   vector of map {:family family (keyword) :qualifier qualifier(string) :timestamp timestamp(int) :value value(string)};
   returns a sequence of such vectors maintaining the order in result."
  [#^Result result]
  (loop [remaining-kvs (seq (.raw result))
         kv-vec (transient [])]
    (if-let [^KeyValue kv (first remaining-kvs)]
      (let [family    (hu/as-kw (.getFamily kv))
            qualifier (hu/as-str (.getQualifier kv))
            timestamp (.getTimestamp kv)
            value     (hu/as-str (.getValue kv))
            m {:family family :qualifier qualifier :value value :timestamp timestamp}]
        (recur (next remaining-kvs)
               (conj! kv-vec m)))
      (persistent! kv-vec))))

(defn get-row
  "get from the table-name from the connection associated with hbase component with options:
    :column [:family-name :qualifier]
    :columns [:family-name [:qual1 :qual2...]...]
    :family :family-name
    :families [:family1 :family2 ...]
    :filter <a filter you've made>
    :all-versions
    :max-versions <int>
    :time-range [start end]
    :time-stamp time
    Returns vector of maps preserving the ordering in the result"
  [comp table-name row & opts]
  (hb/with-table [tbl (hb/table-conn (:conn comp) table-name)]
                 (-> (apply hb/get tbl row opts)
                     (as-vecmap))))

(defrecord HBaseComponent [zk-peers core]
  component/Lifecycle

  (start [this]
    (let [config (get core :config)
          zk (get config zk-peers)]
      (log/info "hbase client " zk-peers " connected to " zk)
      (->> {:hbase.zookeeper.quorum zk}
           (hb/new-connection)
           (assoc this :zk-peers zk-peers :conn))))

  (stop [this]
        (log/info "stopped HBaseComponent" (:zk-peers this))
    (dissoc this :zk-peer :conn)))

(defn new-hbase
  "create a hbase connection component for a given zk-peers config name in services.config.
  e.g. flzk-cl2d-peers for cl2d hbase"
  [zk-peers]
  (log/info "Creating new hbase " zk-peers)
  (map->HBaseComponent {:zk-peers zk-peers}))
