;; -*- coding: utf-8 -*-
;; (c)2016 Flipboard Inc, All Rights Reserved.
;; flipboard.component.cache
;;
;; Purpose: local cache component using core.cache
;;

(ns flipboard.component.ttlcache
  (:require [clojure.core.cache :as cache]
            [com.stuartsierra.component :as component]))

(defn get-val
  "get a value with key from cache"
  [comp key]
  (-> (:cache comp)
      (deref)
      (cache/lookup key)))

(defn delete-val
  "delete a key from cache"
  [comp key]
  (let [c (:cache comp)]
    (reset! c (cache/evict (deref c) key))))

(defn set-val
  "set value for key in cache.
  returns val for clojure threading convenience"
  [comp key val]
  (let [c (:cache comp)]
    (reset! c (cache/miss (deref c) key val)))
  val)

; component for local ttl cache
(defrecord TTLCache [ttl]
  component/Lifecycle

  (start [this]
    (->> (cache/ttl-cache-factory {} :ttl ttl)
         (atom)
         (assoc this :cache)))

  (stop [this]
    (dissoc this :cache)))

(defn new-ttl-cache
  "create a new ttl cache component with specified ttl"
  [ttl]
  (map->TTLCache {:ttl ttl}))
