(ns sock.task.deps
  "Sock Book tasks for dependency resolution."
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.lib.core :refer [apply-kw]]
    [boot-tools-deps.core :refer [load-deps]]))

(core/deftask deps
  "Use tools.deps to read and resolve the specified deps.edn files.

  The dependencies read in are added to your Boot :dependencies vector.

  With the exception of -A, and -v, the arguments are intended to match
  the clj script usage (as passed to clojure.tools.deps.alpha.makecp/-main).
  Note, in particular, that -c / --config-paths is assumed to be the COMPLETE
  list of EDN files to read (and therefore overrides the default set of
  system deps and local deps)."
  [c config-paths    PATH [str] "the list of deps.edn files to read"
   A aliases           KW [kw]  "the list of aliases (for both -C and -R)"
   C classpath-aliases KW [kw]  "the list of classpath aliases to use"
   R resolve-aliases   KW [kw]  "the list of resolve aliases to use"
   v verbose              bool  "Be verbose (and ask tools.deps to be verbose too)"]
  (let [opts {:config-paths        config-paths
              :classpath-aliases   (into (vec aliases) classpath-aliases)
              :resolve-aliases     (into (vec aliases) resolve-aliases)
              :overwrite-boot-deps true
              :verbose             verbose}]
    (load-deps opts)
    identity))

(core/deftask test-deps
  "Configure the boot environment with typical defaults for testing."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn  "Options for seancorfield/boot-tools-deps task."]
  (let [aliases     (or aliases [:test])
        tools-deps' (merge {:aliases aliases} tools-deps)]
       (apply-kw deps tools-deps')
       identity))

(core/deftask integration-deps
  "Configure the boot environment with typical defaults for integration testing."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn  "Options for seancorfield/boot-tools-deps task."]
  (let [aliases     (or aliases [:integration-test])
        tools-deps' (merge {:aliases aliases} tools-deps)]
       (apply-kw deps tools-deps')
       identity))

(core/deftask property-deps
  "Configure the boot environment with typical defaults for property-based ('generative') testing."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn  "Options for seancorfield/boot-tools-deps task."]
  (let [aliases     (or aliases [:property-test])
        tools-deps' (merge {:aliases aliases} tools-deps)]
       (apply-kw deps tools-deps')
       identity))

(core/deftask conflicts
  "Print graph of dependency conflicts."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."]
  (comp
    (apply-kw deps (merge {:aliases aliases} tools-deps))
    (built-in/show :pedantic true)))

(core/deftask fake-classpath
  "Print the project's fake classpath."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."]
  (comp
    (apply-kw deps (merge {:aliases aliases} tools-deps))
    (built-in/show :fake-classpath true)))

(core/deftask classpath
  "Print the project's classpath."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."]
  (comp
    (apply-kw deps (merge {:aliases aliases} tools-deps))
    (built-in/show :classpath true)))

(core/deftask graph
  "Print the project's dependency graph."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   v verify-deps        bool "Include signature status of each dependency in graph."]
  (comp
    (apply-kw deps (merge {:aliases aliases} tools-deps))
    (built-in/show :deps true :verify-deps verify-deps)))

(core/deftask boot-env
  "Print the project's boot env map."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."]
  (comp
    (apply-kw deps (merge {:aliases aliases} tools-deps))
    (built-in/show :env true)))

(core/deftask updates
  "Print newer releases of outdated dependencies."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   S snapshots          bool "Include snapshot versions in update searches."]
  (comp
    (apply-kw deps (merge {:aliases aliases} tools-deps))
    (built-in/show :updates (not snapshots) :update-snapshots snapshots)))

