(ns sock.task.clj
  "Sock Boot tasks for Clojure projects."
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.init]
    [sock.lib.core :refer [apply-kw]]
    [sock.task.deps :refer [deps test-deps]])
  (:refer-clojure :exclude [test]))

(core/deftask environ
  "If 'environ is a dependency of the current project then require and run the
   boot-environ task, otherwise do nothing."
  [e env        FOO=BAR {kw str} "Environ environment map."]
  (let [environ? (some #(= (first %) 'environ) (:dependencies (core/get-env)))]
    (if environ?
      (do
        (core/set-env! :dependencies #(conj % '[boot-environ "1.1.0" :scope "test"]))
        (require 'environ.boot)
        (let [boot-environ (resolve 'environ.boot/environ)]
          (boot-environ :env env)))
      identity)))

(core/deftask midje
  "If 'midje is a dependency of the current project then require and run the
   zhuangxm/boot-midje Midje test runner, otherwise do nothing."
  []
  (let [midje? (some #(= (first %) 'midje) (:dependencies (core/get-env)))]
    (if midje?
      (do
        (core/set-env! :dependencies #(conj % '[zhuangxm/boot-midje "0.1.2" :scope "test"]))
        (require 'zhuangxm.boot-midje)
        (let [boot-midje (resolve 'zhuangxm.boot-midje/midje)]
          (boot-midje)))
      identity)))

(core/deftask clojure-test
  "Require and run the clojure.test test runner."
  [_ opts OPTIONS edn "Options for adzerk/boot-test task."]
  (core/set-env! :dependencies #(conj % '[adzerk/boot-test "1.2.0" :scope "test"]))
  (require 'adzerk.boot-test)
  (let [boot-test (resolve 'adzerk.boot-test/test)]
    (apply-kw boot-test opts)))

(core/deftask bat-test
  "Require and run the alternate clojure.test test runner."
  [_ opts OPTIONS edn "Options for metosin/bat-test task."]
  (core/set-env! :dependencies #(conj % '[metosin/bat-test "0.4.0" :scope "test"]))
  (require '[metosin.bat-test])
  (let [bat-test (resolve 'metosin.bat-test/bat-test)]
    (apply-kw bat-test opts)))

(core/deftask check []
  ;; ref: https://github.com/tolitius/boot-check
  (core/set-env! :dependencies #(conj % '[tolitius/boot-check "0.1.11" :scope "test"]))
  (require '[tolitius.boot-check :as check])
  (let [with-yagni (resolve 'tolitius.boot-check/with-yagni)
        with-eastwood (resolve 'tolitius.boot-check/with-eastwood)
        with-kibit (resolve 'tolitius.boot-check/with-kibit)
        with-bikeshed (resolve 'tolitius.boot-check/with-bikeshed)]
    (comp
     (with-yagni)
     (with-eastwood)
     (with-kibit)
     (with-bikeshed))))

(core/deftask env-spec
  "[env] `spec`able"
  []
  (core/set-env! :dependencies #(conj % '[orchestra "2018.09.10-1" :scope "test"]))
  (core/with-pass-thru _

    (require '[orchestra.spec.test :refer [instrument]])
    ((resolve 'orchestra.spec.test/instrument))
    (require '[clojure.spec [alpha :as s]])
    ((resolve 'clojure.spec.alpha/check-asserts) true)))

(core/deftask build
  "Resolve dependencies, create project pom.xml file and build a jar file."
  [_ tools-deps OPTIONS edn      "Options for seancorfield/boot-tools-deps task."
   e env        FOO=BAR {kw str} "Environ environment map."
   _ pom        OPTIONS edn      "Options for built-in/pom task."]
  (comp (apply-kw deps tools-deps)
        (environ :env env)
        (apply-kw built-in/pom pom)))

(core/deftask fmt
  "Auto-format Clojure code."
  []
  (core/set-env! :dependencies #(conj % '[boot-fmt/boot-fmt "0.1.8" :scope "test"]))
  (require '[boot-fmt.core])
  (let [boot-fmt (resolve 'boot-fmt.core/fmt)]
    (boot-fmt
      :mode :overwrite
      :really true
      :source true
      :options {:style :community
                :binding {:justify? true}
                :map {:justify? true
                      :comma? false}})))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)

(core/deftask test
  "Resolve dependencies including :test alias and run unit tests."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   _ tools-deps OPTIONS edn "seancorfield/boot-tools-deps task."
   e env        FOO=BAR {kw str} "Environ environment map."]
  (let [aliases' (or aliases [:test])]
    (comp (test-deps
            :aliases aliases'
            :tools-deps tools-deps)
          (environ :env env)
          (midje)
          (env-spec)
          (bat-test))))

(core/deftask integration
  "Resolve dependencies including :integration-test alias and run integration tests."
  []
  (test :aliases [:integration]))

(core/deftask quickcheck
  "Resolve dependencies including :property-test alias and run property-based ('generative') tests."
  []
  (test :aliases [:quickcheck]))

(core/deftask local
  "Build then install project jar file to local Maven repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ install    OPTIONS edn "Options for built-in/install task."]
  (comp (build :tools-deps tools-deps
               :pom pom)
        (apply-kw built-in/jar jar)
        (apply-kw built-in/install install)))

(core/deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-snapshots"} push)]
    (comp (build :tools-deps tools-deps
                 :pom pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask release
  "Build then deploy project jar file to company releases repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-releases"} push)]
    (comp (build :tools-deps tools-deps
                 :pom pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask assemble
  "Build, add jar entries from dependencies to fileset and writes output files
   to the target directory. Mainly used for server projects which subsequently
   copy the target directory into a Docker image."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ uber       OPTIONS edn "Options for built-in/uber task."
   _ target     OPTIONS edn "Options for built-in/target task."]
  (comp (build :tools-deps tools-deps
               :pom        pom)
        (apply-kw built-in/uber uber)
        (apply-kw built-in/target target)))
