(ns sock.lib.deps-edn
  "Sock Boot functions for transformation of deps.edn to boot dependencies."
  (:require
    [boot.core :as core]
    [boot.util :as util]
    [sock.lib.io :as io]))

(defn obtain
  "Returns a map that is the result of reading and parsing deps.edn."
  []
  (io/slurp-edn (io/file "deps.edn")))

(defn ->dependencies
  "Returns a boot-compatible representation of dependencies from deps.edn."
  [deps-edn aliases]
  (->>
    ;; First merge all aliases deps into the main deps.
    (reduce
      (fn [ret k]
        (merge ret (get-in deps-edn [:aliases k :extra-deps])))
      (:deps deps-edn)
      aliases)
    ;; Second convert deps.edn data structure to boot.
    (reduce
      (fn [ret [sym {:mvn/keys [version] :keys [exclusions]}]]
        (conj! ret (cond-> [sym version]
                           exclusions
                           (into [:exclusions exclusions]))))
      (transient []))
    (persistent!)))

(defn ->paths
  "Returns a boot-compatible representation of paths from deps.edn."
  [deps-edn aliases]
  (->>
    (reduce
      (fn [ret k]
        (into ret (get-in deps-edn [:aliases k :extra-paths])))
      (:paths deps-edn)
      aliases)
    (into #{})))

(defn ->repositories
  "Returns a boot-compatible representation of repositories from deps.edn."
  [deps-edn]
  (let [[username password] (map #(core/get-sys-env % :required) ["AWS_ACCESS_KEY_ID" "AWS_SECRET_ACCESS_KEY"])]
    (reduce-kv #(conj %1 [%2 (merge
                               {:username username
                                :password password}
                               %3)]) [] (:mvn/repos deps-edn))))

(defn repositories!
  "Merge repositories from deps.edn into the boot environment."
  [deps-edn]
  (let [wagons '[[s3-wagon-private "1.2.0"]]
        repositories (->repositories deps-edn)]
    (util/dbug* "Adding these wagons: %s\n" (util/pp-str wagons))
    (util/dbug* "Adding these repositories: %s\n" (util/pp-str repositories))
    (core/merge-env! :wagons wagons)
    (core/merge-env! :repositories repositories)))


(defn dependencies!
  "Replace boot environment dependencies with those from deps.edn."
  [deps-edn aliases]
  (let [dependencies (->dependencies deps-edn aliases)]
    (util/dbug* "Setting these dependencies: %s\n" (util/pp-str dependencies))
    (core/set-env!
      ;; Always globally exclude Clojure as we should always explicitly specify
      ;; our Clojure version in each project, not as a transitive dependency!
      ;; This also makes builds a bit faster as every incorrectly configured
      ;; transitive dependency (which is most of them...) will not download the
      ;; Clojure release metadata.
      :exclusions '[org.clojure/clojure]
      :dependencies dependencies)))

(defn paths!
  "Replace boot environment paths with those from deps.edn."
  [deps-edn aliases]
  (let [paths (->paths deps-edn aliases)]
    (util/dbug* "Setting these paths: %s\n" (util/pp-str paths))
    (core/set-env! :resource-paths paths)
    (core/set-env! :source-paths paths)))
