(ns sock.task.deps
  "Sock Book tasks for dependency resolution."
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.lib.core :refer [apply-kw]]
    [sock.lib.deps-edn :as deps-edn]
    [sock.lib.project-edn :as project-edn]))

(core/deftask deps
  "Resolve dependencies according to the ::aliases environment merged with the aliases task option."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [deps-edn (deps-edn/obtain)
        project-edn (project-edn/obtain)
        aliases (into (or aliases []) (or (core/get-env ::aliases) []))]
    (deps-edn/repositories! deps-edn)
    (deps-edn/paths! deps-edn aliases)
    (deps-edn/dependencies! deps-edn aliases)
    (project-edn/pom-options! project-edn)
    identity))

(core/deftask dev
  "Resolve dependencies including :common and :dev aliases for local development builds."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [aliases     (or aliases [:common :dev])]
    (core/set-env! ::aliases #(into (or % []) aliases))
    identity))

(core/deftask ci
  "Resolve dependencies including :common and :ci aliases for continuous integration builds."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [aliases     (or aliases [:common :ci])]
    (core/set-env! ::aliases #(into (or % []) aliases))
    identity))

(core/deftask production
  "Resolve dependencies including :common and :production aliases for production builds."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [aliases     (or aliases [:common :production])]
    (core/set-env! ::aliases #(into (or % []) aliases))
    identity))

(core/deftask unit
  "Resolve dependencies including :unit alias for running unit tests."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [aliases (or aliases [:unit])]
    (core/set-env! ::aliases #(into (or % []) aliases))
    identity))

(core/deftask integration
  "Resolve dependencies including :integration alias for running integration tests."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [aliases (or aliases [:integration])]
    (core/set-env! ::aliases #(into (or % []) aliases))
    identity))

(core/deftask quickcheck
  "Resolve dependencies including :quickcheck alias for running property-based ('generative') tests."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [aliases (or aliases [:quickcheck])]
    (core/set-env! ::aliases #(into (or % []) aliases))
    identity))

;; TODO(@isaac) obsolete, but currently used by golden-island build.boot
(core/deftask test-deps
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (let [aliases     (or aliases [:test])]
    (deps :aliases aliases)
    identity))

(core/deftask conflicts
  "Print graph of dependency conflicts."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (comp
    (deps :aliases aliases)
    (built-in/show :pedantic true)))

(core/deftask fake-classpath
  "Print the project's fake classpath."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (comp
    (deps :aliases aliases)
    (built-in/show :fake-classpath true)))

(core/deftask classpath
  "Print the project's classpath."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (comp
    (deps :aliases aliases)
    (built-in/show :classpath true)))

(core/deftask graph
  "Print the project's dependency graph."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   v verify-deps        bool "Include signature status of each dependency in graph."]
  (comp
    (deps :aliases aliases)
    (built-in/show :deps true :verify-deps verify-deps)))

(core/deftask boot-env
  "Print the project's boot env map."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."]
  (comp
    (deps :aliases aliases)
    (built-in/show :env true)))

(core/deftask updates
  "Print newer releases of outdated dependencies."
  [A aliases    KW      [kw] "Aliases used for classpath construction and dependency resolution."
   S snapshots          bool "Include snapshot versions in update searches."]
  (comp
    (deps :aliases aliases)
    (built-in/show :updates (not snapshots) :update-snapshots snapshots)))

