(ns sock.task.porcelain.java
  "Sock porcelain boot tasks for Java"
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.init]
    [sock.lib.core :refer [apply-kw]]
    [sock.task.plumbing.deps :as plumbing.deps]
    [sock.task.plumbing.java :as plumbing.java])
  (:refer-clojure :exclude [test]))

(core/deftask build
  "Resolve dependencies, compile Java sources, create project pom.xml file and
   build a jar file."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."]
  (comp (apply-kw plumbing.deps/deps tools-deps)
        (apply-kw built-in/javac javac)
        (apply-kw built-in/pom pom)
        (apply-kw built-in/jar jar)))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)

(core/deftask test
  "Resolve dependencies including :test alias, compile Java sources and run
   jUnit tests."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ junit      OPTIONS edn "Options for radicalzephyr/boot-junit task."]
  (comp (plumbing.deps/test-deps :tools-deps tools-deps)
        (apply-kw built-in/javac javac)
        (plumbing.java/junit :junit junit)))

(core/deftask local
  "Build then install project jar file to local Maven repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ install    OPTIONS edn "Options for built-in/install task."]
  (comp (build :tools-deps tools-deps
               :javac      javac
               :pom        pom
               :jar        jar)
        (apply-kw built-in/install install)))

(core/deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-snapshots"} push)]
    (comp (build :tools-deps tools-deps
                 :javac      javac
                 :pom        pom
                 :jar        jar)
          (apply-kw built-in/push push'))))

(core/deftask release
  "Build then deploy project jar file to company releases repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-releases"} push)]
    (comp (build :tools-deps tools-deps
                 :javac      javac
                 :pom        pom
                 :jar        jar)
          (apply-kw built-in/push push'))))

