(ns sock.task.porcelain.java
  "Sock porcelain boot tasks for Java"
  {:boot/export-tasks true}
  (:require
    [boot.core :as core]
    [boot.task.built-in :as built-in]
    [sock.init]
    [sock.lib.core :refer [apply-kw]]
    [sock.task.deps :refer [deps test-deps]])
  (:refer-clojure :exclude [test]))

(core/deftask junit
  "Require and run the jUnit test runner."
  [_ junit OPTIONS edn "Options for radicalzephyr/boot-junit task."]
  (core/set-env! :dependencies #(conj % '[radicalzephyr/boot-junit "0.3.0" :scope "test"]
                                      '[junit/junit              "4.12"  :scope "test"]))
  (require 'radicalzephyr.boot-junit)
  (let [boot-junit (resolve 'radicalzephyr.boot-junit/junit)]
    (apply-kw boot-junit junit)))

(core/deftask build
  "Resolve dependencies, compile Java sources, create project pom.xml file and
   build a jar file."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."]
  (comp (apply-kw deps tools-deps)
        (apply-kw built-in/javac javac)
        (apply-kw built-in/pom pom)))

;;; This prevents a name collision WARNING between the test task and
;;; clojure.core/test, a function that nobody really uses or cares
;;; about.
(ns-unmap 'boot.user 'test)

(core/deftask test
  "Resolve dependencies including :test alias, compile Java sources and run
   jUnit tests."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ junit      OPTIONS edn "Options for radicalzephyr/boot-junit task."]
  (comp (test-deps :tools-deps tools-deps)
        (apply-kw built-in/javac javac)
        (plumbing.java/junit :junit junit)))

(core/deftask local
  "Build then install project jar file to local Maven repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ install    OPTIONS edn "Options for built-in/install task."]
  (comp (build :tools-deps tools-deps
               :javac      javac
               :pom        pom)
        (apply-kw built-in/jar jar)
        (apply-kw built-in/install install)))

(core/deftask snapshot
  "Build then deploy project jar file to company snapshots repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-snapshots"} push)]
    (comp (build :tools-deps tools-deps
                 :javac      javac
                 :pom        pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask release
  "Build then deploy project jar file to company releases repository."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ jar        OPTIONS edn "Options for built-in/jar task."
   _ push       OPTIONS edn "Options for built-in/push task."]
  (let [push' (merge {:repo "company-releases"} push)]
    (comp (build :tools-deps tools-deps
                 :javac      javac
                 :pom        pom)
          (apply-kw built-in/jar jar)
          (apply-kw built-in/push push'))))

(core/deftask assemble
  "Build, add jar entries from dependencies to fileset and writes output files
   to the target directory. Mainly used for server projects which subsequently
   copy the target directory into a Docker image."
  [_ tools-deps OPTIONS edn "Options for seancorfield/boot-tools-deps task."
   _ javac      OPTIONS edn "Options for built-in/javac task."
   _ pom        OPTIONS edn "Options for built-in/pom task."
   _ uber       OPTIONS edn "Options for built-in/uber task."
   _ target     OPTIONS edn "Options for built-in/target task."]
  (comp (build :tools-deps tools-deps
               :javac      javac
               :pom        pom)
        (apply-kw built-in/uber uber)
        (apply-kw built-in/target target)))
