(ns reinen.vernunft.prolog
  (require [clojure.core.unify :as core]
           [clojure.walk :refer (postwalk-replace walk)]))

(defn wrap [x] (if (coll? x) x [x])) 

(defn fail [msg & {:as m}] (throw (ex-info msg m)))

(defn fact
  [db clause]
  (when-not (map? db) (fail (str "Expected a map-like for the factbase, got: " db)
                            :db db :clause clause))
  (let [clause (map wrap clause)
        [[head & _] & _] clause]
    (update-in db [head] conj clause)))

(defn garner-lvars
  [clauses]
  (set (mapcat core/extract-lvars clauses)))

(defn  ^:private fresh [clause]
  (let [var-map (reduce merge
                        (map (fn [var]
                               {var (gensym var)})
                             (garner-lvars clause)))]
    (postwalk-replace var-map clause)))

;; prove

(declare prove)

(defn ^:private possibilities
  [db [functor & _ :as goal] bindings]
  (mapcat (fn [clause]
            (let [[head & body :as clause] (fresh clause)]
              (prove db body (core/unify goal head bindings))))
          (remove empty? (get db functor))))

(defn prove
  [db goals bindings]
  (cond (nil? bindings) nil
        (empty? goals) (list bindings)
        :else (mapcat (fn [possibility]
                        (prove db (rest goals) possibility))
                      (remove empty? (possibilities db (first goals) bindings)))))

(defn smooth
  [vars solutions]
  (cond (every? empty? solutions) :no
        (and (empty? vars) (seq solutions)) :yes
        :default (map #(select-keys % vars)
                      (map core/flatten-bindings solutions))))

(defmacro <- [db & clause]
  `(fact ~db '~clause))

(defmacro ?- [db & goals]
  `(if (map? ~db)
     (smooth (garner-lvars '~goals)
             (prove ~db '~goals {}))
     (fail (str "Expected a map-like for the factbase in a query, got: " ~db)
           :db ~db :goals '~goals)))

(comment
  (-> {}
      ;; rules
      (<- (planet ?p) (orbits ?p Sun))
      (<- (satellite ?s) (orbits ?s ?p))
      (<- (planet ?p))

      (<- (orbits Mercury Sun))
      (<- (orbits Venus Sun))
      (<- (orbits Earth Sun))
      (<- (orbits Mars Sun))
      (<- (orbits Moon Earth))
      (<- (orbits Phobos Mars))
      (<- (orbits Deimos Mars))
      (<- (orbits Jupiter Sun))
      (<- (orbits Saturn Sun))
      (<- (orbits Uranus Sun))
      (<- (orbits Neptune Sun))

      (?- (orbits ?p Sun))
      ))