(ns fogus.rv.production-rules
  (:require [fogus.rv.matching :as match]
            clojure.set))

;; This namespace is currently used only for the purposes of
;; demonstration during some internal work presentations.
;; The rules engine machinery is from https://github.com/madnl/toy-rule-engine/

(declare solve)

;; # Resolution

(defmulti ^:private resolve-pattern
  "Multi-method to handle various types of patterns. It dispatches
  on the first symbol in the pattern"
  (fn [pat _ _] (first pat)))

(defmethod resolve-pattern 'if [pat _ b]
  (let [expr (second pat)
        simple-expr (match/subst expr b)
        truth (eval simple-expr)]
    (if truth (list b) '())))

(defmethod resolve-pattern 'not [pat kb b]
  (let [negated-patterns (rest pat)
        solutions (solve negated-patterns kb b)]
    (if (empty? solutions)
      (list b)
      '())))

(defmethod resolve-pattern :default [pat kb b]
  (keep #(match/unify pat % b) kb))

;; # Solving

(defn ^:private solve [patterns kb b]
  "Takes a sequence of patterns, a kb and a variable binding
  and tries to match each pattern sequentially, adding new
  variables in the binding as it iterates. It returns
  a sequence of enriched bindings corresponding to the valid
  matches"
  (if (empty? patterns)
    (list b)
    (let [pat (first patterns)
          bs (resolve-pattern pat kb b)]
      (mapcat #(solve (rest patterns) kb %) bs))))

;; # Bindings

(defn ^:private unifications [rule kb]
  "Given a rule and a kb of facts, return all possible bindings
  for the variables in the rule, that make the rule patterns
  match the facts of the kb"
  (solve (:patterns rule) kb {}))

;; # Resolution strategy

(def ^:private selection-strategy
  "Function which takes a range of possible applicable rules
  and returns one of them"
  rand-nth)

(defn ^:private select-rule [rules kb]
  "If there are applicable rules with different variable bindings,
  select one of those rules with one binding. If no rules are applicable,
  return nil"
  (let [possibilities (for [r rules
                            b (unifications r kb)]
                        [r b])]
    (selection-strategy possibilities)))

;; # The guts of the production rules system

(defn ^:private apply-rule [rule binds kb]
  "Apply a rule on a kb, given a set of variable bindings
  for that rule"
  (let [new-facts (for [p (:assertions rule)]
                    (match/subst p binds))]
    (clojure.set/union new-facts kb)))

(defn ^:private step [kb rules]
  "Run a single step in the rule engine algorithm. Pick
  an applicable rule and run it on the database. If no rules
  are applicable, return nil"
  (when-let [[rule binds] (select-rule rules kb)]
    (apply-rule rule binds kb)))

(defn ^:private states [domain]
  "Takes a domain and returns the lazy stream
  of states that occur when running iteratively
  rules in the domain"
  (let [initial-kb (set (:facts domain))]
    (iterate #(step % (:rules domain)) initial-kb)))


;; # Pulling it all together

(defn cycle [quiesce domain]
  "Iterate through the domain until no more
  rules are applicable or max-steps was reached"
  (->> (states domain)
       (take-while (complement nil?))
       quiesce))

(defn naive-quiesce [states]
  (last (take 256 states)))

