(ns fogus.thneed.threads
  (:refer-clojure :exclude [aget aset count seq])
  (:import [java.util.concurrent.locks ReentrantLock]))

(defprotocol SafeArray ;; #: SafeArray features a small set of functions
  (aset  [this i f])
  (aget  [this i])
  (count [this])
  (seq   [this]))

(defn make-safe-array [t sz]
  (let [a (make-array t sz)] ;; #: Array creation is the same
    (reify
      SafeArray
      (count [_] (clojure.core/count a))
      (seq [_] (clojure.core/seq a))
      (aget [_ i] ;; #: aget is locked
        (locking a
          (clojure.core/aget a i)))
      (aset [this i f] ;; #: aset is locked
        (locking a
          (clojure.core/aset a i (f (aget this i)))))))) ;; #: aset uses aget


(defn- lock-i [target-index num-locks]
  (mod target-index num-locks))

(defn make-smart-array [t sz]
  (let [a   (make-array t sz) ;; #: The array
        Lsz (/ sz 2)
        L   (into-array (take Lsz ;; #: The locks
                              (repeatedly #(ReentrantLock.))))]
    (reify
      SafeArray
      (count [_] (clojure.core/count a))
      (seq [_] (clojure.core/seq a))
      (aget [_ i]
        (let [lk (clojure.core/aget L (lock-i (inc i) Lsz))]
          (.lock lk) ;; #: Explicit locking
          (try
            (clojure.core/aget a i)
            (finally (.unlock lk))))) ;; #: Explicit unlocking
      (aset [this i f]
        (let [lk (clojure.core/aget L (lock-i (inc i) Lsz))]
          (.lock lk)
          (try
            (clojure.core/aset a i (f (aget this i))) ;; #: Reentrant locking
            (finally (.unlock lk))))))))

