(ns trout.collect
  (:require [trout.util :as util] 
            [clojure.java.io :as io]
            [clojure.string :as string]
            [fs.core :as fs]
            [fogus.thneed.lein :as proj]
            clojure.pprint
            trout.generator.markdown
            trout.generator.html
            trout.generator.docbook))

(def +BASE+ (or (System/getenv "trout-base") "."))
(def +TEXT+ "txt")
(def +TARG+ "tmp")

(defn gather-trout-parts
  [{trout-root :root}]
  (let [base (util/path trout-root +TEXT+)]
    (println "Processing " base)
    (println "and " (fs/list-dir base))
    (for [f (fs/list-dir base)]
         (let [section (util/path base f)] 
           (for [txt (sort (fs/list-dir section))]
             {:path (.replace section trout-root "")
              :file txt
              :root trout-root})))))

(defn chapter-order
  [s]
  (let [n (->> s first :path (re-find #"/ch(.*)") second)]
    (cond
     (not n) nil               ; use nil here to put frontmatter first
     (re-find #"^\d*$" n) (Integer/parseInt n)
     :else (int (first n)))))

(defn build-metadata [root]
  (let [project-file (util/path root "project.clj")
        defaults {:title "Utitled work"
                  :lang  "en"
                  :authors [{:first-name "Anonymous" :last-name "Anonymous"}]}]
    (assoc
        (merge defaults
               (try
                 (:trout (proj/parse-project-file project-file))
                 (catch Exception _
                   (println "WARNING: Unable to find " project-file " -- using defaults instead.")
                   {})))
      :root root)))

(defn trout-tree
  ([]
     (trout-tree +BASE+))
  ([root]
     (trout-tree chapter-order root (build-metadata root)))
  ([sorter root metadata]
     (with-meta
       (seq (sort-by sorter (gather-trout-parts metadata)))
       metadata)))

(defn each-section
  [bk fun & args]
  (with-meta
    (for [chap bk]
      (for [sect chap] 
        (apply fun sect args)))
    (meta bk)))

(defn each-chapter
  [bk fun & args]
  (with-meta
    (for [chap bk]
      (apply fun chap args))
    (meta bk)))

(defn raw-text
  [section]
  (assoc section
    :format/raw
    (slurp (util/path (:root section) (:path section) (:file section)))))

(defn done [s]
  (dorun (tree-seq seq? seq s))
  nil)

(defn markdown->html
  [tree]
  (-> tree
      (each-section raw-text)
      (each-section trout.generator.markdown/process-markdown)
      trout.generator.html/prepare-html!
      (each-section trout.generator.html/save-html! #"\.md$")))

(defn html->docbook
  [tree]
  (-> tree
      (each-section trout.generator.docbook/take-html)
      (each-section trout.generator.docbook/sectionize)
      (each-section trout.generator.docbook/convert-tags)
      trout.generator.docbook/prepare-docbook!
      trout.generator.docbook/save-docbook!))

(defn print-order!
  [tree]
  (println)
  (each-chapter tree
                (fn [chapter]
                  (println "CHAPTER")
                  (println "=======")
                  (doseq [section chapter]
                    (println (util/path (:path section) (:file section))))
                  (println)
                  chapter)))

(defn ensure-target [tree target-path]
  (fs/mkdir target-path)
  tree)

(defn -main [& [root & _]]
  (let [target (if root root +BASE+)]
    (-> target
        trout-tree
        (ensure-target (util/path target +TARG+))
        markdown->html
        html->docbook
        print-order!
        done)))
