(ns trout.generator.docbook
  (:require [trout.util :as util]
            [clojure.java.io :as io]
            [fs.core :as fs]
            [clojure.string :as string]
            clojure.pprint)
  (:import [org.apache.commons.lang StringUtils]))

(def prepare-docbook! identity)

(defn ^:private split-sections
  "Splits a string of html just before header (like <h1>, <h2>) tags,
  returning a lazy seq of sections.  Each section is a two-element
  vector [n txt] where n is the integer of the header level (like 1, 2)
  and txt is the full text of that section starting with the header itself."
  [html]
  (->> (cons nil (util/str-part #"<h(\d)>" html))
       (partition 2)
       next
       (map (fn [[[h n] txt]] [(Integer/parseInt n) (str h txt)]))))

(defn ^:private compute-close-open
  "Takes a seq of ints representing header levels.  Returns a lazy seq
  of two-element vectors [closes opens].  'closes' is a seq of ints
  representing the tag levels that need to be closed before the this
  element.  The returned seq is one element longer than the input seq
  so that the final list of closing levels can be returned."
  [head-levels]
  (->> (concat [0] head-levels [1] [nil])
       (partition 3 1)
       (map (fn [[p t not-last]]
              [(range p (dec t) -1)
               (when not-last (range (min (inc p) t) (inc t)))]))))

(def open  (vec (.split #" +" "mxyplyzyk <sect1> <sect2> <sect3> <subsection>")))
(def close (vec (.split #" +" "mxyplyzyk </sect1> </sect2> </sect3> </subsection>")))

(defn sectionize
  "Takes an html string containing header tags and returns an html
  string with head section wrapped by the appropriate level of section
  tags."
  [trout-section]
  (assoc trout-section
    :format/docbook
    (let [sections (split-sections (:format/html trout-section))
          open-close-ints (compute-close-open (map first sections))]
      (apply str (mapcat (fn [[closes opens] [_ txt]]
                           (concat (map close closes) (map open opens) [txt]))
                         open-close-ints (concat sections [[]]))))))

(defn massage-bq [txt]
  (let [attr #"-- (.*)</p>\W</blockquote>"]
    (if-let [m (re-find attr txt)]
      (util/replace-re attr (str "</p><attribution>" (m 1)  "</attribution></blockquote>") txt)
      txt)))

(def figure-tmpl "
<figure id=\"%s\">
          <title>%s</title>
          <mediaobject>
            <imageobject>                                                                                                  
              <imagedata fileref=\"%s\" />
            </imageobject>
            <caption>
              <para>%s</para>
            </caption>
          </mediaobject>
        </figure>")

(defn massage-img [txt]
  (let [img #"<img src=\"(.*)\" alt=\"(.*)\" title=\"\[(.*)\] (.*)\" />"]
    (apply str (for [e (util/str-part img txt)]
                 (if (vector? e)
                   (let [parts [(e 2) (str (e 3) ": " (e 4)) (e 1) (e 4) ""] 
                         strs (util/split #"%s" figure-tmpl)]
                      (apply str (interleave strs parts)))
                   e)))))

(def CO   #";;\s?(?:#(.*):|#)\s*(.*)")
(def COJ  #"//\s?(?:#(.*):|#)\s*(.*)")
(def FCB  #"^;; \[(.*?)\]: (.*)\n")
(def CODE #"<pre><code>([^<]+)</code></pre>")

(defn massage-callouts [base-id txt]
  (if-let [m (re-find CO txt)]
    (apply str (for [[i [text [_ cid ctext]]]
                     (util/indexed (partition 2 2 nil (util/str-part CO txt)))]
                 (if ctext
                   (str text
                        "<co id=\""
                        (if (seq cid) cid (str i "_" base-id))
                        "\" />")
                   text)))
    txt))

(defn gather
  ([regex txt] (gather "?" regex txt))
  ([base-id regex txt]
     (into {} (for [[i [_ id text]] (util/indexed (re-seq regex txt))]
                [(if (seq id) id (str i "_" base-id)) text]))))

(defn emit-callouts [base-id cos]
  (str \newline "<calloutlist>" \newline
       (apply str (for [[i [cid ctext]] (util/indexed cos)]
                    (str "<callout arearefs=\"" cid "\">" ctext "</callout>" \newline)))
       "</calloutlist>" \newline))

(defn build-fcb [cb]
  (let [[_ id title] (re-find FCB cb)
        callouts (into (gather id CO cb) (gather id COJ cb))
        cb (util/replace-re FCB (str "<example id=\"" id "\"><title>" title "</title><programlisting xml:space=\"preserve\">") cb)
        cb (massage-callouts id cb)
        cb (binding [CO COJ] (massage-callouts id cb))
        cb (str cb "</programlisting>" (emit-callouts id callouts)  "</example>")]
    cb))  

(defn build-ifcb [cb]
  (str "<informalexample><programlisting xml:space=\"preserve\">" cb "</programlisting></informalexample>"))

(defn build-specific-cb [cb]
  (if (re-find FCB cb)
    (build-fcb cb)
    (build-ifcb cb)))

(defn build-cb [cb]
  (if (vector? cb)
    (build-specific-cb (cb 1))
    cb))

(defn massage-code [txt]
  (apply str (for [cb (util/str-part CODE txt)] (build-cb cb))))

(def FNDEF #"<p>\[\^(.*)\]:\W(.*)</p>")
(def FNTAG #"\[\^(.*?)\]")

(defn emit-footnotes [txt footnotes]
  (apply str (for [e (util/str-part FNTAG txt)] 
               (if (vector? e) 
                 (if-let [dfn (footnotes (e 1))]
                   (str "<footnote label=\"" (e 1) "\"><para>" dfn "</para></footnote>")
                   (str "<footnote label=\"" (e 1) "\"><para>TBD</para></footnote>"))
                 e))))

(defn massage-footnotes [txt]
  (let [footnotes (gather FNDEF txt)
        txt (util/replace-re FNDEF "" txt)]
    (emit-footnotes txt footnotes)))

(defn convert-tags [sect]
  (assoc sect
    :format/docbook
    (->> (:format/docbook sect)
         (util/replace-re #"<br />" "")
         massage-bq
         massage-img
         massage-code
         massage-footnotes
         (util/replace-re #"([\n>])[ \t]*#!check.*\n" "$1")
         (util/replace-re #"<p><h0>" "<h0>")
         (util/replace-re #"</h0></p>" "</h0>")
         (util/replace-re #"<p>" "<para>")
         (util/replace-re #"</p>" "</para>")
         (util/replace-re #"<h(\d)>" "<title>")
         (util/replace-re #"</h(\d)>" "</title>")
         (util/replace-re #"<ol>" "<para><numberedlist>")
         (util/replace-re #"</ol>" "</numberedlist></para>")
         (util/replace-re #"<ul>" "<para><itemizedlist>")
         (util/replace-re #"</ul>" "</itemizedlist></para>")
         (util/replace-re #"em>" "emphasis>")
         (util/replace-re #"strong>" "emphasis>")
         (util/replace-re #"<li>" "<listitem><para>")
         (util/replace-re #"</li>" "</para></listitem>"))))

(defn save-section-docbook! [sect extension-re]
  (let [tmp     (util/replace-re #"txt.*$" "tmp" (:path sect))
        target  (str (util/replace-re #".*txt\/(.*)" "$1-" (:path sect))
                     (util/replace-re extension-re ".xml" (:file sect)))
        xml    (:format/docbook sect) 
        outpath (util/path (:root sect) tmp target)]
    (println (str "Writing Docbook XML from " (util/path tmp target)))
    (spit outpath xml)
    sect))

(def +PRELUDE+ "<?xml version=\"1.0\" encoding=\"UTF-8\"?>
<book xmlns=\"http://docbook.org/ns/docbook\" version=\"5.0\" xml:lang=\"%s\">
  <bookinfo>
    <title>%s</title>

    <subtitle>%s</subtitle>

    <edition>%s</edition>

    <authorgroup>
    %s
    </authorgroup>
  </bookinfo>")

(def +AUTHOR+ "<author>
      <surname>%s</surname>
      <firstname>%s</firstname>
    </author>")

(def +FINALE+ "</book>")

(defn build-authors [authors]
  (apply str (map #(format +AUTHOR+ (:last-name %) (:first-name %)) authors)))

(defn build-prelude [bk]
  (let [{:keys [lang title subtitle edition authors]} (meta bk)]
    (format +PRELUDE+ lang title subtitle edition (build-authors authors))))

(defn save-docbook! [bk]
  (let [tmp     "tmp"
        target  "book.xml"
        outpath (util/path (:root (meta bk)) tmp target)]
    (println "Writing Docbook XML!")
    (spit outpath 
          (str (build-prelude bk)
               (apply str
                      (map :format/docbook
                           (for [sections bk
                                 :let [paths          (map :path sections)
                                       [_ ch-num & _] (re-find #"(?<=ch)(.+?)(-.*)?$" (first paths))]]
                             {:chapter ch-num
                              :path (do
                                      (when-not (apply = paths)
                                        (print "Warning, path mismatch within chapter: ")
                                        (prn paths))
                                      (first paths))
                              :files (map :file sections)
                              :file (str "ch" ch-num ".xml")
                              :format/docbook (apply str
                                                     (case ch-num 
                                                       "0"  "\n\n<preface  id=\"pref_"
                                                       "00" "\n\n<preface  id=\"pref_"
                                                       "Y"  "\n\n<appendix id=\"appendix_"
                                                       "Z"  "\n\n<appendix id=\"appendix_"
                                                       "\n\n<chapter  id=\"ch_")
                                                     ch-num "\""
                                                     " xreflabel=\"" ch-num "\">\n"
                                                     (concat 
                                                      (map :format/docbook sections)
                                                      (case ch-num
                                                        "0"  ["</preface>"]
                                                        "00" ["</preface>"]                                                          
                                                        "Y"  ["</appendix>"]
                                                        "Z"  ["</appendix>"]
                                                        ["</chapter>"])))})))
               +FINALE+))
    bk))

(defn take-html
  [trout-section]
  (assoc trout-section
    :format/docbook
    (:format/html trout-section)))
