(ns linked.set
  (:require [linked.map :refer [linked-map]]
            [clojure.string :as string]
                                           )
        (:import (clojure.lang Counted
                               IObj
                               IFn
                               ILookup
                               IPersistentCollection
                               IPersistentSet
                               IPersistentVector
                               Reversible
                               Seqable
                               SeqIterator)
                 (java.util Set)
                 (java.lang Iterable)))

(declare empty-linked-set)

(defn linked-set
  ([] empty-linked-set)
  ([x] (if (coll? x) (into empty-linked-set x) (linked-set [x])))
  ([x & more] (apply conj empty-linked-set x more)))

     
(deftype LinkedSet [linked-map]
  IPersistentSet
  (disjoin [_ k]
    (LinkedSet. (dissoc linked-map k)))
  (contains [_ k]
    (contains? linked-map k))
  (get [this k]
    (when (.contains this k) k))

  Set
  (size [this]
    (.count this))

  Iterable
  (iterator [this]
    (SeqIterator. (.seq this)))

  Counted

  IPersistentCollection
  (count [_]
    (count linked-map))
  (cons [_ o]
    (LinkedSet. (cons linked-map [o nil])))
  (empty [_]
    (linked-set))
  (equiv [this o]
    (and (= (.count this) (count o))
         (every? (fn [e] (contains? o e))
                 (.seq this))))

  Seqable
  (seq [_]
    (when-let [s (seq linked-map)] (map key s)))

  Reversible
  (rseq [_]
    (when-let [s (rseq linked-map)] (map key s)))

  IFn
  (invoke [this k]
    (get this k))

  IObj
  (meta [this]
    (.meta ^IObj linked-map))
  (withMeta [this m]
    (LinkedSet. (.withMeta ^IObj linked-map m)))

  Object
  (toString [this]
    (str "#{" (string/join " " (map str this)) "}"))
  (hashCode [this]
    (reduce + (map hash (.seq this))))
  (equals [this other]
    (or (identical? this other)
        (and (instance? Set other)
             (let [^Set s other]
               (and (= (.size this) (.size s))
                    (every? #(.contains s %) (.seq this))))))))

     
(defmethod print-method LinkedSet [o ^java.io.Writer w]
  (.write w "#linked/set ")
  (print-method (seq o) w))


      
                               
        
                  
                                                    
                     
                        

            
                                      

           
                                                                   

       
                                  

             
                 
                                          

                      
                                                   

        
                      
        
                 
                                    
                                
                     

       
                                           

          
                                                           

             
                                                             

             

          
                                     

         
                   
                         
                             
                                     
       
                 

      
                    
                                        

     
                   
                     
                             
                               

                        

                  
                                
                                                                        

                                                            

(def ^{:private true,:tag LinkedSet} empty-linked-set
  (LinkedSet. (linked-map)))

;;;;;;;;;;;; This file autogenerated from src/linked/set.cljx
