(ns linked.set
  (:require [linked.map :refer [empty-linked-map]]
            [clojure.string :as string]
                                           )
        (:import (clojure.lang Counted
                               IObj
                               IFn
                               ILookup
                               IPersistentCollection
                               IPersistentSet
                               IPersistentVector
                               Reversible
                               Seqable
                               SeqIterator)
                 (java.util Set)
                 (java.lang Iterable)))

(declare empty-linked-set)

     
(deftype LinkedSet [linked-map]
  IPersistentSet
  (disjoin [_ k]
    (LinkedSet. (dissoc linked-map k)))
  (contains [_ k]
    (contains? linked-map k))
  (get [this k]
    (when (.contains this k) k))

  Set
  (size [this]
    (.count this))

  Iterable
  (iterator [this]
    (SeqIterator. (.seq this)))

  Counted

  IPersistentCollection
  (count [_]
    (count linked-map))
  (cons [this o]
    (if (contains? linked-map o)
      this
      (LinkedSet. (assoc linked-map o nil))))
  (empty [_]
    empty-linked-set)
  (equiv [this o]
    (and (= (.count this) (count o))
         (every? (fn [e] (contains? o e))
                 (.seq this))))

  Seqable
  (seq [_]
    (when-let [s (seq linked-map)] (map key s)))

  Reversible
  (rseq [_]
    (when-let [s (rseq linked-map)] (map key s)))

  IFn
  (invoke [this k]
    (get this k))

  IObj
  (meta [this]
    (.meta ^IObj linked-map))
  (withMeta [this m]
    (LinkedSet. (.withMeta ^IObj linked-map m)))

  Object
  (toString [this]
    (str "[" (string/join " " (map str this)) "]"))
  (hashCode [this]
    (reduce + (map hash (.seq this))))
  (equals [this other]
    (or (identical? this other)
        (and (instance? Set other)
             (let [^Set s other]
               (and (= (.size this) (.size s))
                    (every? #(.contains s %) (.seq this))))))))

     
(defmethod print-method LinkedSet [o ^java.io.Writer w]
  (.write w "#linked/set ")
  (print-method (into [] o) w))


      
                               
        
                  
                                                   
                     
                        

            
                                      

           
                                                                   

       
                                  

             
                 
                                          

                      
                                                   

        
                      
        
                 
                                    
                                
                     

       
                                           

          
                                                           

             
                                                             

             

          
                                     

         
                   
                         
                             
                                     
       
                 

      
                    
                                        

     
                   
                     
                             
                               

                        

                  
                                
                                                         

(def ^{:tag LinkedSet} empty-linked-set
  (LinkedSet. empty-linked-map))

(def ->linked-set (partial into empty-linked-set))

                                                              

;;;;;;;;;;;; This file autogenerated from src/linked/set.cljx
