(ns linked.map
  (:require [clojure.string :as string]
                                           )
        (:import (clojure.lang Associative
                               Counted
                               IObj
                               IFn
                               ILookup
                               IPersistentCollection
                               IPersistentVector
                               IPersistentMap
                               MapEntry
                               MapEquivalence
                               Reversible
                               Seqable
                               SeqIterator)
                 (java.util Map
                            Map$Entry)
                 (java.lang Iterable)))


(declare empty-linked-map)




(defrecord Node [value left right])

(declare assoc*)
(declare dissoc*)
(declare seq*)
(declare rseq*)

     
(deftype LinkedMap [head delegate-map]
  IPersistentMap
  (assoc [this k v]
    (assoc* this k v))
  (assocEx [this k v]
    (if (.containsKey this k)
      (throw (RuntimeException. "Key already present"))
      (assoc this k v)))
  (without [this k]
    (dissoc* this k))

  MapEquivalence

  Map
  (get [this k]
    (.valAt this k))
  (containsValue [this v]
    (boolean (seq (filter #(= % v) (.values this)))))
  (values [this]
    (map val (.seq this)))
  (size [_]
    (count delegate-map))

  Counted

  IPersistentCollection
  (count [this]
    (.size this))
  (cons [this o]
    (condp instance? o
      Map$Entry (let [^Map$Entry e o]
                  (.assoc this (.getKey e) (.getValue e)))
      IPersistentVector (if (= 2 (count o))
                          (.assoc this (nth o 0) (nth o 1))
                          (throw (IllegalArgumentException. "Vector arg to map conj must be a pair")))
      ;; TODO support for transient to speed up multiple assoc?
      (reduce (fn [^IPersistentMap m ^Map$Entry e]
                (.assoc m (.getKey e) (.getValue e)))
              this
              o)))
  (empty [_]
    (with-meta empty-linked-map (meta delegate-map)))
  (equiv [this o]
    (and (instance? Map o)
         (= (.count this) (count o))
         (every? (fn [[k v]]
                   (= v (get o k)))
                 (.seq this))))

  Seqable
  (seq [this]
    (seq* this))

  Reversible
  (rseq [this]
    (rseq* this))

  Iterable
  (iterator [this]
    (SeqIterator. (.seq this)))

  Associative
  (containsKey [_ k]
    (contains? delegate-map k))
  (entryAt [this k]
    (when (.containsKey this k)
      (MapEntry. k (.valAt this k))))

  ILookup
  (valAt [this k]
    (.valAt this k nil))
  (valAt [_ k not-found]
    (if-let [entry (find delegate-map k)]
      (-> entry val :value)
      not-found))

  IFn
  (invoke [this k]
    (.valAt this k))
  (invoke [this k not-found]
    (.valAt this k not-found))

  IObj
  (meta [this]
    (.meta ^IObj delegate-map))
  (withMeta [this m]
    (LinkedMap. head (.withMeta ^IObj delegate-map m)))

  ;; IEditableCollection

  Object
  (toString [this]
    (str "{" (string/join ", " (for [[k v] this] (str k " " v))) "}"))
  (equals [this other]
    (.equiv this other))
  (hashCode [this]
    (hash (into {} this))))

     
(defmethod print-method LinkedMap [o ^java.io.Writer w]
  (.write w "#linked/map ")
  (.write w (pr-str (into [] o))))

      
                                      
        
                  
                                                                      
                     
                        

            
             
                                   

           
                         
                                                    

       
                                    

             
                     
                       
                                                 
                                     
                     
             
                             
                           
                                                       
                               
                                                                                                    

                      
                                                                   

        
                                              

       
                                           

             

          
                           

             
                             

          
                
                         

         
                   
                         

                             
                                         
                           
                 

              
                    
                      

                          
                               

      
                   
                     

           
                           
                               

     
                   
                     

                             
                               

                        

                  
                                                                                      

;;;; assoc and dissoc impl

(defn- assoc* [^LinkedMap this k v]
  (let [head (.-head this)
        delegate-map (.-delegate-map this)]
    (if-let [entry (find delegate-map k)]
      (LinkedMap. head (assoc-in delegate-map [k :value] v))
      (if (empty? delegate-map)
        (LinkedMap. k (assoc delegate-map k (Node. v k k)))
        (let [tail (get-in delegate-map [head :left])]
          (LinkedMap. head (-> delegate-map
                               (assoc k (Node. v tail head))
                               (assoc-in [head :left] k)
                               (assoc-in [tail :right] k))))))))

(defn- dissoc* [^LinkedMap this k]
  (let [head (.-head this)
        delegate-map (.-delegate-map this)]
    (if-let [entry (find delegate-map k)]
      (if (= 1 (count delegate-map))
        (empty this)
        (let [rk (-> entry val :right)
              lk (-> entry val :left)
              head (if (= k head) rk head)]
          (LinkedMap. head (-> delegate-map
                               (dissoc k)
                               (assoc-in [rk :left] lk)
                               (assoc-in [lk :right] rk)))))
      this)))


;;;; seq and rseq impl

(defn- map-entry [k v]
         (MapEntry. k v)
                     )

(defn- visit-node [delegate-map current last direction]
  (let [[k node] (find delegate-map current)
        entry (map-entry k (:value node))
        next (direction node)]
    (if (= current last)
      (list entry)
      (cons entry (lazy-seq (visit-node delegate-map next last direction))))))

(defn- seq* [^LinkedMap this]
  (let [delegate-map (.-delegate-map this)
        head (.-head this)
        tail (get-in delegate-map [head :left])]
    (when (seq delegate-map)
      (visit-node delegate-map head tail :right))))

(defn- rseq* [^LinkedMap this]
  (let [delegate-map (.-delegate-map this)
        head (.-head this)
        tail (get-in delegate-map [head :left])]
    (when (seq delegate-map)
      (visit-node delegate-map tail head :left))))

(def ^{:tag LinkedMap} empty-linked-map
  (LinkedMap. nil (hash-map)))

(def ->linked-map (partial into empty-linked-map))

                                                              

;;;;;;;;;;;; This file autogenerated from src/linked/map.cljx
