(ns fulcro.incubator.pessimistic-mutations
  "Support for easily defining mutations that automatically update components based on the status and result of a mutation.
  Includes support for a loading, error, and complete status, and makes writing consistent UIs around pessimistic
  mutation (instead of optimistic) easier."
  #?(:cljs (:require-macros fulcro.incubator.pessimistic-mutations))
  (:require
    [fulcro.incubator.db-helpers :as db.h]
    [fulcro.client.mutations :as mutations]
    [fulcro.client.primitives :as fp]
    [fulcro.client.data-fetch :as fetch]
    [fulcro.client.impl.data-targeting :as data-targeting]
    [fulcro.logging :as log]
    [clojure.set :as set]
    [clojure.spec.alpha :as s]))

;; a safe way to save component in app state
(defn- response-component [component] (with-meta {} {:component component}))
(defn- get-response-component [response] (-> response :component meta :component))

(def error-states #{:api-error :network-error})

(defn pessimistic-mutation
  "You must call this function in the remote of mutations that are used with `pmutate!`.

  (defmutation x [_]
    (remote [env] (pessimistic-remote env)))

  NOTES: You *must not* compose this with Fulcro's `returning` or `with-target`.
  You should instead use the special keys of `pmutate`'s params.
  "
  [{:keys [ast ref] :as env}]
  (when (:query ast)
    (log/error "You should not use mutation joins (returning) with `pmutate!`. Use the params of `pmutate!` instead."))
  (-> ast
    (cond-> (:query ast) (update :query vary-meta dissoc :component))
    (mutations/with-target (conj ref ::mutation-response-swap))))

(defn mutation-response
  "Retrieves the mutation response from `this-or-props` (a component or props).  Can also be used against the state-map with an ident."
  ([this-or-props]
   (if (fp/component? this-or-props)
     (mutation-response (-> this-or-props fp/get-reconciler fp/app-state deref) (fp/props this-or-props))
     (-> this-or-props ::mutation-response)))
  ([state props]
   (let [response (-> props ::mutation-response)]
     (if (fulcro.util/ident? response) (get-in state response) response))))

(defn- mutation-status
  ([state props]
   (let [response (mutation-response state props)]
     (-> response ::status)))
  ([this]
   (-> (mutation-response this) ::status)))

(defn mutation-loading?
  "Checks this props of `this` component to see if a mutation is in progress."
  [this]
  (= :loading (mutation-status this)))

(defn mutation-error?
  "Is the mutation in error. This is detected by looking for ::mutation-errors in the ::mutation-response (map) returned by the mutation."
  ([this]
   (contains? error-states (mutation-status this)))
  ([state props]
   (contains? error-states (mutation-status state props))))

(defn get-mutation
  "Runs the side-effect-free multimethod for the given (client) mutation and returns a map describing the mutation:

  {:action (fn [env] ...)
   :remote ...}"
  [env k p]
  (when-let [m (get (methods mutations/mutate) k)]
    (m env k p)))

(defn call-mutation-action
  "Call a Fulcro client mutation action (defined on the multimethod fulcro.client.mutations/mutate). This
  runs the `action` (or `custom-action`) section of the mutation and returns its value."
  ([custom-action env k p]
   (when-let [h (-> (get-mutation env k p) (get (keyword (name custom-action))))]
     (h)))
  ([env k p]
   (call-mutation-action :action env k p)))

(s/def ::mutation-response (s/keys))

(mutations/defmutation mutation-network-error
  "INTERNAL USE mutation."
  [{:keys  [error params]
    ::keys [ref] :as p}]
  (action [env]
    (let [low-level-error (some-> error first second :fulcro.client.primitives/error)
          {::keys [key]} params]
      (db.h/swap-entity! (assoc env :ref ref) assoc ::mutation-response-swap
        (cond-> (dissoc p ::ref :error :params)
          :always (assoc ::status :hard-error)
          key (assoc ::key key)
          low-level-error (assoc ::low-level-error low-level-error))))
    nil))

(mutations/defmutation start-pmutation
  "INTERNAL USE mutation."
  [{::keys [key]}]
  (action [env]
    (db.h/swap-entity! env assoc ::mutation-response {::status :loading
                                                      ::key    key})
    nil))

(mutations/defmutation finish-pmutation
  "INTERNAL USE mutation."
  [{:keys [mutation params]}]
  (action [env]
    (let [{:keys [state ref reconciler]} env
          {::keys [key target returning]} params
          {::keys [mutation-response-swap]} (get-in @state ref)
          {::keys [status]} mutation-response-swap
          hard-error? (= status :hard-error)
          api-error?  (contains? mutation-response-swap ::mutation-errors)
          had-error?  (or hard-error? api-error?)]
      (if had-error?
        (do
          (db.h/swap-entity! env assoc ::mutation-response (merge {::status :api-error} mutation-response-swap {::key key}))
          (call-mutation-action :error-action env mutation params))
        (do
          ;; so the ok action can see it
          (db.h/swap-entity! env (fn [s]
                                   (-> s
                                     (dissoc ::mutation-response-swap)
                                     (assoc ::mutation-response (merge mutation-response-swap {::key key})))))
          (when returning
            (fp/merge-component! reconciler returning mutation-response-swap))
          (when target
            (if-let [return-value-ident (and target (fp/get-ident returning mutation-response-swap))]
              (swap! (:state env) data-targeting/process-target return-value-ident target false)
              (swap! (:state env) data-targeting/process-target (conj ref ::mutation-response-swap) target false)))
          (call-mutation-action :ok-action env mutation params)
          (db.h/swap-entity! env dissoc ::mutation-response)))
      (db.h/swap-entity! env dissoc ::mutation-response-swap))))

(defn pmutate!
  "Run a pmutation defined by `defpmutation`.

  this - The component whose ident will be used for status reporting on the progress of the mutation.
  mutation - The symbol of the mutation you want to run.
  params - The parameter map for the mutation

  The following special keys can be included in `params` to augment how it works:

  - `::pm/key any-value` -
    This k/v pair will be in included in the ::mutation-response at all times. This allows you to distinguish
    among components that share an ident (e.g. one component causes a mutation error, but all with that ident update).
  - `::pm/target - The target for the mutation response (identical to `data-fetch/load`'s target parameter, including support
  for multiple).
  - `::pm/returning` - The component class of the return type, for normalization. If not specified then target will
  not be honored and no merge of the response will remain (only detect loading/errors of mutation).
  "
  [this mutation params]
  (fp/ptransact! this `[(start-pmutation ~params)
                        ~(list mutation params)
                        (fulcro.client.data-fetch/fallback {:action mutation-network-error
                                                            :params ~params
                                                            ::ref   ~(fp/get-ident this)})
                        (finish-pmutation ~{:mutation mutation :params params})]))
