(ns rmap.core
  "The core API for recursive maps."
  (:require [clojure.pprint :refer [simple-dispatch]]))

;;; Internals

(deftype RVal [f]
  clojure.lang.IFn
  (invoke [this ref]
    (f ref))

  Object
  (toString [this] "??"))

(defmethod print-method RVal [rval ^java.io.Writer writer]
  (.write writer (str rval)))

(defmethod simple-dispatch RVal [rval]
  (.write *out* (str rval)))

(declare rval?)

(defn ^:no-doc ->ref*
  [cache]
  (fn ref
    ([key]
     (ref key nil))
    ([key not-found]
     (if-let [[_ val] (find @cache key)]
       (if (rval? val)
         (locking cache
           (let [val (get @cache key)]
             (if (rval? val)
               (let [ret (val ref)]
                 (swap! cache assoc key ret)
                 ret)
               val)))
         val)
       not-found))))

;;; Public API

(defmacro rval
  "Takes a body of expressions and yields an RVal object. The body has
  implicit access to a `ref` function, as would be created by `->ref`."
  [& body]
  `(RVal. (fn [~'ref] ~@body)))

(defn rval?
  "Returns true if x is an RVal."
  [x]
  (instance? RVal x))

(defmacro rmap
  "Takes a literal map (or vector) m, returning m where each of the
  value expressions are wrapped with rval."
  [m]
  (reduce-kv (fn [a k v] (assoc a k `(rval ~v))) m m))

(defn ->ref
  "Given a map (or vector) m, returns a `(fn ref ([key]) ([key
  not-found]))` function. The function returns the value mapped to the
  key in m, not-found or nil if the key is not present. If the value
  is an RVal, it is recursively evaluated using this ref. Evaluation
  of RVals is thread-safe and cached."
  [m]
  (->ref* (atom m)))

(defn finalize
  "Returns updated m, where all rval values are evaluated."
  [m]
  (let [ref (->ref m)]
    (reduce-kv (fn [a k _] (assoc a k (ref k))) m m)))

(defmacro rmap!
  "Same as rmap, but instantly finalized."
  [m]
  `(finalize (rmap ~m)))
