(ns topology-grapher.describe
  (:require [topology-grapher.generator :as ggen]
            [topology-grapher.zipit :as zipit]
            [clojure.string :as s]
            [clojure.java.io :as io])
  (:import (org.apache.kafka.streams StreamsBuilder Topology)))

(defn- get-env-var
  [env-var]
  (System/getenv env-var))

(defn name-for-graph
  "Generate a human readable name for the topology"
  [graph]
  (format "%s.edn"
          (s/join "-" (vals (select-keys graph [:domain
                                                :subdomain
                                                :application
                                                :topology])))))

;;TODO: would be nice to avoid this tight coupling with CircleCI
(defn git-sha [] (get-env-var "CIRCLE_SHA1"))
(defn git-branch [] (get-env-var "CIRCLE_BRANCH"))

(def default-zip-path "/tmp/zips")

(defn app-id
  [config]
  (get config "application.id"))

(defn topology->graph
  [topology config meta-data]
  (merge
   meta-data
   (ggen/describe-topology
    (let [g (topology (StreamsBuilder.))]
      ;; this could be avoided if we enforce a single type
      (if (instance? Topology g)
        g
        (.build g)))

    (app-id config))))

(defn zipfile-path
  [base-path application-name]
  (format "%s/%s_%s.zip"
          base-path
          application-name
          (git-sha)))

(defn gen-topologies
  [topologies meta-data]
  (->> topologies
       (map (fn [tc]
              (topology->graph (:fn tc) (:config tc) meta-data)))
       (map (fn [g] {(name-for-graph g) g}))
       (into {})))

(defn describe-all
  "Describe all the topologies and create a zip file with the result"
  ([topologies meta-data]
   (describe-all topologies meta-data default-zip-path))
  ([topologies meta-data base-path]
   (let [application-name (:application meta-data)
         graphs-by-name (gen-topologies topologies meta-data)
         zip-file-path (zipfile-path base-path application-name)
         zip-file-master (format "%s/%s_%s.zip" base-path "latest" application-name)]

     (io/make-parents zip-file-path)
     (zipit/zip-content zip-file-path graphs-by-name)
     (when (= "master" (git-branch))
       (zipit/zip-content zip-file-master graphs-by-name)))))
