;  Copyright (C) 2020 Gabriel Ash

; This program and the accompanying materials are made available under
; the terms of the Eclipse Public License 2.0 which is available at
; http://www.eclipse.org/legal/epl-2.0 .

; This code is provided as is, without any guarantee whatsoever.

(ns gabrielash.libmisc.ex
  "Error handling"
  (:require [gabrielash.libmisc.printf :refer [sprintf]]
            [clojure.spec.alpha :as s]))

(defn throw-ex
  "throws an [ExceptionInfo](https://clojuredocs.org/clojure.core/ex-info)
   with multiple messages + structured data (map).  
   an optional Throwable `cause` can be included in the `data-map`.
   "
  
  [{:keys [:cause] :as data-map} msg & msgs]
  
   (let [m (->> (cons msg msgs)
                (interpose "\n")
                (reduce str))]
     (if cause 
       (throw (ex-info m
               data-map cause))
       (throw (ex-info m
               data-map)))))
  
  
(defn throw-ex-fmt
  "generates an exception. uses [[sprintf]] to compose messages.   
   `datamap` can have an optional Throwable `cause`"
  [{:keys [:cause] :as data-map} msg & argv]
  (let [m (apply sprintf msg argv)]
    (if cause
      (throw (ex-info m
                      data-map cause))
      (throw (ex-info m
                      data-map)))))


(defn valid?
  "[s/valid?](https://clojuredocs.org/clojure.spec.alpha/valid_q)
   that also explains on fail, unless third arg is :silent"
  [spec value & [silent?]]
  
  (if (s/valid? spec value)
    true
    (do
      (when-not (= silent? :silent)
        (println "ex-valid? explains:")
        (s/explain spec value))
      false)))


