;  Copyright (C) 2020 Gabriel Ash

; This program and the accompanying materials are made available under
; the terms of the Eclipse Public License 2.0 which is available at
; http://www.eclipse.org/legal/epl-2.0 .

; This code is provided as is, without any guarantee whatsoever.

(ns gabrielash.libmisc.spec
  "spec and generator helpers
  
  | name             | args                     | action  
  | ---------------- | -----------------------  | -------------------------------------------- | 
  | entity-map?      |  dispatcher m & [else]   |  tests each value based on key               |
  | lcascii-gen      |                          |  lower-case ascii                            |
  | ucascii-gen      |                          |  uppercase ascii                             |
  | sensible-kw-gen  |                          |  keywords                                    |
  | signs-gen        |                          |  signs ($ % etc)                             |
  | digits-gen       |                          |  digits                                      |
  | ident-gen        |                          |  asci letters, digits, _ - !                 |
  | string-gen       |  char-gen min max        |  strings from char-gen                       |
  | regex-gen        |  [opt char-gen min max]  |  regex from char-gen or lcascii              |
  | predicate-gen    |                          |  random predicates                           |  
  |                  |                          |                                              |    
   
  
  "
  (:require [gabrielash.defnw.core :refer [defnw]]
              [gabrielash.libmisc.core :refer :all]

              [clojure.string :as str]
              [clojure.spec.alpha :as s]
              ;[clojure.spec.test.alpha :as st]
              ;[clojure.test.check.properties :as prop]
              [clojure.test.check.generators :as gen]
              ;[clojure.spec.gen.alpha :as sgen]
              
              ))



(defnw entity-map?
  "validates map `m` against a `dispatcher` map of allowed keys and spec names  
   dispatcher should be `{ :a ::spec-a :b ::spec-b}`   
   `else` is a catchall spec for not found keys. If else is not defined, unfound keys fail  
   for an open test, set else to true"
  
  [dispatcher m & [else]]
  
  {:cond  [(!! seq m) (println "empty map not valid!")]}
  
  (every?
   (fn [[k v]]
     (if-let [spec (get dispatcher k)]
       (s/valid? spec v)
       
       (when else
         (if (= true else)
           true
           (s/valid? else v)))))
   m))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def ^:private field-names 
  [:name :last-name :first-name :email :phone :city :state  
   :role :notes :street :uid :password :page :dob :zipcode 
   :country :region :rank :size :proirity])



(defn sensible-kw-gen 
"generates meaningful keywords"
  [] (gen/elements field-names))
(defn lcascii-gen "lc ascii "    [] (gen/elements [\a \b \c \d \f \g \h \i \j \k \l \m \n \o \p \q \r \s \t \u \v \w \x \y \z]))
(defn ucascii-gen "up ascii "    [] (gen/elements [\A \B \C \D \F \G \H \I \J \K \L \M \N \O \P \Q \R \S \T \U \V \W \X \Y \Z]))
(defn punct1-gen  "ascii signs"  [] (gen/elements [\. \, \; \\ \/ \^ \& \$ \# \@ \~ \% \* \+ \= \? \| \- \_ \! ]))
(defn digits-gen  "digits"       []  (gen/elements [\1 \2 \3 \4 \5 \6 \7 \8 \9 \0])) 
(defn punct2-gen  "_ - !"        []  (gen/elements [\_ \- \!]))

(defn ident-gen  
  "[A-Za-z0-9_!-]"
   []  (gen/frequency [[76 (lcascii-gen)]
                         [8 (ucascii-gen)]
                         [8 (digits-gen)]
                         [8 (punct2-gen)]]))
 
(defn string-gen 
  "strings from `char-generator' with length `min` to `max`   
   default is `lcascii 4 8`"
  ([] 
   (string-gen lcascii-gen 4 8))
  
  ([char-generator min max]
   (gen/fmap str/join
             (gen/vector (char-generator)  min max))))
  
(defn regex-gen
   "regexes from `char-generator' with length `min` to `max`   
   default is `lcascii 4 8`"

  ([]  
   (regex-gen lcascii-gen 5 10))
  
  ([char-generator min max]  
   (gen/fmap  re-pattern
              (string-gen char-generator min max))))

(defn predicate-gen
  "generates functions that randomly return true or false"
  []  
  (gen/elements [(constantly false) (constantly true)]))

