(ns api.remote
  (:require
   [api.util :as util]
   [api.cookie :as cookie]
   [goog.net.XhrIo :as xhr]
   [re-frame.db :refer [app-db]]
   [glossop.core :refer-macros [<? go-catching]]
   [reagent.core :as reagent]
   [cognitect.transit :as t]
   [re-frame.core :refer [reg-event-db path reg-sub dispatch dispatch-sync subscribe]]
   [cljs.core.async :as async :refer [chan close! put! <! timeout]])
  (:require-macros [cljs.core.async.macros :refer [go alt!]]))

(def endpoint (atom nil))

(defn GET [url]
  (let [ch (chan 1)]
    (xhr/send url
              (fn [event]
                (let [res (-> event .-target .getResponseText)]
                  (go (>! ch res)
                      (close! ch)))))
    ch))

(defn cordova-post [url content]
  (let [ch (chan 1)]
    (.post js/cordovaHTTP url
           (clj->js {:body content})
           #js {"Accept-Encoding" "gzip"
                "Cookie" (if-let [token (:state/token @app-db)]
                           (str "token=" token)
                           "")
                "Content-Type" "application/x-www-form-urlencoded"}
           (fn [response]
             (let [res (aget response "data")
                   status (aget response "status")
                   headers (js->clj (aget response "headers") :keywordize-keys true)]
               (async/put! ch (if (= 200 status)
                                {:status :ok :body res :headers headers}
                                {:status :er :body res :headers headers}))
               (close! ch)))
           (fn [response]
             (async/put! ch {:status :er :body (aget response "error")})))
    ch))

(defn post [url content]
  (let [content (.encodeURIComponent js/window content)
        ch (chan 1)
        ctoken (cookie/get "token")
        token (or ctoken (:state/token @app-db))]
    (when (and (not ctoken) token)
      (cookie/set "token" token))
    (xhr/send url
              (fn [event]
                (let [target (.-target event)
                      res (.getResponseText target)
                      status (.getStatus target)
                      headers (js->clj (.getResponseHeaders target) :keywordize-keys true)]
                  (go (>! ch (if (= 200 status)
                               {:status :ok :body res :headers headers}
                               {:status :er :body res :headers headers}))
                      (close! ch))))
              "POST" content #js {"Access-Control-Allow-Origin" "*"
                                  ;;"Accept-Encoding" "gzip"
                                  #_"Cookie" #_(if token
                                             (str "token=" token)
                                             "")
                                  "Content-Type" "application/x-www-form-urlencoded"}
              5000
              true)
    ch))

(def android?
  (try
    (= "Android" (aget js/device "platform"))
    (catch js/Error e
      false)))

(defn api-background
  ([path] (api-background path nil))
  ([path & params]
   (go
    (let [{:keys [status body headers] :as all}
          (<! ((if (and (exists? js/cordovaHTTP)
                        (exists? js/device)
                        (not android?)) cordova-post post)
                  (if (exists? js/cordovaHTTP)
                    (str @endpoint path)
                    path)
               (t/write (t/writer :json) (vec params))))]
      (if (= status :ok)
        (try
          (let [res (t/read (t/reader :json) body)]
            (try
              (with-meta res {:headers headers})
              (catch js/Error e
                res)))
          (catch js/Error e
            (println "API ERROR:")
            (println e)
            ^{:headers headers} {:status :er :msg "UNKNOWN ERROR" :payload all :details :unknown}))
        (let [body (try (t/read (t/reader :json) body) (catch js/Error e body))]
          (if (string? body)
            ^{:headers headers} {:status :er :msg body :details :unknown}
            (try
              (with-meta body {:headers headers})
              (catch js/Error e
                body)))))))))
