(ns api.time
  (:require [cljs-time.core :as time]
            [cljs-time.local :as local]
            [cljs-time.format :as time-format]
            [cljs-time.coerce :as coerce]
            [clojure.string :as string]))

(defn from-date [d] (coerce/from-date d))
(defn to-date [d] (coerce/to-date d))
(defn plus [& args] (apply time/plus args))
(defn days [n] (time/days n))
(defn months [n] (time/months n))
(defn years [n] (time/years n))

(defn before? [d1 d2] (time/before? (from-date d1) (from-date d2)))
(defn after? [d1 d2] (time/after? (from-date d1) (from-date d2)))

(defn day-plus [n]
  (coerce/to-date (time/plus (time/now) (time/days n))))

(defn yesterday []
  (coerce/to-date (time/plus (time/now) (time/days -1))))

(defn tomorrow []
  (coerce/to-date (time/plus (time/now) (time/days 1))))

(defn now []
  (coerce/to-date (time/now)))

(defn parse [format string]
  (when string
    (try (js/Date.
          (coerce/to-long
           (time-format/parse (time-format/formatter format) string)))
         (catch js/Error e
           (.log js/console string)
           (.log js/console e)
           nil))))

(defn format [format date]
  (when date
    (try
      (time-format/unparse (time-format/formatter format)
                           (coerce/from-long (.getTime date)))
      (catch js/Error e
        (.log js/console date)
        (.log js/console e)
        nil))))

(defn today [timezone] (time/plus (time/now) (time/hours timezone)))

(defn apply-timezone [date timezone]
  (when date
    (coerce/to-date (time/plus (coerce/from-date date) (time/hours timezone)))))

(defn default-timezone []
  (let [[s n] (:offset (time/default-time-zone))]
    (js/parseInt (str (name s) n))))

(defn local-now []
  (apply-timezone (js/Date.) (default-timezone)))

(defn day-of-week [d]
  (time/day-of-week (coerce/from-date d)))

(defn hour [d]
  (time/hour (coerce/from-date d)))

(defn month-start [timezone]
  (time/first-day-of-the-month (today timezone)))

(defn month-end [timezone]
  (-> (today timezone)
      (time/plus (time/months 1))
      (time/first-day-of-the-month)
      (time/plus (time/days -1))))

(defn week-start [timezone]
  (let [dt (today timezone)]
    (->> (inc (- (-> dt (time/day-of-week))))
         time/days
         (time/plus dt))))

(defn week-end [timezone]
  (-> (week-start timezone)
      (time/plus (time/weeks 1) (time/days -1))))

(defn year-start [timezone]
  (time/date-time (time/year (today timezone))))

(defn year-end [timezone]
  (-> (year-start timezone)
      (time/plus (time/years 1) (time/days -1))))

(defn business-dt [business-start-time]
  (let [bs (string/replace (str business-start-time) #"[0]" " ")
        hh (subs bs 0 (- (count bs) 2))
        mm (subs bs (count hh))
        hh (js/parseInt hh)
        mm (js/parseInt mm)]
    [hh mm]))

(defn fiscal-start [timezone fiscal-date]
  (let [today (today timezone)]
    (-> fiscal-date
        (time/plus (time/years (- (time/year today) (time/year fiscal-date)))))))

(defn fiscal-end [timezone fiscal-date]
  (let [today (today timezone)]
    (-> fiscal-date
        (time/plus (time/years (inc (- (time/year today)
                                       (time/year fiscal-date)))))
        (time/plus (time/days -1)))))

(defn remove-time [d]
  (time/date-time (time/year d) (time/month d) (time/day d)))

(defn business-date [timezone bus-time]
  (let [d (time/plus (time/plus (time/now) (time/hours 0)) (time/hours timezone))
        [hh mm] (business-dt bus-time)]
    (remove-time
     (if (and (<= (time/hour d) hh)
              (or (not= (time/hour d) hh)
                  (< (time/minute d) mm)))
       (time/plus d (time/days -1))
       d))))

(defn current-shift [timezone bus-time]
  (let [d (business-date timezone bus-time)]
    [(time/plus d (time/days -1)) (time/plus d (time/seconds 1))]))

(defn last-shift [timezone bus-time]
  (let [d (business-date timezone bus-time)]
    [(time/plus d (time/days -2)) (time/plus d (time/days -1) (time/seconds 1))]))

(defn payroll-offset [timezone payroll-date interval]
  (let [days (time/in-days (time/interval payroll-date (today timezone)))
        diff (* (int (/ days interval)) interval)]
    (time/plus payroll-date (time/days diff))))

(defn payroll-start [timezone payroll_type payroll-date]
  (case payroll_type
    :bi-weekly (payroll-offset timezone payroll-date 14)
    :weekly (week-start timezone)
    :monthly (month-start timezone)
    :semi-monthly (let [dt (today timezone)]
                    (time/date-time (time/year dt)
                                    (time/month dt)
                                    (if (< (time/day dt) 16) 1 16)))
    :four-week (payroll-offset timezone payroll-date 28)))

(defn payroll-end [timezone payroll_type payroll-date]
  (case payroll_type
    :bi-weekly (time/plus (payroll-offset timezone payroll-date 14) (time/days 13))
    :weekly (week-end timezone)
    :monthly (month-end timezone)
    :semi-monthly
    (let [dt (today timezone)]
      (time/date-time
       (time/year dt)
       (time/month dt)
       (if (< (time/day dt) 16)
         15
         (time/number-of-days-in-the-month dt))))
    :four-week (time/plus (payroll-offset timezone payroll-date 28) (time/days 28))))

(defn dec-period [dt]
  (let [d (time/day dt)]
    (cond (= 1 d) (time/plus dt (time/months -1) (time/days 15))
          (= 15 d) (let [dt (time/plus dt (time/months -1))]
                     (time/plus
                      dt (time/days (- (time/number-of-days-in-the-month dt) 15))))
          (= 16 d) (time/plus dt (time/days -15))
          :else (time/local-date-time
                 (time/year dt)
                 (time/month dt)
                 15
                 (time/hour dt)
                 (time/minute dt)
                 (time/second dt)))))

(defn inc-period [dt]
  (let [d (time/day dt)]
    (cond (= 1 d) (time/plus dt (time/days 15))
          (= 15 d) (time/plus dt (time/days (- (time/number-of-days-in-the-month dt) 15)))
          (= 16 d) (time/plus dt (time/months 1) (time/days -15))
          :else (let [dt (time/plus dt (time/months 1))]
                  (time/local-date-time
                   (time/year dt)
                   (time/month dt)
                   15
                   (time/hour dt)
                   (time/minute dt)
                   (time/second dt))))))

(defn period-offset [dt offset payroll_type]
  (case payroll_type
    :bi-weekly (time/plus dt (time/weeks (* offset 2)))
    :weekly (time/plus dt (time/weeks offset))
    :monthly (time/plus dt (time/months offset))
    :semi-monthly
    (loop [n offset dt dt]
      (if (zero? n)
        dt
        (if (pos? n)
          (recur (dec n) (dec-period dt))
          (recur (inc n) (inc-period dt)))))
    :four-week (time/plus dt (time/days (* offset 28)))))

(defn days-interval [from to]
  (time/in-days (time/interval (from-date from) (from-date to))))

(defn minutes-interval [from to]
  (time/in-minutes (time/interval (from-date from) (from-date to))))

(defn plus-days [date days]
  (to-date (time/plus (from-date date) (time/days days))))

(defn plus-hours [date hours]
  (to-date (time/plus (from-date date) (time/hours hours))))
