(ns api.remote
  (:require
   [clojure.string :as string]
   [goog.net.XhrIo :as xhr]
   [cljs.reader :refer [read-string]]
   [cognitect.transit :as t]
   [cljs.core.async :as async :refer [chan close! <! >!]])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(def endpoint (atom nil))

(defn GET [url]
  (let [ch (chan 1)]
    (xhr/send url
              (fn [event]
                (let [res (-> event .-target .getResponseText)]
                  (go (>! ch res)
                      (close! ch)))))
    ch))

(defn cordova-post [url content]
  (let [ch (chan 1)]
    (.post js/cordova.plugin.http url
           (clj->js {:body content})
           #js {"Accept-Encoding" "gzip"
                "Content-Type" "application/x-www-form-urlencoded"}
           (fn [response]
             (let [res (aget response "data")
                   status (aget response "status")
                   headers (js->clj (aget response "headers") :keywordize-keys true)]
               (async/put! ch (if (= 200 status)
                                {:status :ok :body res :headers headers}
                                {:status :er :body res :headers headers}))
               (close! ch)))
           (fn [response]
             (async/put! ch {:status :er :body (aget response "error")})))
    ch))

(defonce post-timeout (atom (* 1000 70)))

(defn post [url content]
  (let [content (.encodeURIComponent js/window content)
        ch (chan 1)]
    (xhr/send url
              (fn [event]
                (let [target (.-target event)
                      res (.getResponseText target)
                      status (.getStatus target)
                      headers (js->clj (.getResponseHeaders target) :keywordize-keys true)]
                  (go (>! ch (if (= 200 status)
                               {:status :ok :body res :headers headers}
                               {:status :er :body res :headers headers}))
                      (close! ch))))
              "POST" content #js {"Access-Control-Allow-Origin" "*"
                                  "Content-Type" "application/x-www-form-urlencoded"}
              (if (and url (string/ends-with? url "/send-message")) @post-timeout 0)
              true)
    ch))

(def android?
  (try
    (= "Android" (aget js/device "platform"))
    (catch js/Error _
      false)))

(defn support-transit? []
  (or (not (exists? js/window))
      (not (aget js/window "no_transit_support"))))

(defn api-background
  ([path] (api-background path nil))
  ([path & params]
   (go
    (let [{:keys [status body headers] :as all}
          (<! ((if (and (exists? js/cordova)
                        (exists? js/device)) cordova-post post)
                  (if (exists? js/cordova)
                    (str @endpoint path)
                    path)
               (if (support-transit?)
                 (t/write (t/writer :json) (vec params))
                 (pr-str (vec params)))))]
      (if (= status :ok)
        (try
          (let [res (if (support-transit?)
                      (t/read (t/reader :json) body)
                      (read-string body))]
            (try
              (with-meta res {:headers headers})
              (catch js/Error _
                res)))
          (catch js/Error e
            (println "API ERROR:")
            (println e)
            ^{:headers headers} {:status :er :msg "UNKNOWN ERROR" :payload all :details :unknown}))
        (let [body (try (if (support-transit?)
                          (t/read (t/reader :json) body)
                          (read-string body)) (catch js/Error _ body))]
          (if (string? body)
            ^{:headers headers} {:status :er :msg body :details :unknown}
            (try
              (with-meta body {:headers headers})
              (catch js/Error _
                body)))))))))
