(ns fastmath.special.poly
  (:require [fastmath.core :as m]
            [fastmath.vector :as v]
            [fastmath.complex :as cplx]
            [fastmath.polynomials :as poly])
  (:import [fastmath.java Array]
           [fastmath.vector Vec2]))

(set! *warn-on-reflection* true)
(set! *unchecked-math* :warn-on-boxed)

(defn clenshaw-chebyshev
  ^double [^double x ^doubles c]
  (let [x2 (m/* 2.0 x)
        N (dec (alength c))]
    (loop [i (m/- N 2)
           c0 (Array/aget c (m/dec N))
           c1 (Array/aget c N)]
      (if (m/neg? i)
        (m/+ c0 (m/* c1 x))
        (recur (m/dec i) (m/- (Array/aget c i) c1) (m/+ c0 (m/* c1 x2)))))))

(defn split-poly
  ^Vec2 [^double x ^doubles P]
  (let [N (dec (alength P))
        xx (m/* x x)]
    (loop [i (m/- N 2)
           out (Array/aget P N)
           out2 (Array/aget P (m/dec N))]
      (if (m/< i 1)
        (if (m/odd? N)
          (let [out (m/* out x)]
            (Vec2. (m/- out2 out) (m/+ out2 out)))
          (let [out (m/muladd xx out (Array/aget P 0))
                out2 (m/* out2 x)]
            (Vec2. (m/- out out2) (m/+ out out2))))
        (recur (m/- i 2)
               (m/muladd xx out (Array/aget P i))
               (m/muladd xx out2 (Array/aget P (m/dec i))))))))

(defn split-poly-c
  [^Vec2 x ^doubles P]
  (let [N (dec (alength P))
        xx (cplx/sq x)]
    (loop [i (m/- N 2)
           out (Vec2. (Array/aget P N) 0.0)
           out2 (Vec2. (Array/aget P (m/dec N)) 0.0)]
      (if (m/< i 1)
        (if (m/odd? N)
          (let [out (cplx/mult out x)]
            [(cplx/sub out2 out) (cplx/add out2 out)])
          (let [out (poly/complex-muladd xx out (Vec2. (Array/aget P 0) 0.0))
                out2 (cplx/mult out2 x)]
            [(cplx/sub out out2) (cplx/add out out2)]))
        (recur (m/- i 2)
               (poly/complex-muladd xx out (Vec2. (Array/aget P i) 0.0))
               (poly/complex-muladd xx out2 (Vec2. (Array/aget P (m/dec i)) 0.0)))))))

(defn split-phase-poly
  ^Vec2 [^double x ^doubles P]
  (let [N (dec (alength P))
        xx (m/* x x)
        ap1 (Array/aget P N)
        ap2 (Array/aget P (m/dec N))
        d (m/- (m/muladd 2.0 (m/inc N) -3.0))
        a1 (m// ap1 d)
        a2 (m// (m/- ap2) (m/+ d 2.0))]
    (loop [i (m/- N 2)
           ap1 ap1
           ap2 ap2
           a1 a1
           a2 a2
           d (m/+ d 4.0)]
      (if (m/< i 1)
        (if (m/odd? N)
          (Vec2. (m/muladd x a1 a2) (m/muladd x ap1 ap2))
          (let [p0 (Array/aget P 0)
                ap1 (m/muladd xx ap1 p0)
                a1 (m/muladd xx a1 p0)]
            (Vec2. (m/muladd x a2 a1) (m/muladd x ap2 ap1))))
        (let [p1 (Array/aget P i)
              p2 (Array/aget P (m/dec i))]
          (recur (m/- i 2)
                 (m/muladd xx ap1 p1)
                 (m/muladd xx ap2 p2)
                 (m/muladd xx a1 (m// p1 d))
                 (m/muladd xx a2 (m// p2 (m/+ d 2.0)))
                 (m/+ d 4.0)))))))

(defn even-odd-poly-add
  ^double [^double x ^doubles P]
  (let [N (dec (alength P))
        xx (m/* x x)]
    (loop [i (m/- N 2)
           out (Array/aget P N)
           out2 (Array/aget P (m/dec N))]
      (if (m/< i 1)
        (if (m/odd? N)
          (m/muladd out x out2)
          (let [out (m/muladd xx out (Array/aget P 0))]
            (m/muladd x out2 out)))
        (recur (m/- i 2)
               (m/muladd xx out (Array/aget P i))
               (m/muladd xx out2 (Array/aget P (m/dec i))))))))

(def j0-roots [(Vec2. 2.4048255576957730, -1.176691651530894e-16),
             (Vec2. 3.8317059702075125, -1.5269184090088067e-16),
             (Vec2. 5.5200781102863110, 8.088597146146722e-17),
             (Vec2. 7.0155866698156190, -9.414165653410389e-17),
             (Vec2. 8.6537279129110130, -2.92812607320779e-16),
             (Vec2. 10.173468135062722, 4.482162274768888e-16),
             (Vec2. 11.791534439014281, 2.812956912778735e-16),
             (Vec2. 13.323691936314223, 2.600408064718813e-16),
             (Vec2. 14.930917708487787, -7.070514505983074e-16),
             (Vec2. 16.470630050877634, -1.619019544798128e-15),
             (Vec2. 18.071063967910924, -9.658048089426209e-16),
             (Vec2. 19.615858510468243, -1.004445634526616e-15),
             (Vec2. 21.211636629879260, 4.947077428784068e-16),
             (Vec2. 22.760084380592772, -4.925749373614922e-16),
             (Vec2. 24.352471530749302, 9.169067133951066e-16),
             (Vec2. 25.903672087618382, 4.894530726419825e-16)])

(defn j0-polys
  ^double [^long n ^double r]
  (case (int n)
    0 (poly/mevalpoly r 0.0000000000000000000, -0.5191474972894669, 0.10793870175491979, 0.05660177443794914, -0.008657669593292222, -0.0021942003590739974, 0.0002643770365942964, 4.37291931443113e-5, -4.338825419759815e-6, -5.304927679598406e-7, 4.469819175606667e-8, 4.3284827345621115e-9, -3.135111000732148e-10, -2.628876489517534e-11),
    1 (poly/mevalpoly r -0.402759395702553, 2.476919088072758e-16, 0.20137969785127532, -0.017518715285670765, -0.013352611033152278, 0.0010359438492839443, 0.00037218755624680334, -2.4952042421113972e-5, -5.776086353844158e-6, 3.374317129436161e-7, 5.727482259215452e-8, -2.9561880489355444e-9, -3.905845672635605e-10, 1.971332566705736e-11),
    2 (poly/mevalpoly r 0.000000000000000000, 0.34026480655836816, -0.030820651425593214, -0.05298855286760721, 0.004631042145890388, 0.002257440229081131, -0.00017518572879518415, -4.6521091062878115e-5, 3.199785909661533e-6, 5.716500268232186e-7, -3.5112898510841466e-8, -4.684643389757727e-9, 2.562685034682206e-10, 2.7958958795750104e-11),
    3 (poly/mevalpoly r 0.30011575252613254, -1.6640272822046001e-16, -0.15005787626306408, 0.007129737603121546, 0.011742619737383848, -0.0006260583453094324, -0.00035093119008693753, 1.7929701912295164e-5, 5.6239324892485754e-6, -2.668437501970219e-7, -5.6648488273749086e-8, 2.48117399780498e-9, 3.8876537586241154e-10, -1.6657136713437192e-11),
    4 (poly/mevalpoly r 0.000000000000000000, -0.27145229992838193, 0.015684124960953488, 0.044033774963413, -0.0025093022271948434, -0.0020603351551475315, 0.00011243486771159352, 4.482303558813413e-5, -2.288390108003442e-6, -5.679383588459768e-7, 2.693939234375692e-8, 4.737285529934781e-9, -2.0612709555352797e-10, -2.8163166483726606e-11),
    5 (poly/mevalpoly r -0.2497048770578432, 1.1807897766765572e-16, 0.12485243852891914, -0.0040907858517059345, -0.010102792347641438, 0.00038536375952213334, 0.00031859711440332953, -1.2373899600646271e-5, -5.3013932979548665e-6, 2.001098153528186e-7, 5.4711629662471434e-8, -1.9724572531751518e-9, -3.8121398193699247e-10, 1.3667679743782715e-11),
    6 (poly/mevalpoly r 0.000000000000000000, 0.23245983136472478, -0.009857064513825458, -0.03818600911162367, 0.0016073972920762946, 0.0018420433388794816, -7.581358465623415e-5, -4.159284549011371e-5, 1.650645590334915e-6, 5.425453494592871e-7, -2.0556207467977526e-8, -4.620018928884712e-9, 1.642028058414746e-10, 2.7701605444102412e-11),
    7 (poly/mevalpoly r 0.21835940724787295, -8.89726402965429e-17, -0.10917970362393398, 0.0027314677279632535, 0.008944552393700088, -0.00026391472261453583, -0.00028847875053074687, 8.858193371737123e-6, 4.9233776180403375e-6, -1.5077786827161215e-7, -5.190218733666561e-8, 1.5539413886301204e-9, 3.674809363354973e-10, -1.1113645791216594e-11),
    8 (poly/mevalpoly r 0.00000000000000000, -0.20654643307799603, 0.006916736034268416, 0.034115572697347704, -0.001137276252948717, -0.0016680057255530109, 5.4841792064182565e-5, 3.837965853474541e-5, -1.2335804050962046e-6, -5.106259295634553e-7, 1.592333632709497e-8, 4.423517565793139e-9, -1.3138837384184105e-10, -2.6809397212536384e-11),
    9 (poly/mevalpoly r -0.19646537146865717, 6.979167865106427e-17, 0.09823268573432613, -0.001988037402152532, -0.008095530671166083, 0.00019440675128712672, 0.0002640383898036336, -6.666777683303928e-6, -4.5715696772304925e-6, 1.1666296153560847e-7, 4.8913639696764225e-8, -1.2379867207945651e-9, -3.508930968415813e-10, 9.07632091591013e-12),
    10 (poly/mevalpoly r 0.00000000000000000, 0.18772880304043943, -0.005194182350684612, -0.031096513233785917, 0.0008577442641273341, 0.0015312251534677639, -4.184307585284775e-5, -3.5603170534217916e-5, 9.58002109234601e-7, 4.795250964600283e-7, -1.263434500625308e-8, -4.20550167685701e-9, 1.065791122326672e-10, 2.5727417675684577e-11),
    11 (poly/mevalpoly r 0.18006337534431555, -5.638484737644332e-17, -0.09003168767215539, 0.0015299132863046024, 0.0074441453680234426, -0.00015060569680378768, -0.0002443398416353605, 5.227001519013193e-6, 4.267152607972633e-6, -9.295966007495808e-8, -4.610438011417262e-8, 1.0049092632275165e-9, 3.339442682325105e-10, -7.4991079301099e-12),
    12 (poly/mevalpoly r 0.00000000000000000, -0.17326589422922986, 0.004084217951979124, 0.028749284970146657, -0.0006761643016121907, -0.0014215899173758441, 3.320978125391802e-5, 3.3264379323815026e-5, -7.684444100376941e-7, -4.515479818833484e-7, 1.0271599456634145e-8, 3.993903280527723e-9, -8.793975528824604e-11, -2.4610374225652004e-11),
    13 (poly/mevalpoly r -0.16718460047381803, 4.662597138876655e-17, 0.08359230023690671, -0.0012242529339116864, -0.006925682915280748, 0.00012100729852042988, 0.00022821854128498174, -4.230799709959437e-6, -4.007618360337058e-6, 7.601217108010105e-8, 4.358483157250522e-8, -8.317621452517008e-10, -3.178805429572483e-10, 6.284538399593043e-12),
    14 (poly/mevalpoly r 0.000000000000000000, 0.16170155068925002, -0.0033200234006036146, -0.02685937038656159, 0.0005505380905909195, 0.0013316994659124243, -2.715683205388875e-5, -3.1289544794362e-5, 6.326900360777323e-7, 4.2697141781883964e-7, -8.532447325590315e-9, -3.799009445641295e-9, 7.379552811590934e-11, 2.353654960834717e-11),
    15 (poly/mevalpoly r 0.15672498625285222, 1.1464880342445208e-19, -0.07836249312642612, 0.0010083833270351796, 0.006501011610557426, -9.993664895655577e-5, -0.00021478298462967253, 3.511086890959515e-6, 3.7857022791122945e-6, -6.350944888510364e-8, -4.135588012575766e-8, 7.135988717747828e-10, 3.2075281131621564e-10, 2.208506585533542e-12)))

(def j1-roots [(Vec2. 1.8411837813406593, 4.7898393919093694e-18),
             (Vec2. 3.8317059702075125, -1.5269184090088067e-16),
             (Vec2. 5.3314427735250325, 1.5109105349471405e-16),
             (Vec2. 7.015586669815619, -9.414165653410389e-17),
             (Vec2. 8.536316366346286, -1.5433871213307537e-16),
             (Vec2. 10.173468135062722, 4.482162274768888e-16),
             (Vec2. 11.706004902592063, 7.121366942298246e-16),
             (Vec2. 13.323691936314223, 2.600408064718813e-16),
             (Vec2. 14.863588633909034, -6.265788988781879e-16),
             (Vec2. 16.470630050877634, -1.619019544798128e-15),
             (Vec2. 18.015527862681804, -1.1196999448424267e-16),
             (Vec2. 19.615858510468243, -1.004445634526616e-15),
             (Vec2. 21.16436985918879, 1.7024131380423588e-15),
             (Vec2. 22.760084380592772, -4.925749373614922e-16),
             (Vec2. 24.311326857210776, -2.614798558537172e-16),
             (Vec2. 25.903672087618382, 4.894530726419825e-16)])

(defn j1-polys
  ^double [^long n ^double r]
  (case (int n)
    0 (poly/mevalpoly r 0.5818652242815964, 8.973834293036876e-17, -0.20511071214777496, 0.006058948324597746, 0.013801769808047723, -0.0003723170971916689, -0.0003949590750416506, 9.202953658715798e-6, 6.267292697927347e-6, -1.2681571675674473e-7, -6.319326736018798e-8, 1.057852721352679e-9, 4.839653511789343e-10, -2.1534683937074676e-11),
    1 (poly/mevalpoly r 0.0, -0.402759395702553, 0.05255614585697725, 0.053410444132722765, -0.005179719245639115, -0.00223312533910116, 0.00017466429071755533, 4.6208700909637e-5, -3.0368632747366735e-6, -5.727790851165856e-7, 3.248220586398651e-8, 4.732693354840469e-9, -2.346792988667587e-10, -2.607612190048509e-11),
    2 (poly/mevalpoly r -0.34612620185379156, 2.6002760076991505e-17, 0.16697453550109428, -0.009678268542879972, -0.012099225779175138, 0.0006654009006841076, 0.0003541389011182841, -1.742720356958137e-5, -5.655294960447268e-6, 2.484316221112215e-7, 5.7101369594782676e-8, -2.2587104440771904e-9, -3.9854619701676605e-10, 1.854508641982197e-11),
    3 (poly/mevalpoly r 0.0, 0.30011575252613254, -0.02138921280934146, -0.04697047894974007, 0.0031302917260424344, 0.0021055871432169443, -0.00012550790947431806, -4.499147505171648e-5, 2.401580334889493e-6, 5.665260872870135e-7, -2.7272058756924113e-8, -4.718921752439826e-9, 2.045523137027535e-10, 2.6923755198436e-11),
    4 (poly/mevalpoly r 0.27329994163319987, -1.3511933704624036e-16, -0.13477468037992396, 0.005116340346495344, 0.010631861751990702, -0.000448743683849691, -0.0003268000185531749, 1.3382556752353113e-5, 5.363177082527157e-6, -2.0647461181851255e-7, -5.499892476485602e-8, 1.9780335410646812e-9, 3.8456897851545063e-10, -1.5543946040865833e-11),
    5 (poly/mevalpoly r 0.0, -0.2497048770578432, 0.012272357555101388, 0.04041116939079181, -0.0019268187972545589, -0.0019115826893618788, 8.661729445925766e-5, 4.241116264477775e-5, -1.800978985430059e-6, -5.47159694982648e-7, 2.1682608288127194e-8, 4.629875997238198e-9, -1.6946465290961774e-10, -2.7001263147077087e-11),
    6 (poly/mevalpoly r -0.23330441717143408, 1.3347222236161232e-16, 0.11580092244607755, -0.0032489977328297747, -0.009372527206041913, 0.0003036138212749204, 0.00029804555521007126, -9.813819271763006e-6, -5.0242293237237345e-6, 1.6136482548346424e-7, 5.251755806094281e-8, -1.6213723779041528e-9, -3.7130918416720704e-10, 1.2840797195120732e-11),
    7 (poly/mevalpoly r 0.0, 0.21835940724787295, -0.008194403183877394, -0.03577820957502993, 0.001319573612804675, 0.0017308725061602815, -6.200735153951766e-5, -3.938703730297954e-5, 1.3569938030086903e-6, 5.190652330962483e-7, -1.7082573736581586e-8, -4.464580500726025e-9, 1.3835552663370753e-10, 2.6508049073606443e-11),
    8 (poly/mevalpoly r 0.20701265272531905, -1.1572365385200866e-16, -0.10303781563402706, 0.002289729546293356, 0.008432435058846974, -0.00021965941971569214, -0.000272721543802475, 7.36909520166486e-6, 4.682127415369e-6, -1.2611406291992532e-7, -4.9733737497256667e-8, 1.313333637862963e-9, 3.5584811981981077e-10, -1.0581432980865016e-11),
    9 (poly/mevalpoly r 0.0, -0.19646537146865717, 0.005964112206447895, 0.032382122684890324, -0.0009720337562254532, -0.0015842303417456071, 4.6667442166602986e-5, 3.6572573450276185e-5, -1.0499611641790313e-6, -4.891787571238003e-7, 1.3609422867092438e-8, 4.2640592578446675e-9, -1.1321195999108497e-10, -2.5685037344174917e-11),
    10 (poly/mevalpoly r -0.18801748852581776, 9.806595982257605e-17, 0.09371909377243136, -0.0017233243363111925, -0.007714266760106843, 0.00016755862384546863, 0.000251827174474777, -5.7335177376628445e-6, -4.372425030172481e-6, 1.0045839852031556e-7, 4.698184959516747e-8, -1.0717704296017814e-9, -3.3952344362582674e-10, 8.773851724781324e-12),
    11 (poly/mevalpoly r 0.0, 0.18006337534431555, -0.004589739858905968, -0.029776581472313136, 0.0007530284838491313, 0.0014660390526475475, -3.658900932751018e-5, -3.4137236390211275e-5, 8.366325091910564e-7, 4.6108475352995054e-7, -1.1047200388967438e-8, -4.058793425595929e-9, 9.363513647344476e-11, 2.4720051360744732e-11),
    12 (poly/mevalpoly r 0.1734590492857464, -8.319859584484218e-17, -0.08653590193876039, 0.0013568189856187354, 0.007147216520714834, -0.00013295776598760658, -0.00023462959289705953, 4.6031437800947915e-6, 4.103196271673978e-6, -8.18373301294018e-8, -4.44403211204062e-8, 8.872617322304839e-10, 3.2359730989751983e-10, -7.354373969368619e-12),
    13 (poly/mevalpoly r 0.0, -0.16718460047381806, 0.0036727588017285762, 0.02770273166133465, -0.0006050364924617241, -0.0013693112504440895, 2.9615596890234806e-5, 3.206096184277923e-5, -6.841058767433935e-7, -4.358877331737043e-7, 9.143763763164119e-9, 3.864037532317281e-9, -7.851667555762691e-11, -2.3728757802646805e-11),
    14 (poly/mevalpoly r -0.1618382095526585, 7.115722920799028e-17, 0.08078219522574913, -0.0011038527651152986, -0.006686444726790514, 0.00010870535083026479, 0.00022030197347228358, -3.791805814546456e-6, -3.871147931227952e-6, 6.80607764766372e-8, 4.2160129307780366e-8, -7.460427055677985e-10, -3.0872139357605097e-10, 6.24158293622982e-12),
    15 (poly/mevalpoly r 0.0, 0.15672498625285222, -0.0030251499811055965, -0.026004046442225856, 0.0004996832448037306, 0.001288697907658241, -2.457760943647847e-5, -3.02856199504406e-5, 5.715983781675481e-7, 4.136275513873783e-7, -7.703846567878601e-9, -3.6853569596797437e-9, 6.676439988396374e-11, 2.277101500922319e-11)))


(defn uk-poly-10
  ^doubles [^double p2]
  (let [u10 (poly/mevalpoly p2, 110.01714026924674, -13886.08975371704, 308186.40461266245, -2.7856181280864547e6, 1.3288767166421818e7, -3.7567176660763346e7, 6.634451227472903e7, -7.410514821153265e7, 5.095260249266464e7, -1.9706819118432228e7, 3.284469853072038e6)
        u9 (poly/mevalpoly p2, 24.380529699556064, -2499.8304818112097, 45218.76898136273, -331645.17248456355, 1.2683652733216248e6, -2.8135632265865337e6, 3.763271297656404e6, -2.998015918538107e6, 1.3117636146629772e6, -242919.18790055133)
        u8 (poly/mevalpoly p2, 6.074042001273483, -493.91530477308805, 7109.514302489364, -41192.65496889755, 122200.46498301747, -203400.17728041555, 192547.00123253153, -96980.59838863752, 20204.29133096615)
        u7 (poly/mevalpoly p2, 1.7277275025844574, -108.09091978839466, 1200.9029132163525, -5305.646978613403, 11655.393336864534, -13586.550006434136, 8061.722181737309, -1919.457662318407)
        u6 (poly/mevalpoly p2, 0.5725014209747314, -26.491430486951554, 218.1905117442116, -699.5796273761325, 1059.9904525279999, -765.2524681411817, 212.57013003921713)
        u5 (poly/mevalpoly p2, 0.22710800170898438, -7.368794359479632, 42.53499874538846, -91.81824154324002, 84.63621767460073, -28.212072558200244)
        u4 (poly/mevalpoly p2, 0.112152099609375, -2.3640869140625, 8.78912353515625, -11.207002616222994, 4.669584423426247)
        u3 (poly/mevalpoly p2, 0.0732421875, -0.8912109375, 1.8464626736111112, -1.0258125964506173)
        u2 (poly/mevalpoly p2, 0.0703125, -0.4010416666666667, 0.3342013888888889)
        u1 (poly/mevalpoly p2, 0.125, -0.20833333333333334)]
    (double-array [1.0, u1, u2, u3, u4, u5, u6, u7, u8, u9, u10])))

(defn uk-poly-20
  ^doubles [^double p2]
  (let [u20 (poly/mevalpoly p2 3.646840080706556e10, -1.818726203851104e13, 1.5613123930484672e15, -5.48403360388329e16, 1.0461721131134344e18, -1.2483700995047234e19, 1.0126774169536592e20, -5.8917941350694964e20, 2.548961114664972e21, -8.405915817108351e21, 2.1487414815055883e22, -4.302534303482379e22, 6.783661642951883e22, -8.423222750084323e22, 8.19433100543513e22, -6.173206302884415e22, 3.528435843903409e22, -1.4787743528433614e22, 4.285296082829494e21, -7.671943936729004e20, 6.393286613940837e19)
        u19 (poly/mevalpoly p2 3.8362551802304335e9, -1.7277040123529995e12, 1.3412416915180639e14, -4.2619355104268985e15, 7.351663610930971e16, -7.921651119323832e17, 5.789887667664653e18, -3.025566598990372e19, 1.1707490535797259e20, -3.434621399768417e20, 7.756704953461136e20, -1.360203777284994e21, 1.8571089321463453e21, -1.9677247077053125e21, 1.6016898573693598e21, -9.824438427689858e20, 4.392792200888712e20, -1.351217503435996e20, 2.5563802960529236e19, -2.242438856186775e18)
        u18 (poly/mevalpoly p2 4.259392165047669e8, -1.722832387173505e11, 1.2030115826419191e13, -3.4396530474307594e14, 5.335106978708839e15, -5.1605093193485224e16, 3.37667624979061e17, -1.5736434765189599e18, 5.402894876715982e18, -1.3970803516443374e19, 2.757282981650519e19, -4.178861444656839e19, 4.859942729324836e19, -4.301555703831444e19, 2.846521225167657e19, -1.3639420410571592e19, 4.47020096401231e18, -8.966114215270463e17, 8.30195760673191e16)
        u17 (poly/mevalpoly p2 5.0069589531988926e7, -1.8078220384658062e10, 1.128709145410874e12, -2.886383763141476e13, 4.0004445704303625e14, -3.4503855118462725e15, 2.0064271476309532e16, -8.270945651585064e16, 2.4960365126160426e17, -5.62631788074636e17, 9.575335098169139e17, -1.2336116931960694e18, 1.1961991142756308e18, -8.592577980317548e17, 4.4347954614171904e17, -1.5552983504313904e17, 3.3192764720355224e16, -3.254192619642669e15)
        u16 (poly/mevalpoly p2 6.252951493434797e6, -2.0016469281917763e9, 1.1099740513917902e11, -2.5215584749128545e12, 3.100743647289646e13, -2.3665253045164925e14, 1.2126758042503475e15, -4.3793258383640155e15, 1.1486706978449752e16, -2.2268225133911144e16, 3.213827526858624e16, -3.4447226006485144e16, 2.705471130619708e16, -1.5129826322457682e16, 5.705782159023671e15, -1.3010127235496995e15, 1.3552215870309369e14)
        u15 (poly/mevalpoly p2 832859.3040162893, -2.3455796352225152e8, 1.1465754899448236e10, -2.2961937296824646e11, 2.4850009280340854e12, -1.663482472489248e13, 7.437312290867914e13, -2.3260483118893994e14, 5.230548825784446e14, -8.57461032982895e14, 1.0269551960827625e15, -8.894969398810265e14, 5.4273966498765975e14, -2.213496387025252e14, 5.417751075510605e13, -6.019723417234006e12)
        u14 (poly/mevalpoly p2 118838.42625678326, -2.9188388122220814e7, 1.2470092935127103e9, -2.1822927757529224e10, 2.0591450323241e11, -1.1965528801961816e12, 4.612725780849132e12, -1.2320491305598287e13, 2.334836404458184e13, -3.166708858478516e13, 3.056512551993532e13, -2.0516899410934438e13, 9.109341185239898e12, -2.406297900028504e12, 2.86464035717679e11)   
        u13 (poly/mevalpoly p2 18257.755474293175, -3.8718334425726123e6, 1.43157876718889e8, -2.167164983223795e9, 1.763473060683497e10, -8.786707217802327e10, 2.879006499061506e11, -6.453648692453765e11, 1.0081581068653821e12, -1.0983751560812233e12, 8.192186695485773e11, -3.990961752244665e11, 1.144982377320258e11, -1.4679261247695616e10)
        u12 (poly/mevalpoly p2 3038.090510922384, -549842.3275722887, 1.7395107553978164e7, -2.2510566188941526e8, 1.5592798648792574e9, -6.563293792619285e9, 1.79542137311556e10, -3.3026599749800724e10, 4.1280185579753975e10, -3.4632043388158775e10, 1.8688207509295826e10, -5.866481492051847e9, 8.147890961183121e8)
        u11 (poly/mevalpoly p2 551.3358961220206, -84005.4336030241, 2.2437681779224495e6, -2.4474062725738727e7, 1.420629077975331e8, -4.9588978427503026e8, 1.1068428168230145e9, -1.6210805521083372e9, 1.5535968995705802e9, -9.394623596815784e8, 3.2557307418576574e8, -4.932925366450996e7)
        u10 (poly/mevalpoly p2, 110.01714026924674, -13886.08975371704, 308186.40461266245, -2.7856181280864547e6, 1.3288767166421818e7, -3.7567176660763346e7, 6.634451227472903e7, -7.410514821153265e7, 5.095260249266464e7, -1.9706819118432228e7, 3.284469853072038e6)
        u9 (poly/mevalpoly p2, 24.380529699556064, -2499.8304818112097, 45218.76898136273, -331645.17248456355, 1.2683652733216248e6, -2.8135632265865337e6, 3.763271297656404e6, -2.998015918538107e6, 1.3117636146629772e6, -242919.18790055133)
        u8 (poly/mevalpoly p2, 6.074042001273483, -493.91530477308805, 7109.514302489364, -41192.65496889755, 122200.46498301747, -203400.17728041555, 192547.00123253153, -96980.59838863752, 20204.29133096615)
        u7 (poly/mevalpoly p2, 1.7277275025844574, -108.09091978839466, 1200.9029132163525, -5305.646978613403, 11655.393336864534, -13586.550006434136, 8061.722181737309, -1919.457662318407)
        u6 (poly/mevalpoly p2, 0.5725014209747314, -26.491430486951554, 218.1905117442116, -699.5796273761325, 1059.9904525279999, -765.2524681411817, 212.57013003921713)
        u5 (poly/mevalpoly p2, 0.22710800170898438, -7.368794359479632, 42.53499874538846, -91.81824154324002, 84.63621767460073, -28.212072558200244)
        u4 (poly/mevalpoly p2, 0.112152099609375, -2.3640869140625, 8.78912353515625, -11.207002616222994, 4.669584423426247)
        u3 (poly/mevalpoly p2, 0.0732421875, -0.8912109375, 1.8464626736111112, -1.0258125964506173)
        u2 (poly/mevalpoly p2, 0.0703125, -0.4010416666666667, 0.3342013888888889)
        u1 (poly/mevalpoly p2, 0.125, -0.20833333333333334)]
    (double-array [1.0, u1, u2, u3, u4, u5, u6, u7, u8, u9, u10
                   u11, u12, u13, u14, u15, u16,  u17, u18, u19, u20])))

(defmacro ^:private mevalpoly-m
  [p2 xs v]
  `(poly/mevalpoly ~p2 ~@(map (fn [x] (double (clojure.lang.Ratio. (biginteger x) (biginteger v)))) xs)))

(defn uk-poly-jn
  ^doubles [^double p2]
  (let [u21 (mevalpoly-m p2 [1990919576929585163761247434580873723897677550573731281207275390625, -1094071724893410272201314846735800345180930752601602224440504638671875, 103359845691236916270005420886293061281368263471845448279855946394531250, -3993558675585831919973605001813276532326292885934464373884268722794718750, 83832389176247515253189196480455786065824463879054932596114704927571390625, -1101977288759423115916484463959767795703317649005495798822550381533967842875, 9865413224996821913093061266347613112205666995211786544733850490740903131000, -63509799534443193918054738326913581450607131662586232327271654588312820660616, 305080179205137176095342856930260393560550505786421683990077008259370104309410, -1122099959965657526344757894103766710826629020929459670948834078441128579791750, 3217185258543282976637373169047731813093578126603884759856059695249999306047500, -7276835259402589085458442196686990645669654847254510877738804991391058411412500, 13076651508858985557227319801010895818335803028865299751194038965212274849406250, -18719023753854332443081892087572754119280558462994056787647908433841920235468750, 21307327225910629020600045595173860611314592374884901403059587980149276271875000, -19156728115567236234371593788102013666866274144599851347429312225076676478125000, 13430107219321888006291381503763318985372222835071804983658005207838642173828125, -7186150593017474773099947110903785965879266917528290917207712073663895771484375, 2834031566467842832851805860262261718160136985649155528263587796394390332031250, -776308737033643856044315139790308583914612827783322139063211433790569824218750, 131897976398031751060811874321878385924211075364673046295410087596954345703125, -10468093364923154846096180501736379835254847251164527483762705364837646484375] 5456052820246562178600318839637653652351064473600000000)
        u20 (mevalpoly-m p2 [24030618487110150352755402740028995969072485932314476318359375, -11984379509393886990049682627416290126645799829432886859195312500, 1028816773596376675865176501621547688509187479681533744365175781250, -36136687211104276266628386141898079997731413843884113675698994212500, 689368394712060288513879526213143279387995331221181709158682715671875, -8226054591925395046897310265711439061232478740113589342810359827971600, 66729727980314206345594148553023187689855688854650386958448499394886808, -388235990422199036481157075090292917209702371811112714609151030385239376, 1679621555358289731717827244171539003686077073065087747585589973854567950, -5539024239213671573375139503069183935283937579486175606155069574939719000, 14158993985687610069291256086138044030538819617545349322200892501730615500, -28351273454978996507857547213496220444080209057761228571819950144231575000, 44700502703654649774362294361156754154712937287563245669685024068473468750, -55504285315413734114196925709059066734450069228526494661323483726671250000, 53996017865021964397812742861916826936440170527545458081631955240159375000, -40677946447845849750014263157052100427921007881029648608310681741346250000, 23250401373415767366970579801426233116241475047289980901600427561701171875, -9744288621182737996606001891415854305391392962559389056448157541132812500, 2823768330714179467082676027577350697536971031741905897356560592675781250, -505537818270094147077012813306995849751437826286925078626556809570312500, 42128151522507845589751067775582987479286485523910423218879734130859375] 658943577324464031231922565173629668158341120000000)
        u19 (mevalpoly-m p2 [3761719809509744584195141470215239968860161850335693359375, -1694136104847790923543013061207680485991618397125797873046875, 131518243789528012257323287684549590712219590887856743595703125, -4179129511260217209133623104486559148268490002722848244991240625, 72088266652871136165181276891337618088740053757385007292240377500, -776773977214820033621339638022401758862209813648991378355261599500, 5677394779818071523358076045292335690018332546439714641415150037636, -29667822591847140970922062603154078948366431649274897672067206014580, 114800233558787974267088388638654159779991991716667850063043161177890, -336788945225975997668966486515468748933754942064781256509404101275850, 760599837839891632010677514872412397427959022966733699428711208643750, -1333776105497652608917805362422659621440699261908512326717363285968750, 1821026790520428617034899967974569000414933301802005578083458082187500, -1929493390007550512825649545883182667978851358936168551128507440937500, 1570570304135828069768211937927131121377451398878663909061910457812500, -963355744456233323886326422779275308259891167457840027230510476562500, 430744376085805585076333464506126217301082462279781595185335615234375, -132496442776420617057548516091451843431549350710841121872822607421875, 25067118709566753991500713640672585065184296412786618544560205078125, -2198870062242697718552694179006367110981078632700580574084228515625] 980570799589976236952265721984567958568960000000)
        u18 (mevalpoly-m p2 [17402648254366970318896442155853313710334325579833984375, -7038996381042630440706320955207218277836259136214144531250, 491515845627995608630308828025937223574193727753376100859375, -14053430579296992819276840860026711644143962037898462198750000, 217977088167107844476671248240850549184342689357081793983627500, -2108435312318793739666233505677933977608124223061200130648299000, 13796125542556918366536045481173150324297945459901440053845493532, -64294535084989436534160300189785402934009324116808204634140090960, 220746706223415685134843293656748091845034254294736425040548604450, -570806748959722117700578156666367879819767325936645047354739587500, 1126546324172034608288726239622146620694308510621969350884171781250, -1707362295067866555400074337741150183369276187369879432556935250000, 1985632470023144029098600377248581775333320983961309350778555937500, -1757491631661175749558620292180560395062012011157079033996696875000, 1163006497421870129179321717633624986894646606476886508372367187500, -557267390747417776045984733477369082240579802825315540376718750000, 182639522233726398821209338567430722027981720466294556325849609375, -36632957438678377189819992330283878474735514770655121685644531250, 3391940503581331221279628919470729488401436552838437193115234375] 40857116649582343206344405082690331607040000000)
        u17 (mevalpoly-m p2 [86098445290621992919710288959076381992996044921875, -31086866964344776434901340294726759585844523939453125, 1940900724642360326780559417863159500982538936987375000, -49633551391469586967911551642521760042791355027620125000, 687906693873155071735691272079937403263685738018470814500, -5933198793919698447982330692598158428032189878414161012124, 34502031994800767344892422746634773296018107824906632419848, -142225164683992765990030747899409462088308925365682017857240, 429212352515026773475471295511415883382060846328770993005250, -967487903877461958325237917115228310512630590714451916640750, 1646551275524985252738847924398650627672209050734790851265000, -2121288587929318041669932345514745412624685501425210767875000, 2056954829464921151146089484749565775448230537475447606562500, -1477558757838664620053280135342775196503230238657956773437500, 762596613990574496129519527341431797422236075424587596875000, -267445311988545069854943494211913506927580321987738265625000, 57077468859498937494621367436714109067308943816271513671875, -5595830280343033087707977199677853830128327825124658203125] 1719575616564913434610454759372488704000000)
        u16 (mevalpoly-m p2 [23579874823845695868333654121829112397998046875, -7548208883093506123919073318645848440986281250000, 418571121445853600940987397062487467273966779625000, -9508794888602532435079114127493385198079570710470000, 116929016866180520574682795493722856433879981336112500, -892416493340003758691506680694321990355675722520838096, 4572999438129700505777642155788222350797540050279284760, -16514434054042671665093745143345645271999901089177788400, 43316362356947864694367448897560019067412728259897505250, -83973458255376217113431722753277764119484494850353270000, 121193409013842312774251134029225218007034205572281375000, -129900460304939941764670297010274798325693206430601250000, 102023293586271009821923530738271106285703628596295312500, -57054562339825052419659481088225652020508564894843750000, 21516499723877657209702638528530839860502345904015625000, -4906117886528008036369575428500807148681084440781250000, 511053946513334170455164107135500744654279629248046875] 3770999159133582093443979735465984000000)
        u15 (mevalpoly-m p2 [8178936810213560828419581728001773291015625, -2303431987527333128955769182299845911305390625, 112597271053778779753048514469995937998172890625, -2254933495791765108580529087615802250458013685625, 24403480234538299231733883413666768614198435948125, -163359140754958502896104062604202934925448173291477, 730367145705123976114617970888707594104468177381925, -2284251621937242886581917667066122422330060024456125, 5136561256208409671660362298619778869859994724706875, -8420533422834140468835467666391400380550043688871875, 10085018700249896522602267572484630409640764997271875, -8735135969643867540297524795790262235822823374296875, 5329871927856528282113994744458999865006055974609375, -2173722139119126857509156976742601742357422740234375, 532039967451707060045861997017872377315039521484375, -59115551939078562227317999668652486368337724609375] 9820310310243703368343697227776000000)
        u14 (mevalpoly-m p2 [5448320367052402487647812713291015625, -1338184074771428116079233795614103631250, 57170953417612444837142230812990944671875, -1000503839668383458999731491914516564625300, 9440449669103391509091075981237243128469201, -54857705817658080981995319669299096598482382, 211477117385619365164298957821904115470535555, -564850830044980230366140582063618983657685400, 1070439683260179398514645952824098811025619475, -1451823699927947602004297385351260623500372750, 1401302601668131482630653233972052729323190625, -940627071986145750405920450097257805227812500, 417630985812040040477569028872405152769921875, -110320224449895843354117801955418504167031250, 13133360053559028970728309756597440972265625] 45846453362482275295722209280000)
        u13 (mevalpoly-m p2 [17438611142828905996129258798828125, -3698121486504259988094897605296209375, 136735019134677724428035696765082813750, -2069933923586966756183324291232117362250, 16843538631795795357786827345307534156063, -83924867223075156862785921508524155665245, 274983827478138958623041508409195988431140, -616410216242554698436702353237166008656700, 962926533925253374359704288384340809260875, -1049095945162229046324321461816274931715625, 782463969315283937856703223178540650343750, -381190503845282445953419057314665534156250, 109361210755577700442544717509565392265625, -14020668045586884672121117629431460546875] 955134445051714068660879360000)
        u12 (mevalpoly-m p2 [120907703923613748239829527671875, -21882222767154197351962677311437500, 692277766674325563109617997687563750, -8958590476947726766450604043798559500, 62055079517573388459132793029571876461, -261201165596865827608687437905740780920, 714528665351965363868467348116538170900, -1314368459332124683504418467809129387000, 1642838631056253395899341314188134178125, -1378260730939829908037976894025628887500, 743739612850105971846901081692858843750, -233469939346545526651936774615688437500, 32426380464797989812768996474401171875] 39797268543821419527536640000)
        u11 (mevalpoly-m p2 [15237265774872558064250728125, -2321657500166464779660536015625, 62011003282542082472252466220875, -676389476843440422173605288596087, 3926191452593448964331218647028194, -13704902022868787041097596217578170, 30589806122850866110941936529264950, -44801790321820682384740638703320750, 42936745153513007436411401865860625, -25963913760458280822131901737878125, 8997860461116953237934638500359375, -1363312191078326248171914924296875] 27636992044320430227456000)
        u10 (mevalpoly-m p2 [9745329584487361980740625, -1230031256571145165088463750, 27299183373230345667273718125, -246750339886026017414509498824, 1177120360439828012193658602930, -3327704366990695147540934069220, 5876803711285273203043452095250, -6564241639632418015173104205000, 4513386761946134740461797128125, -1745632061522350031610173343750, 290938676920391671935028890625] 88580102706155225088000)
        u9 (mevalpoly-m p2 [6427469716717690265625, -659033454841709672064375, 11921080954211358275362500, -87432034049652400520788332, 334380732677827878090447630, -741743213039573443221773250, 992115946599792610768672500, -790370708270219620781737500, 345821892003106984030190625, -64041091111686478524109375] 263631258054033408000)
        u8 (mevalpoly-m p2 [134790179652253125, -10960565081605263000, 157768535329832893644, -914113758588905038248, 2711772922412520971550, -4513690624987320777000, 4272845805510421639500, -2152114239059719935000, 448357133137441653125] 22191183337881600)
        u7 (mevalpoly-m p2 [199689155040375, -12493049053044375, 138799253740521843, -613221795981706275, 1347119637570231525, -1570320948552481125, 931766432052080625,  -221849150488590625] 115579079884800)
        u6 (mevalpoly-m p2 [2757049477875, -127577298354750, 1050760774457901, -3369032068261860,5104696716244125, -3685299006138750, 1023694168371875] 4815794995200)
        u5 (mevalpoly-m p2 [1519035525, -49286948607, 284499769554, -614135872350, 566098157625, -188699385875] 6688604160)
        u4 (mevalpoly-m p2 [4465125, -94121676, 349922430, -446185740, 185910725] 39813120)
        u3 (mevalpoly-m p2 [30375, -369603, 765765, -425425] 414720)
        u2 (mevalpoly-m p2 [81, -462, 385] 1152)
        u1 (mevalpoly-m p2 [3, -5] 24)]
    (double-array [1.0, u1, u2, u3, u4, u5, u6, u7, u8, u9, u10, u11, u12, u13, u14, u15, u16,  u17, u18, u19, u20, u21])))

(defn a-ap-poly-10
  ^Vec2 [^double v ^double x]
  (let [xinv (m/sq (m// x))
        mu (m/* 4.0 v v)
        s10 (poly/mevalpoly mu -7.181611223268327e10, 8.659091218989731e10, -1.576853447634526e10, 1.0243675365394899e9, -3.1127612822052997e7, 499001.19046756154, -4427.281508191307, 21.557856538012857, -0.053067848184582544, 5.209072696743533e-5, -8.843926480039954e-9)
        s9 (poly/mevalpoly mu 8.369564497799761e8, -1.0061560971482614e9, 1.80175839935865e8, -1.1293844119552301e7, 322272.3393258663, -4655.44006793818, 34.77845393284224, -0.12591543863527477, 0.0001761710154823959, -4.1618477553129196e-8)
        s8 (poly/mevalpoly mu -1.225106763286615e7, 1.4670878180751745e7, -2.5692206274838205e6, 153362.43350451812, -4001.4928083932027, 49.41718461737037, -0.2788656409829855, 0.0005833245813846588, -1.9976869225502014e-7)
        s7 (poly/mevalpoly mu 231884.6359563172, -276225.9131319225, 46889.68168082833, -2607.1347218453884, 59.29277476668358, -0.5644365847110748, 0.0018794238567352295, -9.834766387939453e-7)
        s6 (poly/mevalpoly mu -5892.0457146167755, 6965.548173427582, -1128.6143367290497, 56.11658954620361, -1.0105445384979248, 0.005837917327880859, -5.0067901611328125e-6)
        s5 (poly/mevalpoly mu 211.27614974975586, -246.8455924987793, 37.067405700683594, -1.5151596069335938, 0.017223358154296875, -2.6702880859375e-5)
        s4 (poly/mevalpoly mu -11.466461181640625, 13.1358642578125, -1.71624755859375, 0.0469970703125, -0.000152587890625)
        s3 (poly/mevalpoly mu 1.0478515625, -1.1591796875, 0.1123046875, -0.0009765625)
        s2 (poly/mevalpoly mu -0.1953125, 0.203125, -0.0078125)
        s1 (poly/mevalpoly mu 0.125, -0.125)
        aap (split-phase-poly xinv (double-array [1.0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10]))]
    (v/emult aap (Vec2. x 1.0))))

(defn a-ap-poly-20
  ^Vec2 [^double v ^double x]
  (let [xinv (m/sq (m// x))
        mu (m/* 4.0 v v)
        s20 (poly/mevalpoly mu -1.1655823449858069e34, 1.4226232333293694e34, -2.7683048917097874e33, 2.0549175959945544e32, -7.766193794571668e30, 1.7289192571208397e29, -2.473355129610242e27, 2.406335820680262e25, -1.6548511836241395e23, 8.261988206301513e20, -3.048956971903377e18, 8.409146358453666e15, -1.7416884938569994e13, 2.7048317602344254e10, -3.1216507081066556e7, 26272.21159305084, -15.578788452840763, 0.0061163849392160095, -1.4107081622121022e-6, 1.4671084930995856e-10, -2.9236916960932355e-15)
        s19 (poly/mevalpoly mu 3.1436617397720927e31, -3.8346121964186915e31, 7.438172213027661e30, -5.487658176284203e29, 2.0543110479331377e28, -4.512190593153004e26, 6.33949148154985e24, -6.024489803885022e22, 4.020727203383251e20, -1.9328978639294188e18, 6.802550072768408e15, -1.767751414955178e13, 3.3964900375451138e10, -4.792703592867111e7, 48822.71795456563, -34.73905348751307, 0.01624286720956346, -4.43222001918643e-6, 5.421693881235296e-10, -1.2642991118241018e-14)
        s18 (poly/mevalpoly mu -9.432675683095812e28, 1.1498154949132059e29, -2.2224123014148547e28, 1.6283472309548362e27, -6.03034763221407e25, 1.3043849059659297e24, -1.7950710383376238e22, 1.6602291470434327e20, -1.0700309670904285e18, 4.92026735148727e15, -1.6365119263912514e13, 3.957355516860315e10, -6.930551343828379e7, 86599.59532013819, -74.82883257079231, 0.042123857348712865, -1.3735762526818516e-5, 1.994760566671595e-9, -5.490670428493242e-14)
        s17 (poly/mevalpoly mu 3.167717078292175e26, -3.8584266638650446e26, 7.4277307068246446e25, -5.399729057323595e24, 1.9752248974202056e23, -4.197842582814349e21, 5.640147455618286e19, -5.0538188484912346e17, 3.1258357574848365e15, -1.3629748027461303e13, 4.233000035928751e10, -9.36296364559977e7, 145724.23973885347, -155.01792293897955, 0.10637041031529318, -4.191933776256961e-5, 7.303030923530292e-9, -2.3959289142515966e-13)
        s16 (poly/mevalpoly mu -1.1986350446061028e24, 1.458736721654471e24, -2.795293505867967e23, 2.0139038975089457e22, -7.262922096794037e20, 1.5122633519520676e19, -1.975555191650512e17, 1.7050149800082885e15, -1.0037673890747877e13, 4.1025284625807785e10, -1.1700278886842367e8, 230919.39384718135, -307.22947128726474, 0.260626935759175, -0.0001257334987819726, 2.658906071879852e-8, -1.051117201091023e-12)
        s15 (poly/mevalpoly mu 5.149237018485509e21, -6.260437088008504e21, 1.1933270317945127e21, -8.508441473311652e19, 3.0180281284560543e18, -6.1344355915029736e16, 7.750474619361342e14, -6.393706192980096e12, 3.543452904857713e10, -1.3358182185805681e8, 341472.1144099866, -578.7806110577233, 0.6169921485050097, -0.00036976968916281605, 9.619824624385043e-8, -4.639413853091412e-12)
        s14 (poly/mevalpoly mu -2.5330554576498356e19, 3.0761765051109855e19, -5.827690025164462e18, 4.1047854971841363e17, -1.4277576770556304e16, 2.819724813029272e14, -3.4214400566025547e12, 2.670056925037507e10, -1.3717353508462998e8, 465890.6725849346, -1028.1881576107903, 1.4039031226739573, -0.0010631539871280848, 3.4554354377824836e-7, -2.061961712485072e-11)
        s13 (poly/mevalpoly mu 1.4409993748135704e17, -1.747635582560889e17, 3.286944341694018e16, -2.2818860422499875e15, 7.75248528894739e13, -1.4783938404512664e12, 1.7064572063331423e10, -1.2415577730993801e8, 578046.6774168271, -1705.2186748496474, 3.0504712266430722, -0.0029777336087200013, 1.2309086638850886e-6, -9.237588471933122e-11)
        s12 (poly/mevalpoly mu -9.588236853265692e14, 1.161016591203577e15, -2.164822377227822e14, 1.4768015151140877e13, -4.8748767898914606e11, 8.900147704533855e9, -9.641151011038888e7, 640028.5410983711, -2606.385824929625, 6.277422845836725, -0.008087900592073538, 4.342405588886322e-6, -4.176996526439325e-10)
        s11 (poly/mevalpoly mu 7.562132599676877e12, -9.139342212003133e12, 1.686263984004851e12, -1.1257747938365613e11, 3.584384568770644e9, -6.188986771037158e7, 616601.7257690447, -3609.808395761912, 12.103919447801218, -0.021179858537379914, 1.5144118606258417e-5, -1.909484126372263e-9)        
        s10 (poly/mevalpoly mu -7.181611223268327e10, 8.659091218989731e10, -1.576853447634526e10, 1.0243675365394899e9, -3.1127612822052997e7, 499001.19046756154, -4427.281508191307, 21.557856538012857, -0.053067848184582544, 5.209072696743533e-5, -8.843926480039954e-9)
        s9 (poly/mevalpoly mu 8.369564497799761e8, -1.0061560971482614e9, 1.80175839935865e8, -1.1293844119552301e7, 322272.3393258663, -4655.44006793818, 34.77845393284224, -0.12591543863527477, 0.0001761710154823959, -4.1618477553129196e-8)
        s8 (poly/mevalpoly mu -1.225106763286615e7, 1.4670878180751745e7, -2.5692206274838205e6, 153362.43350451812, -4001.4928083932027, 49.41718461737037, -0.2788656409829855, 0.0005833245813846588, -1.9976869225502014e-7)
        s7 (poly/mevalpoly mu 231884.6359563172, -276225.9131319225, 46889.68168082833, -2607.1347218453884, 59.29277476668358, -0.5644365847110748, 0.0018794238567352295, -9.834766387939453e-7)
        s6 (poly/mevalpoly mu -5892.0457146167755, 6965.548173427582, -1128.6143367290497, 56.11658954620361, -1.0105445384979248, 0.005837917327880859, -5.0067901611328125e-6)
        s5 (poly/mevalpoly mu 211.27614974975586, -246.8455924987793, 37.067405700683594, -1.5151596069335938, 0.017223358154296875, -2.6702880859375e-5)
        s4 (poly/mevalpoly mu -11.466461181640625, 13.1358642578125, -1.71624755859375, 0.0469970703125, -0.000152587890625)
        s3 (poly/mevalpoly mu 1.0478515625, -1.1591796875, 0.1123046875, -0.0009765625)
        s2 (poly/mevalpoly mu -0.1953125, 0.203125, -0.0078125)
        s1 (poly/mevalpoly mu 0.125, -0.125)
        aap (split-phase-poly xinv (double-array [1.0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10
                                                  s11, s12, s13, s14, s15, s16, s17, s18, s19, s20]))]
    (v/emult aap (Vec2. x 1.0))))

(defn a-ap-poly-30
  ^Vec2 [^double v ^double x]
  (let [xinv (m/sq (m// x))
        mu (m/* 4.0 v v)
        s30 (even-odd-poly-add mu (double-array '(-7.592029625744128e61, 9.300807898884993e61, -1.8453653700408117e61, 1.4213662513968141e60, -5.684267643368571e58, 1.3688721521734006e57, -2.1717210753568686e55, 2.410362055956752e53, -1.953160895382593e51, 1.1926329508677187e49, -5.621526725185766e46, 2.084010993835956e44, -6.16598654084696e41, 1.472798231204338e39, -2.8653790379578294e36, 4.5711759648399396e33, -6.008229949414824e30, 6.525582948238488e27, -5.863525845645105e24, 4.3559252317865593e21, -2.6682229171870643e18, 1.3409000748848645e15, -5.4845348768282446e11, 1.8042426705573353e8, -46913.0207873715, 9.39407807634952, -0.0013913710768178739, 1.4251320396079487e-7, -8.917800204478063e-12, 2.6068908049173686e-16, -1.5080064817014915e-21)))
        s29 (even-odd-poly-add mu (double-array '(8.871640240590993e58, -1.0865691207568645e59, 2.153044379065757e58, -1.654292598675498e57, 6.591312178536168e55, -1.5792626981209235e54, 2.4890507915218916e52, -2.7398535977577976e50, 2.1978557435213864e48, -1.3258581553903244e46, 6.160064953128667e43, -2.24525395106052e41, 6.512651083776791e38, -1.520126754804397e36, 2.879357904563898e33, -4.453253892004961e30, 5.646779978807957e27, -5.882834513685185e24, 5.036061410449725e21, -3.535410901341911e18, 2.0262400502486065e15, -9.410029770358712e11, 3.500781492364049e8, -102565.9203518178, 23.065519747567176, -0.003825019881917803, 4.374396753566091e-7, -3.048458029283708e-11, 9.900967356259312e-16, -6.3495009755852275e-21)))
        s28 (even-odd-poly-add mu (double-array '(-1.1113660587118494e56, 1.360796163396552e56, -2.692648562464633e55, 2.0634410442908336e54, -8.188692555802185e52, 1.9512412867297912e51, -3.0534326199740573e49, 3.3310997303957132e47, -2.6429272553053182e45, 1.573358131379098e43, -7.195503085038196e40, 2.5742410447166102e38, -7.305470671594484e35, 1.6621868613473995e33, -3.056128428896728e30, 4.5656808004766417e27, -5.560280477035352e24, 5.525982611412455e21, -4.4762143769251487e18, 2.944057191090032e15, -1.561376913130651e12, 6.60436019199837e8, -219122.78889988337, 55.602561558610454, -0.010370022987617815, 1.3297453832052619e-6, -1.0361746982713131e-10, 3.753418608703549e-15, -2.6783349569741323e-20)))
        s27 (even-odd-poly-add mu (double-array '(1.4962552090981924e53, -1.8315342344152888e53, 3.6186332587139524e52, -2.7651411770117913e51, 1.092593756883467e50, -2.5880088398643682e48, 4.0185482602383215e46, -4.341325948517365e44, 3.40330197791503e42, -1.996788871925338e40, 8.974814912978575e37, -3.1454308144632116e35, 8.712781142577153e32, -1.926817207371682e30, 3.4266421642703216e27, -4.923376108039163e24, 5.727723667496557e21, -5.393876507549732e18, 4.099268265682976e15, -2.498492707776039e12, 1.2086394541382809e9, -456609.83322422515, 131.40817769348385, -0.027695807224792045, 4.000233368100494e-6, -3.500503095889607e-10, 1.4200746208673407e-14, -1.1319755289852937e-19)))
        s26 (even-odd-poly-add mu (double-array '(-2.170796074547977e50, 2.6563886780963143e50, -5.239744551853566e49, 3.9915243203539826e48, -1.5697710844957169e47, 3.694226454576932e45, -5.687790414754595e43, 6.07923257088654e41, -4.703231744077508e39, 2.715669597352839e37, -1.1973880447446774e35, 4.101798034247621e32, -1.1059082744278117e30, 2.3689970250205793e27, -4.057770223206878e24, 5.577674944614657e21, -6.157880219725851e18, 5.448993990587677e15, -3.8435237604702446e12, 2.1401262084993436e9, -926134.0995426074, 303.9782133922326, -0.07278018905169194, 1.1898982367956518e-5, -1.1748038677566303e-9, 5.361217221358988e-14, -4.794249299231832e-19)))
        s25 (even-odd-poly-add mu (double-array '(3.4037481878842518e47, -4.163722422572948e47, 8.198385202264364e46, -6.22435990386475e45, 2.4353618503436353e44, -5.690731110001809e42, 8.680616408346572e40, -9.169497515863974e38, 6.991529767882026e36, -3.9660440566717967e34, 1.7117897755074633e32, -5.716342746132164e29, 1.495186253411991e27, -3.089698915505606e24, 5.071094031947313e21, -6.62564031827467e18, 6.884682491993269e15, -5.663687113496129e12, 3.6558083666029415e9, -1.8241160174217927e6, 687.0067933018984, -0.1879264018835715, 3.496541491784091e-5, -3.914759734732158e-9, 2.0193378048364477e-13, -2.035109906612696e-18)))
        s24 (even-odd-poly-add mu (double-array '(-5.786059033524315e44, 7.075318200534661e44, -1.3904375539321083e44, 1.051770598467428e43, -4.0921342782576984e41, 9.487906954286141e39, -1.4325415965940442e38, 1.493677625028016e36, -1.1206631946323043e34, 6.2329465039514265e31, -2.626782977025989e29, 8.52403241555426e26, -2.1544180262833415e24, 4.273241060880315e21, -6.678193343397941e18, 8.226793217207085e15, -7.9615210223039795e12, 6.004451011853662e9, -3.4795218728555855e6, 1513.831326319914, -0.4760701870709445, 0.00010139500470517733, -1.2944568154864108e-8, 7.586889731852131e-13, -8.660042155798706e-18)))
        s23 (even-odd-poly-add mu (double-array '(1.0699708275991454e42, -1.3078533633276038e42, 2.5647480608865752e41, -1.9322501955238208e40, 7.471528599453178e38, -1.7175047656287374e37, 2.56398527252613e35, -2.6350896428114214e33, 1.9417708082703794e31, -1.0563755426904998e29, 4.333903992576568e26, -1.3614388053852227e24, 3.3089693719445074e21, -6.261128573879847e18, 9.243346554045246e15, -1.0625592945166727e13, 9.4457596395905e9, -6.4083469576330995e6, 3244.624388704194, -1.1811472686414501, 0.0002898188382446864, -4.2443950630387665e-8, 2.84271079789386e-12, -3.694951319807448e-17)))
        s22 (even-odd-poly-add mu (double-array '(-2.1603839510459968e39, 2.639521237493071e39, -5.164169654060501e38, 3.873385769490781e37, -1.4875551809791352e36, 3.3870740530372218e34, -4.993086633044377e32, 5.049452384590897e30, -3.646484825050138e28, 1.934929035289944e26, -7.699788293506606e23, 2.3306500988206443e21, -5.414879939834449e18, 9.69892178268169e15, -1.3389590376401615e13, 1.4169012254830626e10, -1.1355350548005087e7, 6745.941285011119, -2.8643019266046954, 0.0008154154591207295, -1.3789599324701242e-7, 1.061958568923219e-11, -1.5810954484757452e-16)))
        s21 (even-odd-poly-add mu (double-array '(4.7820456670665866e36, -5.839767806978455e36, 1.139609569657776e36, -8.505736729879162e34, 3.241923388548771e33, -7.303696541971675e31, 1.061598557628623e30, -1.0542835823932327e28, 7.4416439961740466e25, -3.8383112218012e23, 1.4749530840065974e21, -4.277160387520148e18, 9.428034079487212e15, -1.582775791680076e13, 2.0161618573658318e10, -1.9279964972155876e7, 13562.78784549061, -6.773533616718505, 0.0022547380647248, -4.4351435924730097e-7, 3.9542564728199e-11, -6.787141437359297e-16)))
        
        s20 (poly/mevalpoly mu -1.1655823449858069e34, 1.4226232333293694e34, -2.7683048917097874e33, 2.0549175959945544e32, -7.766193794571668e30, 1.7289192571208397e29, -2.473355129610242e27, 2.406335820680262e25, -1.6548511836241395e23, 8.261988206301513e20, -3.048956971903377e18, 8.409146358453666e15, -1.7416884938569994e13, 2.7048317602344254e10, -3.1216507081066556e7, 26272.21159305084, -15.578788452840763, 0.0061163849392160095, -1.4107081622121022e-6, 1.4671084930995856e-10, -2.9236916960932355e-15)
        s19 (poly/mevalpoly mu 3.1436617397720927e31, -3.8346121964186915e31, 7.438172213027661e30, -5.487658176284203e29, 2.0543110479331377e28, -4.512190593153004e26, 6.33949148154985e24, -6.024489803885022e22, 4.020727203383251e20, -1.9328978639294188e18, 6.802550072768408e15, -1.767751414955178e13, 3.3964900375451138e10, -4.792703592867111e7, 48822.71795456563, -34.73905348751307, 0.01624286720956346, -4.43222001918643e-6, 5.421693881235296e-10, -1.2642991118241018e-14)
        s18 (poly/mevalpoly mu -9.432675683095812e28, 1.1498154949132059e29, -2.2224123014148547e28, 1.6283472309548362e27, -6.03034763221407e25, 1.3043849059659297e24, -1.7950710383376238e22, 1.6602291470434327e20, -1.0700309670904285e18, 4.92026735148727e15, -1.6365119263912514e13, 3.957355516860315e10, -6.930551343828379e7, 86599.59532013819, -74.82883257079231, 0.042123857348712865, -1.3735762526818516e-5, 1.994760566671595e-9, -5.490670428493242e-14)
        s17 (poly/mevalpoly mu 3.167717078292175e26, -3.8584266638650446e26, 7.4277307068246446e25, -5.399729057323595e24, 1.9752248974202056e23, -4.197842582814349e21, 5.640147455618286e19, -5.0538188484912346e17, 3.1258357574848365e15, -1.3629748027461303e13, 4.233000035928751e10, -9.36296364559977e7, 145724.23973885347, -155.01792293897955, 0.10637041031529318, -4.191933776256961e-5, 7.303030923530292e-9, -2.3959289142515966e-13)
        s16 (poly/mevalpoly mu -1.1986350446061028e24, 1.458736721654471e24, -2.795293505867967e23, 2.0139038975089457e22, -7.262922096794037e20, 1.5122633519520676e19, -1.975555191650512e17, 1.7050149800082885e15, -1.0037673890747877e13, 4.1025284625807785e10, -1.1700278886842367e8, 230919.39384718135, -307.22947128726474, 0.260626935759175, -0.0001257334987819726, 2.658906071879852e-8, -1.051117201091023e-12)
        s15 (poly/mevalpoly mu 5.149237018485509e21, -6.260437088008504e21, 1.1933270317945127e21, -8.508441473311652e19, 3.0180281284560543e18, -6.1344355915029736e16, 7.750474619361342e14, -6.393706192980096e12, 3.543452904857713e10, -1.3358182185805681e8, 341472.1144099866, -578.7806110577233, 0.6169921485050097, -0.00036976968916281605, 9.619824624385043e-8, -4.639413853091412e-12)
        s14 (poly/mevalpoly mu -2.5330554576498356e19, 3.0761765051109855e19, -5.827690025164462e18, 4.1047854971841363e17, -1.4277576770556304e16, 2.819724813029272e14, -3.4214400566025547e12, 2.670056925037507e10, -1.3717353508462998e8, 465890.6725849346, -1028.1881576107903, 1.4039031226739573, -0.0010631539871280848, 3.4554354377824836e-7, -2.061961712485072e-11)
        s13 (poly/mevalpoly mu 1.4409993748135704e17, -1.747635582560889e17, 3.286944341694018e16, -2.2818860422499875e15, 7.75248528894739e13, -1.4783938404512664e12, 1.7064572063331423e10, -1.2415577730993801e8, 578046.6774168271, -1705.2186748496474, 3.0504712266430722, -0.0029777336087200013, 1.2309086638850886e-6, -9.237588471933122e-11)
        s12 (poly/mevalpoly mu -9.588236853265692e14, 1.161016591203577e15, -2.164822377227822e14, 1.4768015151140877e13, -4.8748767898914606e11, 8.900147704533855e9, -9.641151011038888e7, 640028.5410983711, -2606.385824929625, 6.277422845836725, -0.008087900592073538, 4.342405588886322e-6, -4.176996526439325e-10)
        s11 (poly/mevalpoly mu 7.562132599676877e12, -9.139342212003133e12, 1.686263984004851e12, -1.1257747938365613e11, 3.584384568770644e9, -6.188986771037158e7, 616601.7257690447, -3609.808395761912, 12.103919447801218, -0.021179858537379914, 1.5144118606258417e-5, -1.909484126372263e-9)        
        s10 (poly/mevalpoly mu -7.181611223268327e10, 8.659091218989731e10, -1.576853447634526e10, 1.0243675365394899e9, -3.1127612822052997e7, 499001.19046756154, -4427.281508191307, 21.557856538012857, -0.053067848184582544, 5.209072696743533e-5, -8.843926480039954e-9)
        s9 (poly/mevalpoly mu 8.369564497799761e8, -1.0061560971482614e9, 1.80175839935865e8, -1.1293844119552301e7, 322272.3393258663, -4655.44006793818, 34.77845393284224, -0.12591543863527477, 0.0001761710154823959, -4.1618477553129196e-8)
        s8 (poly/mevalpoly mu -1.225106763286615e7, 1.4670878180751745e7, -2.5692206274838205e6, 153362.43350451812, -4001.4928083932027, 49.41718461737037, -0.2788656409829855, 0.0005833245813846588, -1.9976869225502014e-7)
        s7 (poly/mevalpoly mu 231884.6359563172, -276225.9131319225, 46889.68168082833, -2607.1347218453884, 59.29277476668358, -0.5644365847110748, 0.0018794238567352295, -9.834766387939453e-7)
        s6 (poly/mevalpoly mu -5892.0457146167755, 6965.548173427582, -1128.6143367290497, 56.11658954620361, -1.0105445384979248, 0.005837917327880859, -5.0067901611328125e-6)
        s5 (poly/mevalpoly mu 211.27614974975586, -246.8455924987793, 37.067405700683594, -1.5151596069335938, 0.017223358154296875, -2.6702880859375e-5)
        s4 (poly/mevalpoly mu -11.466461181640625, 13.1358642578125, -1.71624755859375, 0.0469970703125, -0.000152587890625)
        s3 (poly/mevalpoly mu 1.0478515625, -1.1591796875, 0.1123046875, -0.0009765625)
        s2 (poly/mevalpoly mu -0.1953125, 0.203125, -0.0078125)
        s1 (poly/mevalpoly mu 0.125, -0.125)
        aap (split-phase-poly xinv (double-array [1.0, s1, s2, s3, s4, s5, s6, s7, s8, s9, s10
                                                  s11, s12, s13, s14, s15, s16, s17, s18, s19, s20
                                                  s21, s22, s23, s24, s25, s26, s27, s28, s29, s30]))]
    (v/emult aap (Vec2. x 1.0))))

(def bessel-y-chebyshev-weights
  [(double-array '(-0.030017698430846347, 0.04841599615543705, -0.06170112748210926, 0.010992867154582408, -0.03732304718455494, -0.06704282983696781, 0.05527054188949959, 0.023931299324571078, -0.01660982874247215, -0.0038257184915278875, 0.0024486563159734553, 0.00036760669126412974, -0.00022289024621774598, -2.400043085835714e-5, 1.3967165370379739e-5, 1.1481395619070928e-6, -6.449732666954525e-7, -4.1980803435719734e-8, 2.2894937763100794e-8, 1.228873366161081e-9, -6.497632451926297e-10, -2.8492149116234243e-11, 1.4818408405099196e-11, 6.100556567848311e-13, -2.9867774919978274e-13, -6.185528280054444e-15, 3.7311423791866865e-15, 4.123685520036296e-16, -1.3481279584734043e-16)),
   (double-array '(0.04160742192322324, 0.034668235600932615, 0.11174468340078995, -0.04099618095057443, 0.1254503646682523, -0.13898784797365799, -0.09733131946545694, 0.07149834579877713, 0.023375835755720973, -0.014572641168112577, -0.002981779009597293, 0.0016841968136507765, 0.0002422436246301404, -0.000127737004832643, -1.3814377122225202e-5, 6.9111332786206e-6, 5.866764647818481e-7, -2.8135569182003675e-7, -1.9403344167501224e-8, 8.972400084810467e-9, 5.113342130666102e-10, -2.2951010536509386e-10, -1.1304711135449809e-11, 4.905817815473565e-12, 1.894873147278986e-13, -8.217632923341561e-14, -4.218847493575595e-15, 1.6772297836301473e-15, -7.137148015447435e-17)),
   (double-array '(0.022713476918759194, -0.03501714644493984, 0.04900399934159489, -0.004572920361822816, 0.03414506477673644, 0.0583364038403724, -0.04624123224424829, -0.02357706101501925, 0.01424650161095809, 0.004131847507694573, -0.0021768978247202514, -0.0004222406947531938, 0.00020419469487529897, 2.882041410641021e-5, -1.3118879606205788e-5, -1.419547866831786e-6, 6.171197932015276e-7, 5.3215526542899864e-8, -2.2283397359466684e-8, -1.5689346295358737e-9, 6.370162334131838e-10, 3.785641839686755e-11, -1.4927226121841386e-11, -7.280852508197353e-13, 2.822365357297534e-13, 1.4096858601066386e-14, -5.194257722353411e-15, -1.784287003861859e-17, 1.784287003861859e-17)),
   (double-array '(-0.005482743163667869, -0.0026156864489958096, -0.013446106107341984, 0.003959772572751581, -0.013050739904566335, 0.012785648905104951, 0.012302789524159771, -0.007125881422379595, -0.0032178006311644504, 0.0015436443743600617, 0.0004291283948110706, -0.00018601906106350898, -3.5730026469563336e-5, 1.4521000123576475e-5, 2.0641877773886624e-6, -8.0134383976041e-7, -8.833134689166324e-8, 3.3105750509021536e-8, 2.9248427531622776e-9, -1.0652720751644758e-9, -7.761774457286969e-11, 2.758318276166857e-11, 1.6716350526024826e-12, -5.818832518996889e-13, -3.163144349623995e-14, 1.0744381574921492e-14, 3.598312124454748e-16, -1.2688263138573217e-16, -1.4373423086664974e-17)),
   (double-array '(-0.000821102524680051, 0.001369166415512963, -0.0017176445978307614, -0.00038167202153855176, -0.0010693704116036165, -0.0034162564467415902, 0.0019105668931819009, 0.0015608651654028757, -0.000662690872417737, -0.00028896692147194577, 0.00010910809251752316, 3.0258725355010946e-5, -1.0717669999955374e-5, -2.0862028639579696e-6, 7.091627814767861e-7, 1.0306000805131983e-7, -3.4017752731151496e-8, -3.856023535105163e-9, 1.2437339960318393e-9, 1.1358064900927889e-10, -3.594555586962962e-11, -2.7016821319796643e-12, 8.405911755350802e-13, 5.4115690103654197e-14, -1.660105584745862e-14, -8.187894806610529e-16, 2.462687791788503e-16, 2.0816681711721685e-17, -6.443258625056712e-18)),
   (double-array '(0.0002240588217758784, -2.139628782819116e-5, 0.0004963922147532327, -0.00011956525731319659, 0.00038470659301256783, -0.00022845214105494214, -0.0004692565509103452, 0.0001676797752522885, 0.0001335452722506947, -4.206397012022229e-5, -1.847619067631446e-5, 5.51981330361833e-6, 1.5583457267047952e-6, -4.525989246540121e-7, -9.035200300660385e-8, 2.5768522858261958e-8, 3.852458639733455e-9, -1.084135171755376e-9, -1.2724325563785672e-10, 3.5416192958546876e-11, 3.3285729083723912e-12, -9.174161771285542e-13, -7.329222174604474e-14, 2.0034423068009816e-14, 1.2002737593513098e-15, -3.2511254198458856e-16, -2.7507757976203656e-17, 6.753030674338285e-18, -4.336808689942018e-19)),
   (double-array '(4.6589086507212064e-6, -1.419986317236695e-5, 7.726102539451062e-6, 2.5019808838475857e-5, -5.353457546572625e-7, 7.876549687169565e-5, -1.996847202131808e-5, -4.169957016198162e-5, 9.924101147153856e-6, 8.2415996393497e-6, -1.9586527669761305e-6, -8.849155270353938e-7, 2.1145772481562803e-7, 6.146714709097454e-8, -1.4757833806872662e-8, -3.0347995540046756e-9, 7.310991241996968e-10, 1.1291045426976899e-10, -2.724231275323183e-11, -3.306873481836321e-12, 7.984081946667783e-13, 7.777996686699921e-14, -1.874952442498305e-14, -1.5555232495803793e-15, 3.750913580232083e-16, 2.4244503044553087e-17, -6.63977026881971e-18, -8.441288342922856e-19, 3.7947076036992655e-19)),
   (double-array '(-3.837876843131191e-6, 1.5306249581045373e-6, -7.808225833446716e-6, 8.447928863780004e-7, -4.570697567070479e-6, -4.1982552639485557e-7, 8.226238919809268e-6, -9.86994172463081e-7, -2.6319604894434756e-6, 4.2742958932228587e-7, 3.841104494715522e-7, -7.011130365317191e-8, -3.295984466784781e-8, 6.37554898520718e-9, 1.9212842748116095e-9, -3.851870296260733e-10, -8.157917257527438e-11, 1.6727904537992146e-11, 2.682049173124509e-12, -5.589600448593574e-13, -6.949400331178381e-14, 1.4635958089930247e-14, 1.5059865847402264e-15, -3.216721361622551e-16, -2.5000540452330927e-17, 5.001971562950145e-18, -1.961002277600456e-18, 9.647251501576167e-19, -6.969871108835386e-19)),
   (double-array '(1.576952254652365e-7, -9.540862698044819e-8, 3.3917615684812697e-7, -4.813870672453743e-7, 2.6408257938646016e-7, -8.568425702351827e-7, -1.2996848292204935e-7, 5.684699761362054e-7, -2.7544597243559516e-8, -1.247732596836196e-7, 1.4090830850669181e-8, 1.398591135642683e-8, -1.991545843477106e-9, -9.83479157367889e-10, 1.5579619154787597e-10, 4.8812239436949277e-11, -8.259978073280919e-12, -1.7977167324163993e-12, 3.156743262349288e-13, 5.3090323232950275e-14, -9.680314873025469e-15, -1.183751262421577e-15, 2.1703251434347915e-16, 2.5924059718442872e-17, -5.649951767599685e-18, 8.287120783728401e-19, -1.0219906276273448e-18, 2.5227968798668708e-18, -1.4714172340874703e-18)),
   (double-array '(2.8189276354316185e-8, -8.524800253702236e-9, 5.163074759458915e-8, 1.7587774571193398e-8, 1.5532689836449176e-8, 4.569379005731088e-8, -7.364565287663236e-8, -1.3794427678087878e-8, 2.8705869772299055e-8, -2.180238319299374e-10, -4.6124233142017096e-9, 3.6839126250828555e-10, 4.0853238049027397e-10, -4.58540375117209e-11, -2.4105895138400274e-11, 3.1690572156845862e-12, 1.025021315725097e-12, -1.4748687505667043e-13, -3.336472532024983e-14, 5.062095043025821e-15, 8.815484952439681e-16, -1.390034081631452e-16, -1.9199758208465145e-17, 2.8599694051079214e-18, 4.412739143428028e-19, -1.4432387356775567e-18, -2.3578466668818103e-18, -8.403587813975867e-19, -2.0135183203302226e-19)),
   (double-array '(-2.7963515253359716e-9, 2.6599405451754447e-9, -5.363225685883765e-9, 3.919112883175106e-9, -2.6445728931356353e-9, 3.986389181091895e-9, 4.369788563535614e-9, -4.202468942903132e-9, -8.620553821731379e-10, 1.1190715605385904e-9, 1.6458619897562414e-11, -1.364556955782257e-10, 7.770660894600997e-12, 9.797565423719047e-12, -8.709272762295716e-13, -4.951317701122077e-13, 5.5017193500686674e-14, 1.7973668773098524e-14, -2.186053134804191e-15, -5.417976610912264e-16, 7.265668497060572e-17, 1.0657145676262853e-17, -1.4951583811856726e-18, 1.001165055851074e-19, -4.215593642978497e-19, -8.996650316071422e-19, 1.1892060156355519e-18, -5.459522028374813e-19, -9.29316147844718e-19)),
   (double-array '(-2.869285397772589e-11, -1.425452662669203e-10, -1.4765859863811186e-11, -3.584266980792905e-10, 1.2630165211284162e-10, -4.813898813698495e-10, 2.9823341363420924e-10, 2.736833504070133e-10, -1.789940159530945e-10, -3.709803166716433e-11, 3.4844491459805023e-11, 8.72681373200755e-13, -3.2915591662608672e-12, 1.3002232883748757e-13, 1.9753949839973547e-13, -1.4075604115824882e-14, -8.601511614454014e-15, 8.194975088384275e-16, 2.6866096515073e-16, -2.6539953077529138e-17, -6.38116169641218e-18, -1.8536358293679684e-18, 4.749772369976817e-19, -6.718043393812506e-19, 1.608438964002761e-19, 5.165497356149306e-20, 2.605394963005647e-19, 1.0581028516295723e-19, 3.601100072898283e-19)),
   (double-array '(1.7263987678277334e-11, -1.3813366927208365e-11, 2.934247457063122e-11, -7.979165795689896e-12, 4.195660738851445e-12, 3.4756966192072763e-12, -3.8868044414248185e-11, 1.4360156231687025e-11, 1.2343845937660092e-11, -6.048594850472885e-12, -1.1764419064241353e-12, 8.840620586141411e-13, 2.5310500957954275e-14, -6.57823088425572e-14, 1.5812120648028614e-15, 3.437403053731452e-15, -2.0773399722137888e-16, -1.2502298901942747e-16, 1.1188335458126739e-17, 1.318770475529763e-18, 1.826317128659254e-18, -1.371135591032674e-18, -7.556266590585322e-19, -1.3319774595248458e-19, -6.907940207156658e-20, 1.774511101624311e-18, -4.537139020388408e-20, 4.949220345202049e-19, -1.1868141638100254e-18)),
   (double-array '(-8.498193197325338e-13, 1.953201750327066e-12, -1.6836406424518253e-12, 2.5267159459094905e-12, -1.1786713409725903e-12, 2.057320315641163e-12, 2.5591575645466365e-13, -2.0678475108903006e-12, 5.290174964157149e-13, 4.2270269001999225e-13, -1.680245826111033e-13, -2.838265511092047e-14, 1.8427577852297096e-14, 5.755842428415847e-16, -1.1124766162836294e-15, 1.08483878138819e-17, 5.233365672134661e-17, -4.1398608828316095e-18, -1.5252680555707005e-18, 1.5518391236648752e-21, -1.914160606977295e-18, -1.1560851262788287e-18, 8.614032803477796e-19, -2.4613082189127555e-18, 2.3715585034500812e-18, -7.375149035246488e-19, 1.5108762141152207e-18, -1.5506658311469656e-18, 8.17023779980148e-19)),
   (double-array '(-2.5827236864549892e-14, -3.4718790561504414e-14, -2.2011098530056326e-14, -9.110104031477491e-14, 7.290507373316453e-14, -1.234623611877002e-13, 1.6275850841628636e-13, 1.798444625683851e-14, -8.154482889879963e-14, 1.634453336575052e-14, 1.129456516347965e-14, -3.897439775077071e-15, -5.305238116730616e-16, 3.1481637373066765e-16, 1.2762608430395386e-17, -1.6640493419786607e-17, 4.907490184278107e-19, 3.285996490596493e-19, -1.6693969668826682e-18, -6.276996493483055e-19, 2.2721860225308528e-18, -1.0636166658032876e-18, 5.4043840077706323e-20, -6.733003923003965e-19, 8.654687466171472e-19, -1.6353891516146702e-18, -1.1401886941945345e-18, 1.806451000706121e-18, 1.0454806663253078e-18)),
   (double-array '(4.9388506677207545e-15, -8.11264980108463e-15, 8.15564695267678e-15, -6.6179014247901285e-15, 1.2560443562777774e-15, -1.1626485837362674e-15, -8.428647433288211e-15, 8.040878625052107e-15, 7.604076442794149e-16, -2.509940602859596e-15, 4.433411320740124e-16, 2.4051574424535735e-16, -7.97263854654364e-17, -8.468775456115063e-18, 3.0977204928150507e-18, -2.763609694292828e-18, -4.528057493615123e-18, 8.090667992714159e-18, -5.7670273029389466e-18, 6.935237082901993e-18, 2.3956539089710175e-18, 3.5297683311975285e-18, -2.6680862870119677e-19, -1.124331067925833e-18, -3.969891564639305e-18, 3.4988194829519685e-18, 8.636411459915624e-19, -1.3895984430563314e-18, 1.7657006809049643e-18)),
   (double-array '(-2.058125495426769e-16, 6.08114692374987e-16, -4.430737266654462e-16, 7.070791826861885e-16, -4.496672312473068e-16, 4.992680927678948e-16, -2.0005398723221536e-16, -4.085152288266179e-16, 2.963046065367326e-16, 1.9081397648228198e-17, -6.065778883208153e-17, 1.2267254817829564e-17, 2.5241246780253446e-18, 5.364001077188067e-19, -1.1151793774136653e-18, -2.647497194530543e-19, -9.99558548618923e-19, 1.9545043306813755e-18, 1.0295729550672243e-18, 5.850363951633064e-19, 2.039938658283451e-19, -1.622964422184252e-18, -2.1240860691038883e-18, 2.0835979355758296e-18, -1.5423943263121515e-19, -5.576729517866199e-19, -3.7259501367076117e-20, -1.8905808347578018e-19, -1.4869058365515489e-18))])

;;

(def abi-1 (double-array '(0.41739858494452137, -0.06495162894671667, 0.01346635223349219, -0.0034498974696635805, 0.0009181061897552018, -0.0002204042930330767, 3.7976551651085086e-5, -1.3803071542564344e-8, -3.398553997511632e-6, 1.6746478140542785e-6, -5.176623166786395e-7, 1.0972618211669806e-7, -1.2473514618542336e-8, -1.4217711195129223e-9, 1.146316775379731e-9, -3.175280659616182e-10, 5.034723214458251e-11, -2.0319170143404405e-12, -1.4129596145607766e-12, 4.802366525142762e-13, -8.22183850663261e-14, 5.914052924936658e-15, 1.0066800053200003e-15, -3.986773142283109e-16, 6.067332401977342e-17)))

(def abi-2 (double-array '(0.3388998275567636, -0.022239535106198405, 0.001847776356735787, -0.00018642620881156231, 2.0614467307054072e-5, -2.413394583210988e-6, 2.9461272726896696e-7, -3.722819620752169e-8, 4.854723628008316e-9, -6.531662239526833e-10, 9.08219420289064e-11, -1.3082391956275184e-11, 1.953106497996366e-12, -3.007137783175837e-13, 4.709451540463225e-14, -7.327307135432242e-15, 1.0856338884553918e-15, -1.4783310001069625e-16)))

(def abip-1 (double-array '(0.7658021754601785, 0.1218178967795153, -0.016874295932807537, 0.0038022995437998084, -0.0009201686158669025, 0.00018122863592354458, -1.089109598429424e-5, -1.2116775872566089e-5, 7.326775924782161e-6, -2.5316832797383597e-6, 5.814914990161355e-7, -6.775973510473317e-8, -1.0915897862521443e-8, 8.299026220548351e-9, -2.4078260629342257e-9,  4.037468537238015e-10, -1.7933493682633233e-11, -1.216289673737319e-11, 4.331771122129753e-12, -7.552310555170731e-13, 4.643024726419001e-14, 1.4365062751453502e-14, -5.2995707246535766e-15, 8.910837004634583e-16, -6.308304600047433e-17)))

(def abip-2 (double-array '(0.9393816481647195, 0.0621197721009847, -0.0031821373579424077, 0.0002596583618312277, -2.5490889161776127e-5, 2.7833396581896174e-6, -3.270630123629566e-7, 4.070105276526434e-8, -5.322537351397486e-9, 7.293189148846469e-10, -1.0468150599102346e-10, 1.5727589039481186e-11, -2.460184674017499e-12, 3.9506685986702333e-13, -6.352712208186561e-14, 9.868492209739352e-15, -1.3994310387891848e-15, 1.6674714479000505e-16)))

(def hg-2F1-poly (double-array [1.0087391788544393911192, 1.220474262857857637288e-01, 8.7957928919918696061703e-03, 6.9050958578444820505037e-04, 5.7037120050065804396306e-05, 4.8731405131379353370205e-06, 4.2648797509486828820613e-07, 3.800372208946157617901e-08, 3.434168059359993493634e-09, 3.1381484326392473547608e-10, 2.8939845618385022798906e-11, 2.6892186934806386106143e-12, 2.5150879096374730760324e-13, 2.3652490233687788117887e-14, 2.2349973917002118259929e-15, 2.120769988408948118084e-16]))
