(ns tree-form.events
  (:require [re-frame.core :as rf]
            [tree-form.utils :as utils]))


(rf/reg-event-db
  :tree-form/assoc-to-path
  [rf/trim-v]
  (fn [db [path value]]
    (assoc-in db path value)))


(rf/reg-fx
  :tree-form/init!
  (fn [[callback form-fields]]
    (callback form-fields)))


(rf/reg-event-fx
  :tree-form/create-new-form
  [rf/trim-v]
  (fn [{:keys [db]} [form]]
    (let [form-id     (:id form)
          init-fn     (or (:init form) utils/nil-fn)
          form-fields (utils/prepare-form-fields (:fields form))]

      {:tree-form/init!
       [init-fn form-fields]

       :db
       (assoc-in db [:tree-form/forms form-id] {:fields form-fields
                                                :errors {}})})))


(rf/reg-event-db
  :tree-form/update-form-field
  [rf/trim-v]
  (fn [db [form-id field-id value]]
    (-> db
        (assoc-in [:tree-form/forms form-id :fields field-id] value)
        (update-in [:tree-form/forms form-id :errors] dissoc field-id))))


(rf/reg-event-db
  :tree-form/set-form-field-error
  [rf/trim-v]
  (fn [db [form-id field-id error]]
    (assoc-in db [:tree-form/forms form-id :errors field-id] error)))


(rf/reg-event-db
  :tree-form/set-form-errors
  [rf/trim-v]
  (fn [db [form-id errors]]
    (assoc-in db [:tree-form/forms form-id :errors] errors)))


(rf/reg-fx
  :tree-form/submit!
  (fn [[callback form-fields]]
    (callback form-fields)))


(rf/reg-event-fx
  :tree-form/submit
  [rf/trim-v]
  (fn [{:keys [db]} [form]]
    (let [{:keys [id on-submit validate fields]
           :or   {on-submit utils/nil-fn}} form
          form-fields            (get-in db [:tree-form/forms id :fields])
          fields-with-validators (utils/extract-fields-with-validators fields)
          form-errors            (utils/validate-form validate fields-with-validators form-fields)]

      (if-not (empty? form-errors)
        {:dispatch [:tree-form/set-form-errors id form-errors]}
        {:tree-form/submit! [on-submit form-fields]}))))


(rf/reg-fx
  :tree-form/clean-up!
  (fn [[callback form-fields]]
    (callback form-fields)))


(rf/reg-event-fx
  :tree-form/clean-up
  [rf/trim-v]
  (fn [{:keys [db]} [form]]
    (let [{:keys [id clean-up clear-state]
           :or   {clean-up    utils/nil-fn
                  clear-state true}} form
          form-fields (get-in db [:tree-form/forms id :fields])]

      {:tree-form/clean-up!
       [clean-up form-fields]

       :db
       (if clear-state
         (update-in db [:tree-form/forms] dissoc id)
         db)})))
