(ns tree-form.utils
  (:require [reagent.interop :refer-macros [$]]
            [re-frame.core :refer [dispatch subscribe]]))


(def nil-fn
  (constantly nil))


(defn prepare-form-fields
  "Prepare form fields to save inside DB
  Result is:

  {:field-name default-value
   ...}
  "
  [fields]
  (reduce-kv
    (fn [fields id field]
      (assoc fields id (:default field)))
    {}
    fields))


(defn event->value
  "Extract value from DOM event"
  [event]
  ($ event :target.value))


(defn create-fields
  "Prepare form fields to use inside UI components
  Result is:

  {:field-name {:action fn
                :value field-value}
   ...}
  "
  [raw-fields {:keys [id fields]}]
  (reduce-kv
    (fn [fields field-id field]
      (assoc
        fields
        field-id
        {:value  (get raw-fields field-id)
         :action (fn [& args]
                   (let [process            (or (:process field)
                                                event->value)
                         value              (apply process args)
                         validate           (:validate field)
                         validate-on-action (:validate-on-action field)
                         error              (when validate-on-action
                                              (validate value))]

                     (dispatch [:tree-form/update-form-field id field-id value])

                     (when error
                       (dispatch [:tree-form/set-form-field-error id field-id error]))))}))
    {}
    fields))


(defn create-subscriptions
  "Create subscriptions from config map {:subscription-name subscription-path}"
  [config]
  (reduce-kv
    (fn [subs subs-name subs-path]
      (assoc subs subs-name (subscribe [:tree-form/to-path subs-path])))
    {}
    config))


(defn deref-subscriptions
  "Deref all subscriptions in subscriptions map"
  [subs-map]
  (->> subs-map
       (map #(vector (first %) (deref (second %))))
       (into {})))


(defn extract-fields-with-validators [fields]
  (->> fields
       (filter #(:validate (second %)))
       (into {})))


(defn validate-form [validate fields-with-validators form-fields]
  (let [form-errors   (if validate
                        (validate form-fields)
                        {})
        fields-errors (->> fields-with-validators
                           (reduce-kv
                             (fn [errors field-id field-with-validator]
                               (let [value     (get form-fields field-id)
                                     validator (:validate field-with-validator)]
                                 (assoc errors field-id (validator value))))
                             {})
                           (filter #(not (nil? (second %))))
                           (into {}))]

    (merge form-errors fields-errors)))
