(ns tree-form.forms
  (:require [reagent.core :as r]
            [reagent.interop :refer [$]]
            [re-frame.core :refer [dispatch]]
            [clojure.string :refer [lower-case]]
            [tree-form.core :refer [form]]))


(defn basic-form-example []
  [form {:id     :basic-form
         :fields {:first-name nil
                  :last-name  nil}}
   (fn [{:keys [fields on-submit]}]
     [:form {:on-submit on-submit}
      [:div
       [:input {:type        "text"
                :placeholder "First name"
                :on-change   (get-in fields [:first-name :action])
                :value       (get-in fields [:first-name :value])}]

       [:input {:style       {:margin-left "20px"}
                :type        "text"
                :placeholder "Last name"
                :on-change   (get-in fields [:last-name :action])
                :value       (get-in fields [:last-name :value])}]]

      [:div {:style {:margin-top "20px"}}
       (str "Full name: "
            (get-in fields [:first-name :value])
            " "
            (get-in fields [:last-name :value]))]])])


(defn less-than-4-chars [{:keys [first-name]}]
  (when (> (count first-name) 4)
    {:first-name "Invalid"}))

(defn validate-form-example []
  [form {:id       :validate-form
         :validate less-than-4-chars
         :fields   {:first-name nil}}
   (fn [{:keys [fields errors on-submit]}]
     (let [error (:first-name errors)]
       [:form {:on-submit on-submit}
        [:div
         [:input {:type        "text"
                  :placeholder "First name"
                  :on-change   (get-in fields [:first-name :action])
                  :value       (get-in fields [:first-name :value])}]]

        [:div {:style {:margin-top "20px"}}
         (str "Max length - 4 chars. "
              "Field is "
              (if error error "Valid"))]]))])


(defn submit-first-name [fields]
  (js/alert (str "Your first name is " (:first-name fields))))

(defn submit-form-example []
  [form {:id        :submit-form
         :on-submit submit-first-name
         :fields    {:first-name nil}}
   (fn [{:keys [fields on-submit]}]
     [:form {:on-submit on-submit}
      [:div
       [:input {:type        "text"
                :placeholder "First name"
                :on-change   (get-in fields [:first-name :action])
                :value       (get-in fields [:first-name :value])}]]])])


(defn enter-your-name-cta []
  (js/alert "Enter your name, pls"))

(defn init-form-example []
  [form {:id     :init-form
         :init   enter-your-name-cta
         :fields {:first-name nil}}
   (fn [{:keys [fields on-submit]}]
     [:form {:on-submit on-submit}
      [:div
       [:input {:type        "text"
                :placeholder "First name"
                :on-change   (get-in fields [:first-name :action])
                :value       (get-in fields [:first-name :value])}]]])])


(defn say-good-bye []
  (js/alert "See you next time"))

(defn clean-up-form-example []
  (let [form-rendered (r/atom true)]
    (fn []
      [:div
       [:span "Toggle form"]
       [:input {:type      "checkbox"
                :checked   @form-rendered
                :on-change #(swap! form-rendered not)}]

       (when @form-rendered
         [form {:id       :clean-up-form
                :clean-up say-good-bye
                :fields   {:first-name nil}}
          (fn [{:keys [fields on-submit]}]
            [:form {:on-submit on-submit}
             [:div
              [:input {:type        "text"
                       :placeholder "First name"
                       :on-change   (get-in fields [:first-name :action])
                       :value       (get-in fields [:first-name :value])}]]])])])))


(defn do-not-clear-state-form-example []
  (let [form-rendered (r/atom true)]
    (fn []
      [:div
       [:span "Toggle form"]
       [:input {:type      "checkbox"
                :checked   @form-rendered
                :on-change #(swap! form-rendered not)}]

       (when @form-rendered
         [form {:id          :do-not-clear-state-form
                :clear-state false
                :fields      {:first-name nil}}
          (fn [{:keys [fields on-submit]}]
            [:form {:on-submit on-submit}
             [:div
              [:input {:type        "text"
                       :placeholder "First name"
                       :on-change   (get-in fields [:first-name :action])
                       :value       (get-in fields [:first-name :value])}]]])])])))


(defn default-values-form-example []
  [form {:id     :default-values-form
         :fields {:first-name {:default "Martin Luter King"}}}
   (fn [{:keys [fields on-submit]}]
     [:form {:on-submit on-submit}
      [:div
       [:input {:type        "text"
                :placeholder "First name"
                :on-change   (get-in fields [:first-name :action])
                :value       (get-in fields [:first-name :value])}]]])])


(defn process-field-form-example []
  [form {:id     :process-field-form
         :fields {:first-name {:default "Martin Luter King"
                               :process (fn [event]
                                          (lower-case ($ event :target.value)))}}}
   (fn [{:keys [fields on-submit]}]
     [:form {:on-submit on-submit}
      [:div
       [:input {:type        "text"
                :placeholder "First name"
                :on-change   (get-in fields [:first-name :action])
                :value       (get-in fields [:first-name :value])}]]])])


(defn less-than-4-chars-field [first-name]
  (when (> (count first-name) 4)
    "Invalid"))

(defn validate-individual-fields-form-example []
  [form {:id     :validate-individual-fields-form
         :fields {:first-name {:default  "Martin Luter King"
                               :validate less-than-4-chars-field
                               :process  (fn [event]
                                           (lower-case ($ event :target.value)))}}}
   (fn [{:keys [fields on-submit]}]
     [:form {:on-submit on-submit}
      [:div
       [:input {:type        "text"
                :placeholder "First name"
                :on-change   (get-in fields [:first-name :action])
                :value       (get-in fields [:first-name :value])}]]])])


(defn validate-on-demand-form-example []
  [form {:id     :validate-on-demand-form
         :fields {:first-name {:default            "Martin Luter King"
                               :validate           less-than-4-chars-field
                               :validate-on-action true}}}
   (fn [{:keys [fields errors on-submit]}]
     [:form {:on-submit on-submit}
      [:div
       [:input {:type        "text"
                :placeholder "First name"
                :on-change   (get-in fields [:first-name :action])
                :value       (get-in fields [:first-name :value])}]

       (when (:first-name errors)
         [:span {:style {:color "red"}}
          "invalid"])]])])


(defn additional-values-from-state-form-example []
  [form {:id       :additional-values-form
         :init     #(dispatch [:tree-form/assoc-to-path [:something-is-loading] true])
         :includes {:loading? [:something-is-loading]}
         :fields   {:first-name nil}}
   (fn [{:keys [fields includes on-submit]}]
     (let [loading? (:loading? includes)]
       [:div
        [:span "Toggle loading"]
        [:input {:type      "checkbox"
                 :checked   loading?
                 :on-change #(dispatch [:tree-form/assoc-to-path [:something-is-loading] (not loading?)])}]

        [:form {:on-submit on-submit}
         [:div
          [:input {:type        "text"
                   :placeholder "First name"
                   :on-change   (get-in fields [:first-name :action])
                   :value       (get-in fields [:first-name :value])}]

          (when loading?
            [:span {:style {:color "blue"}}
             "Loading"])]]]))])
