(ns tree-form.components
  (:require [reagent.core :refer [create-class props children]]
            [reagent.interop :refer-macros [$]]
            [re-frame.core :refer [dispatch subscribe]]
            [tree-form.utils :as utils]))


(defn form
  "Basic usage:

  [form {:id          :my-super-form
         :validate    my-validate-fn
         :on-submit   my-submit-fn
         :init        my-init-fn
         :clean-up    my-clean-up-fn
         :clear-state false
         :fields      {:my-something nil
                       :my-name      {:default \"Albert\"}
                       :my-age       {:default 10
                                      :validate validate-field-fn
                                      :validate-on-action true
                                      :process (comp js/parseInt #($ % :target.value))}}
         :includes    {:my-custon-prop-name [:path :to :prop :in :db]}}

    (fn [{:keys [fields errors includes on-submit]}]
      [:form {:on-submit on-submit}
       [:input {:on-change (-> fields :my-name :action)
                :value (-> fields :my-name :value)}]])]
  "
  [{:keys [includes] :as form-props}]
  (let [form-fields   (subscribe [:tree-form/get-form-fields form-props])
        form-errors   (subscribe [:tree-form/get-form-errors form-props])
        form-includes (utils/create-subscriptions includes)]
    (create-class
      {:display-name
       "tree-form"

       :component-will-mount
       (fn []
         (dispatch [:tree-form/create-new-form form-props]))

       :component-will-unmount
       (fn [this]
         (dispatch [:tree-form/clean-up (props this)]))

       :render
       (fn [this]
         (let [fields    @form-fields
               errors    @form-errors
               ui-fn     (-> this children first)
               submit-fn (fn [event]
                           ($ event preventDefault)
                           (dispatch [:tree-form/submit (props this)]))
               includes  (utils/deref-subscriptions form-includes)]
           (ui-fn
             {:fields    fields
              :errors    errors
              :includes  includes
              :on-submit submit-fn})))})))
