(ns clairvoyant.core
  (:require
   [clojure.walk :as walk]
   [clojure.pprint :as pprint]
   [cljs.analyzer :as analyzer]))


;; ---------------------------------------------------------------------
;; Utilities

(defmacro
  ^{:private true
    :doc     "Define one or more methods with the same fn-tail."}
  defmethods
  [multifn dispatch-vals & fn-body]
  {:pre [(vector? dispatch-vals)]}
  `(do
     ~@(for [dispatch-val dispatch-vals]
         `(defmethod ~multifn ~dispatch-val ~@fn-body))))


(defn ^:private debug-form
  "Throw an exception containing a pretty printed form. Only useful for
  debugging macros in ClojureScript."
  [form]
  (throw (Exception. (with-out-str (pprint/pprint form)))))


(defn resolve-sym
  "Attempt to return a fully resolved symbol from sym."
  [sym env]
  ;; Check if we're in ClojureScript
  (if (boolean (:ns env))
    (if-let [resolved (:name (analyzer/resolve-var env sym))]
      resolved
      sym)
    (if-let [resolved (resolve sym)]
      (let [sym     (.sym resolved)
            ns-name (.. resolved ns name)]
        (symbol (str ns-name) (str sym)))
      sym)))


;; ---------------------------------------------------------------------
;; API

(def ^:dynamic ^:private *opts*)
(def ^:dynamic ^:private *tracer*)
(def ^:dynamic ^:private *excluded-ops*)
(def ^:dynamic ^:private *env*)

(def ^:dynamic ^:private *nested* false)

;; This isn't actually used, it's just to help with resolving
;; the cljs var with the same name in this namespaces's macros.
(defonce *threading-step-stack nil)

(defonce threading-macros #{'-> '->> 'as-> 'cond-> 'cond->> 'some-> 'some->>})
(defonce conditionals #{'when 'when-not 'when-let 'when-some
                        'if 'if-not 'if-let 'if-some})


(defmulti trace-form
  "Return the trace form for a single form."
  (fn [form]
    (if (and (seq? form)
             (symbol? (first form)))
      (if (contains? *excluded-ops* (first form))
        ::default
        (let [[op & rest] form]
          op))
      form))
  :default ::default)

(defmethod trace-form ::default
  [form]
  (if (and (seq? form)
           (not (contains? *excluded-ops* (first form))))
    (cons (first form)
          (doall (for [x (rest form)]
                   (trace-form x))))
    form))

; (defmacro dev?
;   "True if assertions are enabled."
;   []
;   (if *assert* true false))

(def dev?
  "True if assertions are enabled."
  (vary-meta 'js/clairvoyant.core.devmode assoc :tag 'boolean))

(defmacro trace-forms
  "Recursively trace one or more forms.

  :tracer - custom tracer.

  :enabled? - boolean, override devmode flags and force tracing on/off.

  :exclude - set of op symbols to exclude from tracing, e.g. #{'fn 'fn*}. Also
  supports the `:unnamed-fn` keyword.

  :bindings - a vector of dynamic bindings to wrap a trace function in."
  [{:keys [tracer enabled? exclude] :as opts} & forms]
  (binding [*tracer*       (or tracer
                               (:clairvoyant/tracer (meta *ns*))
                               'clairvoyant.core/default-tracer)
            *opts*         opts
            *env*          &env
            *excluded-ops* (if exclude
                             (->> exclude
                                  (map #(if (seq? %)
                                          (symbol (last %))
                                          %))
                                  (set))
                             #{})]
    (let [traced-forms (doall (for [form forms]
                                (trace-form form)))]
      (cond (nil? enabled?)
            `(if ~dev?
               (do ~@traced-forms)
               (do ~@forms))

            (= enabled? false)
            `(do ~@forms)

            (= enabled? true)
            `(do ~@traced-forms)

            :else
            `(if ~enabled?
               (do ~@traced-forms)
               (do ~@forms))))))


;; ---------------------------------------------------------------------
;; Form tracing

(defn trace-body
  "Given a form and trace data, return the form for a trace life cycle."
  ([form trace-data]
   (trace-body form trace-data nil nil))
  ([form trace-data trace-only?]
   (trace-body form trace-data trace-only? nil))
  ([form trace-data trace-only? bindings-override]
   (let [dynamic-bindings (:bindings *opts*)]
     `(binding [~@dynamic-bindings ~@bindings-override]
        (let [trace-data# ~trace-data]                ;; Cache the initial trace data.
          (when (satisfies? ITraceEnter ~*tracer*)
            (trace-enter ~*tracer* trace-data#))
          (when-not ~trace-only?
            (let [;; Creating a nullary function adds an extra call but reduces
                  ;; the amount of generated code. It also kills two birds with
                  ;; one stone; the trace error and exit steps can occur in the
                  ;; same location.
                  f# (fn []
                       (let [return# ~form]
                         (when (satisfies? ITraceExit ~*tracer*)
                           (trace-exit ~*tracer* (assoc trace-data# :exit return#)))
                         return#))]
              ;; Only setup a try/catch when the programmer expects trace error
              ;; information.
              (if (satisfies? ITraceError ~*tracer*)
                (try
                  (f#)
                  (catch js/Object e#
                    (trace-error ~*tracer* (assoc trace-data#
                                                  :error e#
                                                  :ex-data (ex-data e#)))
                    (throw e#)))
                (f#)))))))))


(defn trace-bindings
  "Return a trace form for bindings (e.g. [x 0 y 1 ...])."
  [bindings & [quote-init?]]
  (let [quote-init? (not (false? quote-init?))]
    (doall (mapcat
            (fn [[binding form]]
              (let [trace-data `{:op   :binding
                                 :form '~binding
                                 :init ~(if quote-init? `'~form form)}]
                `[~binding ~(trace-body (trace-form form) trace-data)]))
            (partition 2 bindings)))))


;;;; let

(defn trace-let
  [[op bindings & body :as form]]
  (let [trace-data `{:op   '~op
                     :form '~form}
        bindings   (trace-bindings bindings)
        body       (doall (for [form body]
                            (trace-form form)))
        form       `(~op ~(vec bindings) ~@body)]
    (trace-body form trace-data)))

(defmethods trace-form ['let* 'let]
  [form]
  (trace-let form))


;;;; fn, fn*

(defn variadic? [arglist]
  (boolean (some '#{&} arglist)))

(defn normalize-arglist
  "Removes variation from an argument list."
  [arglist]
  (vec (remove '#{&} arglist)))

(defn munge-arglist
  "Given an argument list create a new one with generated symbols."
  [arglist]
  (vec (for [arg arglist]
         (if (= '& arg)
           arg
           (gensym "a_")))))

(defn condition-map?
  "Returns true if x is a condition map, false otherwise."
  [x]
  (and (map? x)
       (or (vector? (:pre x))
           (vector? (:post x)))))

(defn condition-map-and-body
  "Given a function body, return a vetor of the condition map and
  the function body."
  [fn-body]
  (let [[x & body] fn-body]
    (if (and (seq body)
             (condition-map? x))
      [x body]
      [nil fn-body])))

(defn trace-fn-spec
  [arglist body trace-data]
  (let [[condition-map body] (condition-map-and-body body)
        body           (doall (for [form body]
                                (trace-form form)))
        munged-arglist (munge-arglist arglist)
        args           (normalize-arglist arglist)
        munged-args    (normalize-arglist munged-arglist)
        trace-data     (assoc trace-data :arglist `'~arglist)
        bindings       (mapcat vector args munged-args)
        fn-form        `(fn ~munged-arglist
                          (let [~@(trace-bindings bindings false)]
                            ((fn []
                               ~condition-map
                               ~@body))))
        form           (if (variadic? arglist)
                         `(apply ~fn-form ~@munged-args)
                         `(~fn-form ~@munged-args))]
    `(~munged-arglist ~(trace-body form trace-data))))

(defn trace-fn
  [form]
  (let [[op & body] form
        named? (symbol? (first body))]
    (if (and (not named?)
             (contains? *excluded-ops* :unnamed-fn))
      form
      (let [[sym specs] (if named?
                          [(first body) (rest body)]
                          [(gensym "fn_") body])
            trace-data `{:op         '~op
                         :form       '~form
                         :ns         '~(.-name *ns*)
                         :name       '~sym
                         :named?     ~named?
                         :anonymous? true}
            specs      (as-> specs specs
                         (if (every? list? specs)
                           specs
                           (list specs))
                         (doall (for [[arglist & body] specs]
                                  (trace-fn-spec arglist body trace-data))))]
        `(~op ~sym ~@specs)))))

(defmethods trace-form ['fn `fn 'fn* `fn*]
  [form]
  (trace-fn form))


;;;; defn

(defn trace-defn
  [[op & body :as form]]
  (let [[_ name] (macroexpand-1 form)
        [_ fn-body] (split-with (complement coll?) form)
        [_ & fn-specs] (macroexpand-1 `(fn ~@fn-body))
        trace-data `{:op         '~op
                     :form       '~form
                     :ns         '~(.-name *ns*)
                     :name       '~name
                     :named?     true
                     :anonymous? false}
        specs      (doall (for [[arglist & body] fn-specs]
                            (trace-fn-spec arglist body trace-data)))]
    `(def ~name (fn ~@specs))))

(defmethods trace-form ['defn `defn 'defn- `defn-]
  [form]
  (trace-defn form))


;;;; defmethod

(defn trace-defmethod
  [[op multifn dispatch-val & [arglist & body] :as form]]
  (let [trace-data `{:op           '~op
                     :form         '~form
                     :ns           '~(.-name *ns*)
                     :name         '~multifn
                     :dispatch-val '~dispatch-val
                     :arglist      '~arglist}]
    `(defmethod ~multifn ~dispatch-val
       ~@(trace-fn-spec arglist body trace-data))))

(defmethods trace-form ['defmethod]
  [form]
  (trace-defmethod form))


;; ---------------------------------------------------------------------
;; Protocol specs

(def skip-protocol?
  ;; Tracing IPrintWithWriter can result in situations where the maximum
  ;; call stack is exceeded, so we avoid it.
  '#{IPrintWithWriter})

(defn trace-protocol-spec
  [spec-form trace-data]
  (let [[name arglist & body] spec-form
        trace-data (assoc trace-data
                          :name `'~name
                          :form `'~spec-form
                          :arglist `'~arglist)]
    (cons name (trace-fn-spec arglist body trace-data))))

(defn trace-protocol-specs
  [protocol-specs trace-data]
  (let [impls (partition-all 2 (partition-by symbol? protocol-specs))]
    (doall (mapcat
            (fn [protos+specs]
              (let [protos (first protos+specs)
                    proto  (last protos)
                    specs  (second protos+specs)]
                (if (skip-protocol? proto)
                  `(~@protos ~@specs)
                  (let [trace-data (assoc trace-data
                                          :protocol `'~(resolve-sym proto *env*))
                        specs      (doall (for [spec specs]
                                            (trace-protocol-spec spec trace-data)))]
                    `(~@protos ~@specs)))))
            impls))))


;;;; reify

(defmethods trace-form ['reify]
  [[op & body :as form]]
  (let [trace-data `{:op   '~op
                     :form '~form
                     :ns   '~(.-name *ns*)}]
    `(~op ~@(trace-protocol-specs body trace-data))))


;;;; extend-type

(defmethods trace-form ['extend-type]
  [[op type & specs :as form]]
  (let [trace-data `{:op   '~op
                     :form '~form
                     :ns   '~(.-name *ns*)}]
    `(~op ~type ~@(trace-protocol-specs specs trace-data))))


;;;; extend-protocol

(defmethods trace-form ['extend-protocol]
  [[op proto & specs :as form]]
  (let [trace-data `{:op   '~op
                     :form '~form
                     :ns   '~(.-name *ns*)}
        fake-specs (trace-protocol-specs
                    (for [x specs]
                      (if (symbol? x)
                        proto
                        x))
                    trace-data)
        real-specs (loop [fake-specs fake-specs
                          types      (filter symbol? specs)
                          new-specs  []]
                     (case [(boolean (seq fake-specs))
                            (boolean (seq types))]
                       [true true]
                       (let [x (first fake-specs)]
                         (if (symbol? x)
                           (recur (next fake-specs)
                                  (next types)
                                  (conj new-specs (first types)))
                           (recur (next fake-specs)
                                  types
                                  (conj new-specs (first fake-specs)))))

                       [true false]
                       (seq (into new-specs fake-specs))

                       :else
                       (seq new-specs)))]
    `(~op ~(resolve-sym proto *env*) ~@real-specs)))


;;; deftype

;; NOTE: Unfortunately, there does not seem to be a way to trace object
;; construction. This is apparently due to the way JavaScript's `new`
;; operator behaves.

(defmethods trace-form ['deftype]
  [[op tsym fields & specs :as form]]
  (let [trace-data `{:op   '~op
                     :form '~form
                     :ns   '~(.-name *ns*)}
        new-specs  (trace-protocol-specs specs trace-data)]
    `(~op ~tsym ~fields ~@new-specs)))


;;; defrecord

;; NOTE: We only trace the protocols which were implemented by the
;; programmer not the ones provided by defrecord. Perhaps this could
;; be configured?

(defmethods trace-form ['defrecord]
  [[op tsym fields & specs :as form]]
  (let [trace-data `{:op   '~op
                     :form '~form
                     :ns   '~(.-name *ns*)}
        new-specs  (trace-protocol-specs specs trace-data)]
    `(~op ~tsym ~fields ~@new-specs)))


(comment
 ;; --------------------------------------------------------------------
 ;; Threading macros
 (comment
  (defmacro *->
    "Traced version of ->"
    [orig-x & orig-forms]
    (loop [x orig-x, x-print orig-x, forms orig-forms]
      (if forms
        (let [form           (first forms)
              threaded       (if (seq? form)
                               (with-meta `(~(first form) ~x ~@(next form)) (meta form))
                               (list form x))
              threaded-print (gen-log-threading-diff x-print threaded form)]
          (recur threaded threaded-print (next forms)))
        `(do
           ~(gen-log-threading-header "->" orig-x &env)
           (dlog ~orig-x ~(str orig-x))
           (let [x# ~x-print]
             (log-exit x#)
             (group-end)
             x#)))))

  (defmacro ->
    "Threads the expr through the forms. Inserts x as the
    second item in the first form, making a list of it if it is not a
    list already. If there are more forms, inserts the first form as the
    second item in second form, etc."
    {:added "1.0"}
    [x & forms]
    (loop [x x, forms forms]
      (if forms
        (let [form     (first forms)
              threaded (if (seq? form)
                         (with-meta `(~(first form) ~x ~@(next form)) (meta form))
                         (list form x))]
          (recur threaded (next forms)))
        x))))


 (defn ->trace*
   [[op orig-x & orig-forms :as all-forms]]
   (let [trace-data `{:op '~op
                      :ns '~(.-name *ns*)}
         orig-x-sym (gensym "orig-x")
         val-sym    (gensym)]
     `(let [~orig-x-sym ~orig-x]
        (binding [*threading-step-stack (atom (list))]
          ~(loop [x orig-x-sym, forms orig-forms, step-headers (rest all-forms), first-iteration? true]
             (if forms
               (let [form     (first forms)
                     x        (trace-body x {:form `'~(if (or (not *nested*)
                                                              (not first-iteration?))
                                                        (first step-headers)
                                                        val-sym)
                                             ;; TODO make sure here that traced code isn't printed
                                             :op   :threading-step})

                     ;:op-attr (when (= n (count orig-forms))
                     ;           :inner)})
                     threaded (if (seq? form)
                                (let [threaded `(~(first form) ~x ~@(next form))]
                                  (if (contains? threading-macros (first threaded))
                                    (with-meta (binding [*nested* true]
                                                 `(binding [*threading-step-stack (atom (list))]
                                                    ~(trace-form threaded)))
                                               (meta form))
                                    (with-meta threaded (meta form))))
                                (list form x))]
                 (recur threaded (next forms) (next step-headers) false))
               (trace-body x
                           (merge trace-data
                                  {:form      `'(~op ~orig-x ~@orig-forms)
                                   :last-form `'~(last step-headers)
                                   :op        :threading-step
                                   :op-attr   :outer
                                   ;; TODO Use different key, init is for something else
                                   :init      `'(~op ~(if *nested* val-sym orig-x) ~@orig-forms)}))))))))


 (comment
  (defn ->trace
    [form]
    (if (and (seq? form)
             (symbol? (first form))
             (contains? threading-macros (first form)))
      (->trace* form)
      form)
    (->trace* `(~op ~@(walk/postwalk (fn [form]
                                       (if (and (seq? form)
                                                (symbol? (first form))
                                                (contains? threading-macros (first form)))
                                         (trace-form form)
                                         form))
                                     forms)))))

 (defmethods trace-form ['->]
   [form]
   (->trace* form))




 ;; TODO: Implement

 (defn trace*->>
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['->>]
   [form]
   (trace*->> form))


 (defn trace*as->
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['as->]
   [form]
   (trace*as-> form))


 (defn trace*cond->
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['cond->]
   [form]
   (trace*cond-> form))


 (defn trace*cond->>
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['cond->>]
   [form]
   (trace*cond->> form))


 (defn trace*some->
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['some->]
   [form]
   (trace*some-> form))


 (defn trace*some->>
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['some->>]
   [form]
   (trace*some->> form))


 (defn trace-case
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['case]
   [form]
   (trace-case form))


 (defn trace-cond
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['cond]
   [form]
   (trace-cond form))


 (defn trace-if
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['if]
   [form]
   (trace-if form))


 (defn trace-if-not
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['if-not]
   [form]
   (trace-if-not form))


 (defn trace-if-let
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['if-let]
   [form]
   (trace-if-let form))


 (defn trace-if-some
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['if-some]
   [form]
   (trace-if-some form))


 (defn trace-when
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['when]
   [form]
   (trace-when form))


 (defn trace-when-not
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['when-not]
   [form]
   (trace-when-not form))


 (defn trace-when-let
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['when-let]
   [form]
   (trace-when-let form))


 (defn trace-when-some
   [[op x & forms :as form]]
   (let [trace-data `{:op      '~op
                      :form    '~form
                      :ns      '~(.-name *ns*)
                      :context '~(meta form)}]
     (trace-body form trace-data)))

 (defmethods trace-form ['when-some]
   [form]
   (trace-when-some form)))

