;; Copyright (c) George Lipov. All rights reserved.
;; The use and distribution terms for this software are covered by the
;; Eclipse Public License 2.0 (https://choosealicense.com/licenses/epl-2.0/)
;; which can be found in the file LICENSE at the root of this distribution.
;; By using this software in any fashion, you are agreeing to be bound by
;; the terms of this license.
;; You must not remove this notice, or any other, from this software.

(ns ghostwheel.tracer
  (:require [clojure.walk :refer [prewalk walk]]
            [clojure.pprint :as ppr :refer [pprint]]
            [clojure.set :as set]
            [clairvoyant.core
             :refer [ITraceEnter ITraceError ITraceExit *threading-step-stack]
             :include-macros true]
            [devtools.prefs]
            [ghostwheel.logging :as logging
             :refer [log dlog group group-collapsed group-end log-exit]]
            [clojure.string :as string]))


(def *inside-let (atom false))


(defn- trace-enter
  [data
   {:keys [color background prefix suffix expand] :as options}]
  (let [{:keys [anonymous? named? arglist args
                dispatch-val form init name ns op op-attr protocol]} data
        binding-group (if (contains? expand :bindings)
                        group
                        group-collapsed)
        ;init          (if (and (seq? init)
        ;                       (= (first init) 'try)
        ;                       (seq? (second init))
        ;                       (-> (second init)
        ;                           first
        ;                           str
        ;                           (string/starts-with? "ghostwheel.threading-macros/*")))
        ;                nil
        ;                init)
        op-sym        (symbol (cljs.core/name op))
        unnamed-fn?   (and (#{'fn 'fn*} op-sym)
                           (not named?))
        group         (if (contains? expand op-sym)
                        (if unnamed-fn?
                          group-collapsed
                          group)
                        group-collapsed)]
    (cond
      (contains? logging/fn-like-ops op)
      (let [title (if protocol
                    (str protocol " " name " " arglist)
                    (str (when prefix (str prefix " – "))
                         ns "/" (when anonymous? "__anon_") name
                         (when dispatch-val
                           (str " " (pr-str dispatch-val)))
                         (str " " arglist)))]
        (group title
               {::logging/background background
                ::logging/foreground color
                ::logging/weight     "bold"}
               80
               suffix))

      (#{'let} op)
      (do
        (reset! *inside-let true)
        (group (str op)
               {::logging/background background
                ::logging/foreground color
                ::logging/weight     "bold"}
               80
               suffix))

      (#{:binding} op)
      (let [max-length 80
            init       (when @*inside-let
                         (logging/truncate-string (str init) max-length))]
        (binding-group (str form) nil nil init)
        #_(when (> (count label) max-length)
            (group-collapsed "...")
            (log (with-out-str (pprint/pprint label)))
            (group-end)))

      ;(#{'->} op)
      ;(log (str form))

      (#{:threading-step} op)
      (do
        (if-not (#{:outer} op-attr)
          (swap! *threading-step-stack #(cons data %))
          (do
            #_(swap! *threading-step-stack #(cons (assoc data :form (:init data)) %))
            (if init
              (do
                (log (str init)))
                ;(log (str (second init))))
              (log (str form))))
          #_(swap! *threading-step-stack #(cons (assoc data :form (:init data)) %))))
      #_(do
          (if init
            (do
              (log (str init))
              (log (str (second init))))
            (log (str form)))
          #_(swap! *threading-step-stack #(cons data %))
          #_(when (#{:inner} op-attr)
              (swap! *threading-step-stack #(cons (assoc data :form (:init data)) %)))
          #_(if (#{:outer} op-attr)
              (log (str form))))

      (#{:threading-step-process} op)
      (do
        #_(dlog data :data)
        (if init
          (log (str init))
          (log (str form)))))))

;:else (log (str form)))))


(defn- trace-exit
  [{:keys [form op exit op-attr] :as data} options]
  (cond
    (#{:binding} op)
    (do (log-exit exit)
        (group-end))

    (contains? logging/complex-trace-ops op)
    (do
      (when (#{'let} op)
        (reset! *inside-let false))
      (log-exit exit)
      (group-end))

    (#{:threading-step} op)
    (do
      (if (#{:outer} op-attr)
        (trace-enter (-> data
                         (assoc :form (:last-form data))
                         (dissoc :init))
                     options)
        (trace-enter (assoc (first @*threading-step-stack) :op :threading-step-process)
                     options))
      (swap! *threading-step-stack rest)
      (log-exit exit))))

;(#{'->} op)
;(do
;  (reset! *threading-step-buffer (list))
;  (log-exit exit))))

;:else
;(log-exit [form exit])))


(defn- trace-error
  [{:keys [op form error ex-data] :as data} options]
  (cond
    (#{:binding} op)
    (do
      (error (.-stack error))
      (when ex-data
        (group-collapsed "ex-data")
        (group-collapsed ex-data)
        (group-end)
        (group-end)))

    (contains? logging/complex-trace-ops op)
    (do (group-end)
        (do
          (error (.-stack error))
          (when ex-data
            (group-collapsed "ex-data")
            (group-collapsed ex-data)
            (group-end)
            (group-end)))
        (group-end))))


(defn tracer
  "Custom Clairvoyant tracer for Ghostwheel

  Parameters:
  :color - string, Example: \"#aabbcc\"
  :background - same as color
  :prefix - string tag to display before the traced op heading
  :suffix - ...
  :expand - set of op symbols to display expanded by default. Use :bindings to expand all bindings.
  Example: #{'defn 'let :bindings}"
  [& options]
  (reify
    ITraceEnter (-trace-enter [_ data] (trace-enter data options))

    ITraceExit (-trace-exit [_ data] (trace-exit data options))

    ITraceError (-trace-error [_ data] (trace-error data options))))

