/*
 * Decompiled with CFR 0.152.
 */
package com.thinkaurelius.titan.diskstorage.log;

import com.google.common.base.Preconditions;
import com.thinkaurelius.titan.diskstorage.BackendException;
import com.thinkaurelius.titan.diskstorage.StaticBuffer;
import com.thinkaurelius.titan.diskstorage.log.Log;
import com.thinkaurelius.titan.diskstorage.log.LogManager;
import com.thinkaurelius.titan.diskstorage.log.Message;
import com.thinkaurelius.titan.diskstorage.log.MessageReader;
import com.thinkaurelius.titan.diskstorage.log.ReadMarker;
import com.thinkaurelius.titan.diskstorage.util.BufferUtil;
import com.thinkaurelius.titan.diskstorage.util.StaticArrayBuffer;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class LogTest {
    private static final Logger log = LoggerFactory.getLogger(LogTest.class);
    public static final String DEFAULT_SENDER_ID = "sender";
    private static final long TIMEOUT_MS = 30000L;
    private LogManager manager;
    @Rule
    public TestName testName = new TestName();

    public abstract LogManager openLogManager(String var1, boolean var2) throws BackendException;

    @Before
    public void setup() throws Exception {
        boolean requiresOrderPreserving = this.testName.getMethodName().toLowerCase().endsWith("serial");
        log.debug("Starting {}.{} - Order preserving {}", new Object[]{this.getClass().getSimpleName(), this.testName.getMethodName(), requiresOrderPreserving});
        this.manager = this.openLogManager(DEFAULT_SENDER_ID, requiresOrderPreserving);
    }

    @After
    public void shutdown() throws Exception {
        this.close();
        log.debug("Finished {}.{}", (Object)this.getClass().getSimpleName(), (Object)this.testName.getMethodName());
    }

    public void close() throws Exception {
        this.manager.close();
    }

    @Test
    public void smallSendReceiveSerial() throws Exception {
        this.simpleSendReceive(100, 50);
    }

    @Test
    public void mediumSendReceiveSerial() throws Exception {
        this.simpleSendReceive(2000, 1);
    }

    @Test
    public void testMultipleReadersOnSingleLogSerial() throws Exception {
        this.sendReceive(4, 2000, 5, true);
    }

    @Test
    public void testMultipleReadersOnSingleLog() throws Exception {
        this.sendReceive(4, 2000, 5, false);
    }

    @Test
    public void testReadMarkerResumesInMiddleOfLog() throws Exception {
        Log log1 = this.manager.openLog("test1");
        log1.add(BufferUtil.getLongBuffer((long)1L));
        log1.close();
        log1 = this.manager.openLog("test1");
        CountingReader count = new CountingReader(1, true);
        log1.registerReader(ReadMarker.fromNow(), new MessageReader[]{count});
        log1.add(BufferUtil.getLongBuffer((long)2L));
        count.await(30000L);
        Assert.assertEquals((long)1L, (long)count.totalMsg.get());
        Assert.assertEquals((long)2L, (long)count.totalValue.get());
    }

    @Test
    public void testLogIsDurableAcrossReopenSerial() throws Exception {
        long past = System.currentTimeMillis() - 10L;
        Log l = this.manager.openLog("durable");
        l.add(BufferUtil.getLongBuffer((long)1L));
        this.manager.close();
        l = this.manager.openLog("durable");
        l.add(BufferUtil.getLongBuffer((long)2L));
        l.close();
        l = this.manager.openLog("durable");
        CountingReader count = new CountingReader(2, true);
        l.registerReader(ReadMarker.fromTime((Instant)Instant.ofEpochMilli(past)), new MessageReader[]{count});
        count.await(30000L);
        Assert.assertEquals((long)2L, (long)count.totalMsg.get());
        Assert.assertEquals((long)3L, (long)count.totalValue.get());
    }

    @Test
    public void testMultipleLogsWithSingleReaderSerial() throws Exception {
        int i;
        int nl = 3;
        Log[] logs = new Log[3];
        CountingReader count = new CountingReader(3, false);
        for (i = 0; i < 3; ++i) {
            logs[i] = this.manager.openLog("ml" + i);
        }
        for (i = 0; i < 3; ++i) {
            logs[i].registerReader(ReadMarker.fromNow(), new MessageReader[]{count});
        }
        long value = 1L;
        for (int i2 = 0; i2 < 3; ++i2) {
            logs[i2].add(BufferUtil.getLongBuffer((long)value));
            value <<= 1;
        }
        count.await(30000L);
        Assert.assertEquals((long)3L, (long)count.totalMsg.get());
        Assert.assertEquals((long)(value - 1L), (long)count.totalValue.get());
    }

    @Test
    public void testSeparateReadersAndLogsInSharedManager() throws Exception {
        int i;
        int n = 5;
        Log[] logs = new Log[5];
        CountingReader[] counts = new CountingReader[5];
        for (i = 0; i < 5; ++i) {
            counts[i] = new CountingReader(1, true);
            logs[i] = this.manager.openLog("loner" + i);
        }
        for (i = 0; i < 5; ++i) {
            logs[i].registerReader(ReadMarker.fromNow(), new MessageReader[]{counts[i]});
            logs[i].add(BufferUtil.getLongBuffer((long)(1L << i + 1)));
        }
        for (i = 0; i < 5; ++i) {
            log.debug("Awaiting CountingReader[{}]", (Object)i);
            counts[i].await(30000L);
            Assert.assertEquals((long)(1L << i + 1), (long)counts[i].totalValue.get());
            Assert.assertEquals((long)1L, (long)counts[i].totalMsg.get());
        }
    }

    @Test
    public void testFuzzMessagesSerial() throws Exception {
        int maxLen = 4096;
        int rounds = 32;
        StoringReader reader = new StoringReader(32);
        ArrayList<StaticArrayBuffer> expected = new ArrayList<StaticArrayBuffer>(32);
        Log l = this.manager.openLog("fuzz");
        l.registerReader(ReadMarker.fromNow(), new MessageReader[]{reader});
        Random rand = new Random();
        for (int i = 0; i < 32; ++i) {
            int len = 4096;
            if (0 == len) {
                len = 1;
            }
            byte[] raw = new byte[len];
            rand.nextBytes(raw);
            StaticArrayBuffer sb = StaticArrayBuffer.of((byte[])raw);
            l.add((StaticBuffer)sb);
            expected.add(sb);
            Thread.sleep(50L);
        }
        reader.await(30000L);
        Assert.assertEquals((long)32L, (long)reader.msgCount);
        Assert.assertEquals(expected, (Object)reader.msgs);
    }

    @Test
    public void testReadMarkerCompatibility() throws Exception {
        Log l1 = this.manager.openLog("testx");
        l1.registerReader(ReadMarker.fromIdentifierOrNow((String)"mark"), new MessageReader[]{new StoringReader(0)});
        l1.registerReader(ReadMarker.fromIdentifierOrTime((String)"mark", (Instant)Instant.now().minusMillis(100L)), new MessageReader[]{new StoringReader(1)});
        try {
            l1.registerReader(ReadMarker.fromIdentifierOrNow((String)"other"), new MessageReader[0]);
            Assert.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        try {
            l1.registerReader(ReadMarker.fromTime((Instant)Instant.now().minusMillis(100L)), new MessageReader[0]);
            Assert.fail();
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        l1.registerReader(ReadMarker.fromNow(), new MessageReader[]{new StoringReader(2)});
    }

    @Test
    public void testUnregisterReaderSerial() throws Exception {
        Log log = this.manager.openLog("test1");
        CountingReader reader1 = new CountingReader(1, true);
        CountingReader reader2 = new CountingReader(2, true);
        log.registerReader(ReadMarker.fromNow(), new MessageReader[]{reader1, reader2});
        log.add(BufferUtil.getLongBuffer((long)1L));
        reader1.await(30000L);
        log.unregisterReader((MessageReader)reader1);
        log.add(BufferUtil.getLongBuffer((long)2L));
        reader2.await(30000L);
        Assert.assertEquals((long)1L, (long)reader1.totalMsg.get());
        Assert.assertEquals((long)1L, (long)reader1.totalValue.get());
        Assert.assertEquals((long)2L, (long)reader2.totalMsg.get());
        Assert.assertEquals((long)3L, (long)reader2.totalValue.get());
    }

    private void simpleSendReceive(int numMessages, int delayMS) throws Exception {
        this.sendReceive(1, numMessages, delayMS, true);
    }

    public void sendReceive(int readers, int numMessages, int delayMS, boolean expectMessageOrder) throws Exception {
        Preconditions.checkState((0 < readers ? 1 : 0) != 0);
        Log log1 = this.manager.openLog("test1");
        Assert.assertEquals((Object)"test1", (Object)log1.getName());
        CountingReader[] counts = new CountingReader[readers];
        for (int i = 0; i < counts.length; ++i) {
            counts[i] = new CountingReader(numMessages, expectMessageOrder);
            log1.registerReader(ReadMarker.fromNow(), new MessageReader[]{counts[i]});
        }
        for (long i = 1L; i <= (long)numMessages; ++i) {
            log1.add(BufferUtil.getLongBuffer((long)i));
            Thread.sleep(delayMS);
        }
        for (int i = 0; i < counts.length; ++i) {
            CountingReader count = counts[i];
            count.await(30000L);
            Assert.assertEquals((String)("counter index " + i + " message count mismatch"), (long)numMessages, (long)count.totalMsg.get());
            Assert.assertEquals((String)("counter index " + i + " value mismatch"), (long)(numMessages * (numMessages + 1) / 2), (long)count.totalValue.get());
            Assert.assertTrue((boolean)log1.unregisterReader((MessageReader)count));
        }
        log1.close();
    }

    private static class StoringReader
    extends LatchMessageReader {
        private List<StaticBuffer> msgs = new ArrayList<StaticBuffer>(64);
        private volatile int msgCount = 0;

        StoringReader(int expectedMessageCount) {
            super(expectedMessageCount);
        }

        @Override
        public void processMessage(Message message) {
            StaticBuffer content = message.getContent();
            this.msgs.add(content);
            ++this.msgCount;
        }
    }

    private static class CountingReader
    extends LatchMessageReader {
        private static final Logger log = LoggerFactory.getLogger(CountingReader.class);
        private final AtomicLong totalMsg = new AtomicLong(0L);
        private final AtomicLong totalValue = new AtomicLong(0L);
        private final boolean expectIncreasingValues;
        private long lastMessageValue = 0L;

        private CountingReader(int expectedMessageCount, boolean expectIncreasingValues) {
            super(expectedMessageCount);
            this.expectIncreasingValues = expectIncreasingValues;
        }

        @Override
        public void processMessage(Message message) {
            StaticBuffer content = message.getContent();
            Assert.assertEquals((long)8L, (long)content.length());
            long value = content.getLong(0);
            log.info("Read log value {} by senderid \"{}\"", (Object)value, (Object)message.getSenderId());
            if (this.expectIncreasingValues) {
                Assert.assertTrue((String)("Message out of order or duplicated: " + this.lastMessageValue + " preceded " + value), (this.lastMessageValue < value ? 1 : 0) != 0);
                this.lastMessageValue = value;
            }
            this.totalMsg.incrementAndGet();
            this.totalValue.addAndGet(value);
        }
    }

    private static class LatchMessageReader
    implements MessageReader {
        private final CountDownLatch latch;

        LatchMessageReader(int expectedMessageCount) {
            this.latch = new CountDownLatch(expectedMessageCount);
        }

        public final void read(Message message) {
            Assert.assertNotNull((Object)message);
            Assert.assertNotNull((Object)message.getSenderId());
            Assert.assertNotNull((Object)message.getContent());
            Instant now = Instant.now();
            Assert.assertTrue((now.isAfter(message.getTimestamp()) || now.equals(message.getTimestamp()) ? 1 : 0) != 0);
            this.processMessage(message);
            this.latch.countDown();
        }

        protected void processMessage(Message message) {
        }

        public void await(long timeoutMillis) throws InterruptedException {
            if (this.latch.await(timeoutMillis, TimeUnit.MILLISECONDS)) {
                return;
            }
            long c = this.latch.getCount();
            Preconditions.checkState((0L < c ? 1 : 0) != 0);
            String msg = "Did not read expected number of messages before timeout was reached (latch count is " + c + ")";
            log.error(msg);
            throw new AssertionError((Object)msg);
        }
    }
}

