# closmop

A Clojure library to provide capabilities similar to those of the Common Lisp Object System (CLOS). 

## Usage

closmop is available in Clojars. Add this to your Leiningen
`project.clj` `:dependencies`:

```clojure
[gov.nist/closmop "0.1.0"]
```
### Clojure does fine with minimal object-orientation, so what is the point of closmop?

Closmop can be used where you need to implement a 'language environment' for an object-oriented
language. For example, I use it to model Unified Modeling Language (UML) classes, and
run Object Constraint Language (OCL). Perhaps there are other scenarios where it might be useful.

### Examples

```clojure
;;; Classes
(defclass! MyClassA ()
  ((slot-1 :initform :a-s1) 
   (slot-2)))

(defclass! MyClassB (MyClassA)
  ((slot-3 :initarg :simple3)))

(def my-b (make-instance 'MyClassB :MyClassA.slot-2 :a-s2 :simple3 :three))

(getv my-b 'cmt/slot-1) 
=> :a-s1

;;; Metaclasses
(defclass! my-metaclass (cm/std-class)
  ((my-class-slot :initform :hello)))

(defclass! MyClassMA ()
  ((slot-a :type-hint long) 
   (slot-b :type-hint string))
  :metaclass my-metaclass)

(defclass! MyClassMB (MyClassMA)
  ((slot-c :type-hint long))
  :metaclass my-metaclass)

(def my-b (make-instance 'MyClassMB :MyClassMA.slot-a :a-s2))

(class-of (find-class (class-of my-mb))) 
=> cmt/my-metaclass

```
Setting slots (attributes, properties) is a bash-in-place operation with mutable objects. So be careful.

Closmop implements CLOS-style methods. 

```clojure
(defclass! ClassA () ())
(defclass! ClassB (ClassA) ())
(defclass! ClassC (ClassB) ())
(defgeneric! ABtest-fn [class])
(defmethod! ABtest-fn [(obj ClassA)] :a)
(defmethod! ABtest-fn [(obj ClassB)] :b)
(defmethod! ABtest-fn [(obj ClassC)] :c)

(ABtest-fn (make-instance 'ClassA))
=> :a 
(ABtest-fn (make-instance 'ClassB))
=> :b
(ABtest-fn (make-instance 'ClassC))
=> :c
```


Closmop also implements CLOS-like methods with method qualifiers (:around :before :after), 
initialization methods and with-slots. 

```clojure
(def run-order-test1 (atom []))

(defgeneric! ABqtest-fn [class])
(defmethod! ABqtest-fn [(obj ClassA)] (swap! run-order-test1 conj :do-not-run-a))
(defmethod! ABqtest-fn [(obj ClassB)] (swap! run-order-test1 conj :do-not-run-b))
(defmethod! ABqtest-fn [(obj ClassC)] (swap! run-order-test1 conj :run-c-primary) :run-c-primary)
(defmethod! ABqtest-fn :before [(obj ClassA)] (swap! run-order-test1 conj :run-a-before))
(defmethod! ABqtest-fn :before [(obj ClassB)] (swap! run-order-test1 conj :run-b-before))
(defmethod! ABqtest-fn :before [(obj ClassC)] (swap! run-order-test1 conj :run-c-before))
(defmethod! ABqtest-fn :after  [(obj ClassA)] (swap! run-order-test1 conj :run-a-after))
(defmethod! ABqtest-fn :after  [(obj ClassB)] (swap! run-order-test1 conj :run-b-after))
(defmethod! ABqtest-fn :after  [(obj ClassC)] (swap! run-order-test1 conj :run-c-after))

(ABqtest-fn (make-instance 'ClassC))

@run-order-test1 
=> [:run-a-before :run-b-before :run-c-before :run-c-primary :run-c-after :run-b-after :run-a-after]
```
See test/closmop_test.clj for more examples.

## To Do

* Ability to make the map underlying closmop objects immutable. 
* Implement Clojure-idiomatic map access using a key in the function position.


## Contacts
* Peter Denno: peter.denno@nist.gov

##Disclaimers

The use of any software or hardware by the project does not imply a recommendation or endorsement by NIST.

The use of the project results in other software or hardware products does not imply a recommendation or endorsement by NIST of those products.

We would appreciate acknowledgement if any of the project results are used, however, the use of the NIST logo is not allowed.

NIST-developed software is provided by NIST as a public service. You may use, copy and distribute copies of the software in any medium, provided that you keep intact this entire notice. You may improve, modify and create derivative works of the software or any portion of the software, and you may copy and distribute such modifications or works. Modified works should carry a notice stating that you changed the software and should note the date and nature of any such change. Please explicitly acknowledge the National Institute of Standards and Technology as the source of the software.

NIST-developed software is expressly provided “AS IS.” NIST MAKES NO WARRANTY OF ANY KIND, EXPRESS, IMPLIED, IN FACT OR ARISING BY OPERATION OF LAW, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, NON-INFRINGEMENT AND DATA ACCURACY. NIST NEITHER REPRESENTS NOR WARRANTS THAT THE OPERATION OF THE SOFTWARE WILL BE UNINTERRUPTED OR ERROR-FREE, OR THAT ANY DEFECTS WILL BE CORRECTED. NIST DOES NOT WARRANT OR MAKE ANY REPRESENTATIONS REGARDING THE USE OF THE SOFTWARE OR THE RESULTS THEREOF, INCLUDING BUT NOT LIMITED TO THE CORRECTNESS, ACCURACY, RELIABILITY, OR USEFULNESS OF THE SOFTWARE.

You are solely responsible for determining the appropriateness of using and distributing the software and you assume all risks associated with its use, including but not limited to the risks and costs of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and the unavailability or interruption of operation. This software is not intended to be used in any situation where a failure could cause risk of injury or damage to property. The software developed by NIST employees is not subject to copyright protection within the United States.
