(ns lcmap-cli.prediction
  (:require [clojure.core.async :as async]
            [clojure.stacktrace :as st]
            [lcmap-cli.config :as cfg]
            [lcmap-cli.functions :as f]
            [lcmap-cli.http :as http]
            [lcmap-cli.state :as state]))


;;
;;TODO: move to http namespace
;; requires eliminating circular dep between http -> function ns's
;;
(defn errors?
  [r]
  (or (-> r deref :error some?)
      (-> r deref http/server-error?)))


(defn handler
  [{:keys [:response :tx :ty :cx :cy :grid :month :day :acquired]}]

  (if (errors? response)
    {:tx tx
     :ty ty
     :cx cx
     :cy cy
     :month month
     :day day
     :acquired acquired
     :error (-> response deref :error str)}
    (-> response deref http/decode :body)))

;;
;;TODO: start-consumers can be generalized for all operations
;;      abstract out function to be executed
;;
(defn start-consumers
  [number retry-limit retry-delay in-chan out-chan]
  (dotimes [_ number]
    (async/thread
      (while (true?  @state/run-threads?)
        (let [func   (f/with-retry retry-limit retry-delay f/predict errors?)
              input  (async/<!! in-chan)
              result (handler (assoc input :response (func input)))]
          (async/>!! out-chan result))))))

(defn tile
  [{g :grid t :tile m :month d :day a :acquired :as all}]
  (let [config      ((keyword g) cfg/grids)
        chunk-size  (:prediction-instance-count config)
        sleep-for   (:prediction-sleep-for config)
        retry-delay (:prediction-retry-delay config)
        retry-limit (:prediction-retry-limit config)
        txy         (f/tile-to-xy (assoc all :dataset "ard"))
        xys         (f/chips (assoc all :dataset "ard"))
        in-chan     (async/chan)
        out-chan    (async/chan)
        consumers   (start-consumers chunk-size retry-limit retry-delay in-chan out-chan)]

    (async/go (doseq [xy xys]
                (Thread/sleep sleep-for)
                (async/>! in-chan {:tx (:x txy)
                                   :ty (:y txy)
                                   :cx (:cx xy)
                                   :cy (:cy xy)
                                   :month m
                                   :day d
                                   :acquired a
                                   :grid g})))
  
    (dotimes [i (count xys)]
      (f/output (async/<!! out-chan))))  
    all)
      
  
(defn chip
  [{:keys [:grid :tx :ty :cx :cy :month :day :acquired] :as all}]
  (handler (assoc all :response (f/predict all))))
