(ns lcmap-cli.products
  (:require [clojure.core.async :as async]
            [clojure.math.combinatorics :as combo]
            [clojure.string :as string]
            [cheshire.core :as json]
            [lcmap-cli.state :as state]
            [lcmap-cli.config :as cfg]
            [lcmap-cli.http :as http]
            [lcmap-cli.functions :as f]
            [lcmap-cli.functions :refer [chips tile-to-xy]]))

;;
;;TODO: move to http namespace
;; requires eliminating circular dep between http -> function ns's
;;
(defn errors?
  [r]
  (or (-> r deref :error some?)
      (-> r deref http/server-error?)))

(defn handler
  [http_response params]

  (if (errors? http_response)
    (merge {:error (-> http_response deref :error str)} params)
    (-> http_response deref http/decode :body)))

(defn post-request
  [{grid :grid resource :resource http-options :http-options :as all}]
  (let [json_body (json/encode (dissoc all :http-options))
        headers {"Content-Type" "application/json"}]
    (http/client :post 
                 (keyword grid) :ccdc (keyword resource)
                 (merge {:body json_body :headers headers} http-options))))

(defn start-consumers
  ([number retry-limit retry-delay in-chan out-chan http-options]
   (dotimes [_ number]
     (async/thread
       (while (true? @state/run-threads?)
         (let [func   (f/with-retry retry-limit retry-delay post-request errors?)
               input  (async/<!! in-chan)
               params (assoc input :http-options http-options)
               result (handler (func params) params)]
           (async/>!! out-chan result))))))
  ([number retry-limit retry-delay in-chan out-chan]
   (start-consumers number retry-limit retry-delay in-chan out-chan cfg/http-options)))

(defn date-range
  [{grid :grid years :years :as all}]
  (let [year_coll (string/split years #"/")
        start (-> year_coll first read-string)
        stop  (-> year_coll last read-string inc)
        year_range (range start stop)
        mmdd (cfg/product-mmdd grid)]
    (map (fn [i] (str i "-" mmdd)) year_range)))

(defn chip
  [{grid :grid names :names years :years cx :cx cy :cy :as all}]
  (let [tile      (f/xy-to-tile {:grid grid :dataset "ard" :x cx :y cy}) 
        date-coll (date-range all)
        product-coll (string/split names #",")
        req-args  (hash-map :grid grid :tile tile :cx cx :cy cy :dates date-coll :products product-coll :resource "product" :http-options cfg/http-options) 
        response  (post-request req-args)
        output    (handler response req-args)]
    (f/output output)
    output))

(defn product
  [{grid :grid tile :tile names :names years :years :as all}]
  (let [config       ((keyword grid) cfg/grids)
        chunk-size   (:product-instance-count config)
        retry-delay  (:product-retry-delay config)
        retry-limit  (:product-retry-limit config)
        in-chan      (async/chan)
        out-chan     (async/chan)
        xys          (chips (assoc all :dataset "ard"))
        date-coll    (date-range all)
        product-coll (string/split names #",")
        consumers    (start-consumers chunk-size retry-limit retry-delay in-chan out-chan)]

    (async/go (doseq [xy xys]
                (async/>! in-chan {:grid grid
                                   :tile tile
                                   :cx (:cx xy)
                                   :cy (:cy xy)
                                   :dates date-coll
                                   :products product-coll
                                   :resource "product"})))

    (dotimes [i (count xys)]
      (f/output (async/<!! out-chan))))
  all)

(defn raster
  [{grid :grid tile :tile names :names years :years :as all}]
  (let [config         ((keyword grid) cfg/grids)
        chunk-size     (:raster-instance-count config)
        retry-delay    (:raster-retry-delay config)
        retry-limit    (:raster-retry-limit config)
        in-chan        (async/chan)
        out-chan       (async/chan)
        xys            (chips (assoc all :dataset "ard"))
        {tx :x ty :y}  (tile-to-xy (assoc all :dataset "ard"))
        dates          (date-range all)
        products       (string/split names #",")
        products-dates (combo/cartesian-product products dates)
        consumers      (start-consumers chunk-size retry-limit retry-delay in-chan out-chan {:timeout 1800000})]


    (async/go (doseq [pd products-dates]
                (async/>! in-chan {:grid grid
                                   :tile tile
                                   :tx tx
                                   :ty ty
                                   :chips xys
                                   :date (last pd)
                                   :product (first pd)
                                   :resource "raster"})))

    (dotimes [i (count products-dates)]
      (f/output (dissoc (async/<!! out-chan) :chips))))
  all)

(defn bundle
  [{grid :grid tile :tile years :years :as all}]
  (let [config        ((keyword grid) cfg/grids)
        chunk-size    (:bundle-instance-count config)
        retry-delay   (:bundle-retry-delay config)
        retry-limit   (:bundle-retry-limit config)
        in-chan       (async/chan)
        out-chan      (async/chan)
        dates         (date-range all)
        {tx :x ty :y} (tile-to-xy (assoc all :dataset "ard"))
        consumers     (start-consumers chunk-size retry-limit retry-delay in-chan out-chan {:timeout 1800000})]

    (async/go (doseq [date dates]
                (async/>! in-chan {:grid grid
                                   :tile tile
                                   :tx tx
                                   :ty ty
                                   :date date
                                   :resource "bundle"})))


    (dotimes [i (count dates)]
      (f/output (async/<!! out-chan))))
  all)
