(ns grafter.extra.sparql
  (:require [clojure.string :as string]
            [selmer.parser :as selmer]))


(defn- column-names-seq
  "Given an alphabet string generate a lazy sequences of column names
  e.g.
  `(column-names-seq \"abcdefghijklmnopqrstuvwxyz\") ;; => (\"a\" \"b\" \"c\" ... \"aa\" \"ab\")`"
  [alphabet]
  (->> (map str alphabet)
       (iterate (fn [chars]
                  (for [x chars
                        y alphabet]
                    (str x y))))
       (apply concat)))

(defn- alphabetical-column-names
  "Returns an infinite sequence of alphabetized column names.  If more
  than 26 are required the sequence will count AA AB AC ... BA BB BC
  ... ZZZA ... etc"
  []
  (column-names-seq "abcdefghijklmnopqrstuvwxyz"))

(defn to-var [string]
  (str "?" string))

(defn to-label [var]
  (str var "_label"))

(defn alphabetical-vars-for [coll]
  (->> (alphabetical-column-names)
       (take (count coll))
       (map to-var)))

(defn bgp
  ([predicate object]
   (str predicate " " object "; "))
  ([subject predicate object]
   (str subject " " predicate " " object " .")))

(defn join-comma [& args]
  (string/join ", " args))

(defn join-space [& args]
  (string/join " " args))

(defn join-newline [& args]
  (string/join "\n" args))

(defn prefix-space [arg]
  (str "  " arg))

(defn indent [arg]
  (map prefix-space arg))

(defn sparql-uri [uri]
  (str "<" uri ">"))

(def to-uri sparql-uri)

(defn object-list [obj-or-list]
  (if (or (vector? obj-or-list) (seq? obj-or-list))
    (apply join-comma obj-or-list)
    obj-or-list))

(defn verb-object-list [[verb obj-list]]
  (str (join-space verb (object-list obj-list)) ";"))

(defn property-list [prop-list]
  (map verb-object-list prop-list))

(defn triples-same-subject [subj prop-list]
  (concat [subj]
          (indent (property-list prop-list))
          ["  ."]))

(defn group-graph-pattern [subj prop-list]
  (apply join-newline
         (concat ["{"]
                 (indent (triples-same-subject subj prop-list))
                 ["}"])))

(def ggp group-graph-pattern)

(defn template
  [filename]
  (partial selmer/render-file filename))
