(ns genesis.providers.aws.iam
  (:require [amazonica.aws.identitymanagement :as iam]
            [clojure.spec.alpha :as s]
            [genesis.core :as g :refer [defresource]]
            [genesis.specs :as gs]
            [genesis.util :refer (validate! instrument-ns)]
            [genesis.providers.aws.ec2 :as gec2]
            [ring.util.codec :as codec]))

(s/def :policy/Version #{"2012-10-17"})
(s/def :policy/Statement coll?)
(s/def :policy/Id string?)

(s/def :role/assume-role-policy-document (s/keys :req-un [:policy/Version :policy/Statement] :opt-un [:policy/Id])) ;; defines who/what is allowed to take on the role

(s/def ::policy (s/keys :req-un [:policy/Version :policy/Statement :policy/Id])) ;; defines permissions of the role

(s/def :role/role-name string?)

(s/def ::role (s/keys :req-un [:role/assume-role-policy-document
                               :role/role-name]
                      :opt-un [:role/policy]))

(s/fdef create-role :args (s/cat :c ::gs/context :i ::gs/instance))
(defn create-role [context instance]
  (let [creds (-> context :aws :creds)
        _ (assert creds)
        role (:properties instance)
        role (update-in role [:assume-role-policy-document] g/->json)]
    (println "create-role:" role)
    (let [created (:role (iam/create-role creds role))]
      (when (:policy role)
        (let [req {:role-name (:role-name role)
                   :policy-name (-> role :policy :Id)
                   :policy-document (-> role :policy g/->json)}]
          (iam/put-role-policy creds req)))
      {:resource :iam/role
       :identity (:role-id created)
       :properties created})))

(defn list-with-marker
  "For an list API call that uses :is-truncated and {:marker}, auto-paginate.

  Takes f, a fn that takes zero or 1 argument, the marker

  arguments: result-key, the key in the response that contains the seq of data"
  [f {:keys [result-key]}]
  {:pre [(keyword? result-key)]}
  (let [thunk (fn thunk [& [marker]]
                (let [resp (if marker
                             (f marker)
                             (f))
                      data (get resp result-key)]
                  (if (:is-truncated resp)
                    (lazy-cat data (thunk (:marker resp)))
                    data)))]
    (thunk)))

(s/fdef list-role :args (s/cat :c ::gs/context) :ret (s/coll-of ::gs/instance))
(defn list-roles [context]
  (let [creds (-> context :aws :creds)
        _ (assert creds)
        resp (list-with-marker (fn [& [marker]]
                                 (let [args (if marker
                                              [creds {:marker marker}]
                                              [creds])]
                                   (->> (apply iam/list-roles args)))) {:result-key :roles})]
    (->>
     resp
     (map (fn [i]
            {:resource :iam/role
             :identity (get i :role-id)
             :properties (assoc i :assume-role-policy-document (-> i :assume-role-policy-document codec/url-decode g/<-json))})))))

(s/fdef get-role :args (s/cat :c ::gs/context :i ::gs/identity) :ret (s/nilable ::gs/instance))
(defn get-role [context identity]
  (->> (list-roles context)
       (filter (fn [r]
                 (= identity (:identity r))))
       first))

(defn delete-role [context identity]
  (validate! ::gs/identity identity)
  (let [creds (-> context :aws :creds)
        instance (get-role context identity)
        role-name (-> instance :properties :role-name)
        _ (assert role-name)
        policies (:policy-names (iam/list-role-policies creds {:role-name role-name}))]
    (doseq [p policies]
      (iam/delete-role-policy creds {:role-name role-name
                                     :policy-name p}))
    (iam/delete-role creds {:role-name role-name})))

(s/fdef update-role :args (s/cat :c ::gs/context :i ::gs/managed-instance))
(defn update-role [context instance]
  (let [{role :properties name :name} instance
        role (-> role (assoc :role-name (:name role)))
        creds (-> context :aws :creds)]
    (when (:policy role)
      (iam/put-role-policy creds {:role-name identity
                                  :policy-name (-> role :policy :Id)
                                  :policy-document (-> role :policy g/->json)}))
    (when (:trust-policy role)
      (iam/update-assume-role-policy {:role-name identity
                                      :policy-document (-> role :trust-policy g/->json)}))
    (-> (get-role context (:identity instance))
        (assoc :name name))))

(defresource :iam/role {:list list-roles
                        :get get-role
                        :create create-role
                        :delete delete-role
                        :update update-role
                        })

(defn list-instance-profiles
  ([context & [marker]]
   (let [creds (-> context :aws :creds)
         resp (list-with-marker (fn [& [marker]]
                                  (let [args (if marker
                                               [creds {:marker marker}]
                                               [creds])]
                                    (apply iam/list-instance-profiles args))) {:result-key :instance-profiles})]
     (->>
      resp
      (map (fn [ip]
             {:resource :iam/instance-profile
              :identity (:instance-profile-id ip)
              :properties ip}))))))

(s/def :instance-profile/path string?)
(s/def :instance-profile/role string?)
(s/def ::instance-profile (s/keys :req-un [::instance-profile-name]
                                  :opt-un [:instance-profile/role
                                           :instance-profile/path]))

(defn maybe-get-instance-profile [context identity]
  (try
    (iam/get-instance-profile (-> context :aws :creds) {:instance-profile-name identity})
    (catch com.amazonaws.services.identitymanagement.model.NoSuchEntityException e
      nil)))

(def get-instance-profile (gec2/get-by-identity list-instance-profiles))

(defn create-instance-profile [context ip]
  (validate! ::instance-profile ip)
  (let [creds (-> context :aws :creds)
        created (:instance-profile (iam/create-instance-profile creds ip))]
    (when (:role ip)
      (iam/add-role-to-instance-profile creds {:instance-profile-name (:instance-profile-name ip)
                                               :role-name (:role ip)}))
    {:resource :iam/instance-profile
     :identity (:instance-profile-id created)
     :properties created}))

(defn update-instance-profile [context identity properties]
  (let [creds (-> context :aws :creds)
        old-role (-> (get-instance-profile creds identity) :roles first)]
    (when (not= (:role properties) old-role)
      (iam/remove-role-from-instance-profile creds {:instance-profile-name identity
                                                    :role-name old-role})
      (iam/add-role-to-instance-profile creds (iam/remove-role-from-instance-profile creds {:instance-profile-name identity
                                                                                            :role-name (:role properties)})))))

(defn delete-instance-profile [context identity]
  (validate! ::gs/identity identity)
  (let [creds (-> context :aws :creds)
        ip (get-instance-profile context identity)
        ip-name (-> ip :properties :instance-profile-name)]
    (assert ip)
    (assert ip-name)
    (doseq [r (-> ip :properties :roles)]
      (iam/remove-role-from-instance-profile creds {:instance-profile-name ip-name
                                                    :role-name (:role-name r)}))
    (iam/delete-instance-profile creds {:instance-profile-name ip-name})))

(defresource :iam/instance-profile {:list list-instance-profiles
                                    :get get-instance-profile
                                    :create create-instance-profile
                                    :delete delete-instance-profile
                                    ;; :update update-instance-profile
                                    })

(instrument-ns)
