(ns genesis.providers.aws.rds
  (:require [amazonica.aws.rds :as rds]
            [clojure.spec.alpha :as s]
            [genesis.core :as g :refer [defresource]]
            [genesis.specs :as gs]
            [genesis.util :refer [base64-encode base64-decode validate! maybe-unwrap instrument-ns unwrap!]]))

(s/def ::dbinstance-identifier string?)
(s/def ::aws-db-instance (s/keys :req-un [::dbinstance-identifier]))

(s/fdef ->db-instance :args (s/cat :i ::aws-db-instance) :ret ::gs/existing-instance)
(defn ->db-instance [i]
  {:resource :rds/database-instance
   :identity (:dbinstance-identifier i)
   :properties i})

(defn list-database-instances [context]
  (->> (rds/describe-db-instances (-> context :aws :creds))
       :dbinstances
       (map ->db-instance)))

(s/def ::allocated-storage pos-int?)
(s/def ::db-instance-class string?)
(s/def ::db-instance-identifier string?)
(s/def ::engine #{"aurora" "aurora-mysql" "aurora-postgresql" "mariadb" "mysql" "oracle-ee" "oracle-se2" "oracle-se1" "oracle-se" "postgres" "sqlserver-ee" "sqlserver-se" "sqlserver-ex" "sqlserver-web"})

(s/def ::master-user (s/and string? (partial re-find #"^\p{Alpha}.{1,62}$")))
(s/def ::master-user-password (s/and string? (partial re-find #"^.{8,128}$")))

(s/def :create-db-instance/properties (s/keys :req-un [::allocated-storage
                                                       ::db-instance-class
                                                       ::db-instance-identifier
                                                       ::engine
                                                       ::master-username
                                                       ::master-user-password]))

(s/def ::create-instance (s/merge ::gs/config-instance (s/keys :req-un [:create-db-instance/properties])))
(s/fdef create-database-instance :args (s/cat :c map? :p ::create-instance) :ret ::gs/managed-instance)
(defn create-database-instance [context instance]
  (-> (rds/create-db-instance (-> context :aws :creds) (:properties instance))
      ->db-instance
      (assoc :name (:name instance))))

(defn get-database-instance [context identity]
  (some->
   (rds/describe-db-instances (-> context :aws :creds) {:db-instance-identifier identity})
   first
   ->db-instance))

(defn delete-database-instance [context identity]
  (rds/delete-db-instance (-> context :aws :creds) {:db-instance-identifier identity
                                                    :skip-final-snapshot true}))

(defn update-database-instance [context identity properties]
  (rds/modify-db-instance (-> context :aws :creds) (assoc properties :db-instance-identifier identity)))

(instrument-ns)

(defresource :rds/database-instance {:list list-database-instances
                                     :create create-database-instance
                                     :get get-database-instance
                                     :delete delete-database-instance
                                     :update update-database-instance})
