(ns genesis.state
  (:refer-clojure :exclude [get ensure list update ])
  (:require [clojure.spec.alpha :as s]
            [genesis.specs :as gs]
            [genesis.util :refer [validate!]]))

(println "reloading state")

(defprotocol State
  (init- [this])
  (close- [this])
  (list- [this])
  (get- [this identity])
  (get-by-name- [this resource name])
  (create- [this instance])
  (delete- [this identity])
  (update- [this instance]))

(defn state? [x]
  (satisfies? State x))

(s/def ::state state?)

(defn list [state]
  {:pre [(validate! state? state)]
   :post [(validate! (s/coll-of ::gs/managed-instance) %)]}
  (list- state))

(defn list [state]
  {:pre [(validate! ::state state)]
   :post [(validate! (s/coll-of ::gs/existing-instance) %)]}
  (list- state))

(defn get [state identity]
  {:pre [(validate! ::state state)
         (validate! ::gs/identity identity)]
   :post [(validate! (s/nilable ::gs/managed-instance) %)
          (if %
            (= identity (:identity %))
            true)]}
  (get- state identity))

(defn get-by-name [state resource name]
  {:pre [(validate! state? state)
         (validate! ::gs/resource resource)
         (validate! ::gs/name name)]
   :post [(validate! (s/nilable ::gs/managed-instance) %)]}
  (get-by-name- state resource name))

(defn create [state instance]
  {:pre [(validate! ::state state)
         (validate! ::gs/managed-instance instance)
         (not (get state (:identity instance)))
         (not (get-by-name state (:resource instance) (:name instance)))]}
  (let [instance (assoc instance ::create-dependencies (:dependencies instance))]
    (create- state instance)))

(defn ensure [state instance]
  (when-not (get state (:identity instance))
    (create state instance)))

(defn delete [state identity]
  {:pre [(validate! state? state)
         (s/valid? ::gs/identity identity)]}
  (delete- state identity))

(defn same-instance? [a b]
  (= (select-keys a [:name :resource])
     (select-keys b [:name :resource])))

(defn update [state instance]
  {:pre [(validate! ::gs/managed-instance instance)
         (let [old-inst (get state (:identity instance))]
           (and old-inst (same-instance? old-inst instance)))]
   :post []}
  (update- state instance))
