(ns genesis.providers.aws.ec2
  (:require [amazonica.aws.ec2 :as ec2]
            [clojure.spec.alpha :as s]
            [genesis.core :as g :refer [defresource]]
            [genesis.specs :as gs]
            [genesis.util :refer [base64-encode base64-decode validate! unwrap! maybe-unwrap]]))

(defn map->tags [m]
  (->> m
       (mapcat (fn [[k v]]
                 [{:key (name k) :value v}]))))

(defn tags->map [tags]
  (->>  tags
        (map (fn [t]
               (assert (:key t))
               (assert (contains? t :value))
               [(keyword (:key t)) (:value t)]))
        (into {})))

(s/def :tag/key string?)
(s/def :tag/value any?)
(s/def ::tag (s/keys :req-un [:tag/key :tag/value]))
(s/def ::tags (s/coll-of ::tag))

(defn tag-resource [context resource-id tags]
  (validate! string? resource-id)
  (validate! ::tags tags)
  (ec2/create-tags (-> context :aws :creds) {:resources [resource-id]
                                             :tags tags}))

(defn untag-resource [context resource-id tags]
  (validate! string? resource-id)
  (validate! ::tags tags)
  (ec2/delete-tags (-> context :aws :creds) {:resources [resource-id]
                                             :tags tags}))

(s/def ::resource-type string?)
(s/def ::key string?)
(s/def ::val string?)
(s/fdef find-by-tag :args (s/cat :c :aws/creds :filters (s/keys :opt-un [::resource-type ::key ::val])))
(defn find-by-tag
  "Returns a seq of resources w/ matching tags. Returns all tags."
  [context {:keys [resource-type key val] :as filters}]
  (->>
   (ec2/describe-tags (-> context :aws :creds)
                      {:filters (->>
                                 [(when resource-type
                                    {:name "resource-type"
                                     :values [resource-type]})
                                  (when key
                                    {:name "key"
                                     :values [key]})
                                  (when val
                                    {:name "value"
                                     :values [val]})]
                                 (filterv identity))})
   :tags))

(defn get-by-tag
  "Find the first instance w/ matching tags. Asserts 0 or 1 result, and returns it or nil"
  [context {:keys [resource-type key val] :as filters}]
  (let [resp (find-by-tag context filters)]
    (when (> (count resp) 1)
      (throw (ex-info "get-by-tag non-unique response" filters)))
    (first resp)))

(defn get-user-data [context instance]
  (let [instance-id (:instance-id instance)]
    (assert (string? instance-id))
    (-> (ec2/describe-instance-attribute
         (-> context :aws :creds)
         {:instance-id instance-id :attribute "userData"})
        :instance-attribute
        :user-data
        base64-decode)))

(defn set-user-data [context instance])

(defn list-instances [context]
  (->> (ec2/describe-instances (-> context :aws :creds))
       :reservations
       (mapcat :instances)
       (map (fn [i]
              {:resource :ec2/instance
               :identity (:instance-id i)
               :properties (assoc i
                                  :user-data (get-user-data context i))}))))

(defn get-instance [context identity]
  (->> (list-instances context)
       (filter (fn [i]
                 (= identity (:identity i))))
       (first)))

(defn with-base64-encode
  [obj path]
  (update-in obj path (fn [s]
                        (when s
                          (base64-encode s)))))

(s/def ::image-id string?)
(s/def ::instance-id string?)
(s/def ::instance-type string?)
(s/def ::min-count pos-int?)
(s/def ::max-count pos-int?)
(s/def ::key-name string?)
(s/def ::create-properties (s/keys :req-un [::image-id ::instance-type ::key-name] :opt-un [::min-count ::max-count]))

(s/fdef create-instance :args (s/cat :c ::creds :i ::instance))
(defn create-instance [context properties]
  (validate! ::create-properties properties)
  (ec2/run-instances (-> context :aws :creds) (with-base64-encode properties [:user-data])))

(s/def ::delete-instance (s/keys :req-un [::instance-id ::gs/identity ::g/resource]))
(s/fdef delete-instance :args (s/cat :c :aws/creds :i ::delete-instance))
(defn delete-instance [context identity]
  (when-let [i (get-instance context identity)]
    (ec2/terminate-instances (-> context :aws :creds) {:instance-ids [(:instance-id i)]})))

(defresource :ec2/instance {:list list-instances
                            :get get-instance
                            :create create-instance
                            :delete delete-instance})

(defn list-key-pairs [context]
  (->> (ec2/describe-key-pairs (-> context :aws :creds))
       unwrap!
       (map (fn [kp]
              {:resource :ec2/key-pair
               :identity (:key-fingerprint kp)
               :properties kp}))))

(defn get-key-pair [context identity]
  (->> (list-key-pairs context)
       (filter (fn [kp]
                 (= identity (:identity kp))))
       first))

(s/def ::key-pair (s/keys :req-un [::key-name ::public-key-material]))

(defn create-key-pair [context properties]
  (let [creds (-> context :aws :creds)
        kp (-> properties
               (with-base64-encode [:public-key-material]))]
    (validate! ::key-pair kp)
    (let [resp (-> (ec2/import-key-pair creds kp) unwrap!)]
      {:resource :ec2/key-pair
       :identity (:key-fingerprint resp)
       :properties resp})))

(defn delete-key-pair [context identity]
  (let [kp (get-key-pair context identity)]
    (assert kp)
    (ec2/delete-key-pair (-> context :aws :creds) {:key-name (-> kp :properties :key-name)})))

(defresource :ec2/key-pair {:list list-key-pairs
                            :get get-key-pair
                            :create create-key-pair
                            :delete delete-key-pair})

(defn list-with-identity [f {:keys [id-key require-args?]}]
  (assert id-key)
  (fn [context]
    (let [creds (-> context :aws :creds)]
      (assert creds (format "list invalid creds:%s %s %s" context f id-key))
      (->> (if require-args?
             (f creds {})
             (f creds))
          unwrap!
          (map (fn [x]
                 (let [id-val (get x id-key)]
                   (assert id-val (format "list-with-identity key %s not found in %s" id-key x))
                   {:identity id-val
                    :properties x})))))))

(defn create-with-identity [f {:keys [spec id-key unwrap? unwrap-array?]}]
  (assert id-key)
  (fn [context properties]
    (let [creds (-> context :aws :creds)]
      (assert creds)
      (when spec
        (validate! spec properties))
      (let [resp (-> (f creds properties))
            inst (condp = unwrap?
                   true (unwrap! resp)
                   false resp
                   nil (maybe-unwrap resp))
            inst (if unwrap-array?
                   (do (assert (sequential? inst))
                       (first inst))
                   inst)]
        (assert resp (format "did not find inst in  %s: unwrap? %s unwrap-array? %s" resp unwrap? unwrap-array?))
        (let [id-val (get inst id-key)]
          (assert id-val (format "did not find %s in %s" id-key inst))
          {:identity id-val
           :properties inst})))))

(defn get-by-identity [f]
  (fn [context identity]
    (->> (f context)
         (filter (fn [i]
                   (= identity (:identity i))))
         first)))

(defn delete-by-identity [{:keys [get-fn delete-fn id-key]}]
  (validate! fn? delete-fn)
  (validate! keyword? id-key)
  (fn [context identity]
    (let [i (get-fn context identity)
          creds (-> context :aws :creds)]
      (assert i)
      (assert identity)
      (delete-fn creds {id-key identity}))))

(s/def :address/domain #{"vpc" "standard"})
(s/def ::address (s/keys :opt-un [:address/domain]))

(def list-addresses (list-with-identity ec2/describe-addresses {:id-key :allocation-id}))
(def get-address (get-by-identity list-addresses))
(def delete-address (fn [context identity]
                      (ec2/release-address (-> context :aws :creds) {:allocation-id identity})))

(defn create-address [context properties]
  (validate! ::address properties)
  (let [resp (ec2/allocate-address (-> context :aws :creds) properties)]
    {:identity (:allocation-id resp)
     :properties resp}))

(defresource :ec2/address {:list list-addresses
                           :get get-address
                           :create create-address
                           :delete delete-address
                           :create-spec ::address})


(s/def :find-ami/filters (s/map-of keyword? any?))
(s/def :find-ami/owners (s/coll-of string?))
(s/def :find-ami/image-ids (s/coll-of string?))

(s/fdef find-ami :args (s/cat :context ::g/context :opts (s/keys* :opt-un [:find-ami/filters :find-ami/owners :find-ami/image-ids])))
(defn find-ami [context {:keys [filter owner image-id] :as req}]
  (let [req (merge req
                   {:filter (mapv (fn [[k v]]
                                    [{:Name k
                                      :values [v]}]) filter)})]
    (ec2/describe-images (-> context :aws :creds) req)))
