(ns genesis.providers.aws.autoscaling.group
  (:require [amazonica.aws.autoscaling :as autoscaling]
            [clojure.core.strint :refer (<<)]
            [clojure.data :refer [diff]]
            [clojure.spec.alpha :as s]
            [clojure.set :as set]
            [genesis.core :as g :refer [defresource]]
            [genesis.util :as util :refer [validate! base64-encode base64-decode unwrap!]]))

(defn list-groups [context]
  (->> (autoscaling/describe-auto-scaling-groups (-> context :aws :creds))
       unwrap!
       (map (fn [group]
              {:resource :autoscaling/group
               :identity (:auto-scaling-group-arn group)
               :properties (-> group
                               (set/rename-keys {:vpczone-identifier :vpc-zone-identifier}))}))))

(defn get-group [context identity]
  (->> (list-groups context)
       (filter (fn [group]
                 (= identity (:identity group))))
       first))

(s/def ::auto-scaling-group-name string?)
(s/def ::availability-zone string?)
(s/def ::availability-zones (s/coll-of ::availability-zone))
(s/def ::desired-capacity nat-int?)
(s/def ::instance-id string?)
(s/def ::launch-configuration-name string?)
(s/def ::max-size nat-int?)
(s/def ::min-size nat-int?)
(s/def ::vpc-zone-identifier string?)

(s/def ::target-group-arns (s/coll-of string?))

(s/def ::auto-scaling-group (s/keys :req-un [::auto-scaling-group-name
                                             (or ::instance-id
                                                 ::launch-configuration-name)
                                             (or ::availability-zones
                                                 ::vpc-zone-identifier)
                                             ::max-size
                                             ::min-size]
                                    :opt-un [::desired-capacity
                                             ::target-group-arns ;; ALB
                                             ;;::load-balancer-name ;; ELB classic
                                             ]))

(def get-group-by-name (util/get-by-list-property {:list-fn list-groups :property-key :auto-scaling-group-name}))

(defn ensure-target-arns [context desired-properties actual-properties]
  (assert (= (:auto-scaling-group-name desired-properties) (:auto-scaling-group-name actual-properties)))
  (let [creds (-> context :aws :creds)
        desired-arns (:target-group-arns desired-properties)
        actual-arns (:target-group-arns actual-properties)
        asg-name (:auto-scaling-group-name actual-properties)
        [to-add to-remove _] (diff desired-arns actual-arns)]
    (assert asg-name)
    (when (seq to-add)
      (autoscaling/attach-load-balancer-target-groups creds {:auto-scaling-group-name asg-name
                                                           :target-group-arns to-add}))

    (when (seq to-remove)
      (autoscaling/detach-load-balancer-target-groups creds {:auto-scaling-group-name asg-name
                                                             :target-group-arns to-remove}))))

(defn create-group [context properties]
  (validate! ::auto-scaling-group properties)
  (autoscaling/create-auto-scaling-group (-> context :aws :creds) properties)
  (let [group (get-group-by-name context (:auto-scaling-group-name properties))
        _ (ensure-target-arns context properties (:properties group))
        group (get-group-by-name context (:auto-scaling-group-name properties))]
    (assert group)
    (println "create launch config:" group)
    group))

(defn delete-group [context identity]
  (let [group (get-group context identity)
        creds (-> context :aws :creds)
        group (get-group context identity)
        group-name (-> group :properties :auto-scaling-group-name)]
    (assert group-name)
    (autoscaling/update-auto-scaling-group creds {:auto-scaling-group-name group-name
                                                  :min-size 0
                                                  :desired-size 0
                                                  :max-size 0})
    (autoscaling/delete-auto-scaling-group creds {:auto-scaling-group-name group-name
                                                  :force-delete true})))



(s/def ::auto-scaling-update (s/keys :req-un [::auto-scaling-group-name]
                                     :opt-un [::max-size
                                              ::min-size
                                              ::desired-capacity]))

(defn update-group [context identity properties]
  (validate! ::auto-scaling-update properties)
  (let [creds (-> context :aws :creds)
        resp (autoscaling/update-auto-scaling-group creds properties)
        group (get-group-by-name context (:auto-scaling-group-name properties))]
    (println "autoscaling update resp:" resp)
    (ensure-target-arns context properties (:properties group))
    {:identity identity
     :properties (get-group-by-name context (:auto-scaling-group-name properties))}))

(defresource :autoscaling/group {:list list-groups
                                 :get get-group
                                 :create create-group
                                 :update update-group
                                 :delete delete-group})
