(ns genesis.providers.aws.ec2.vpc.client-vpn
  (:require [cognitect.aws.client.api :as aws]
            [clojure.spec.alpha :as s]
            [genesis.specs :as gs]
            [genesis.providers.aws :as gaws]
            [genesis.providers.aws.ec2 :as ec2]
            [genesis.util :refer [validate! instrument-ns]]
            [genesis.core :as g :refer [defresource]]))

(defn client [context]
  (gaws/client context :ec2))

(defn list-endpoints [context]
  (-> (client context)
      (aws/invoke {:op :DescribeClientVpnEndpoints})
      :ClientVpnEndpoints
      (->> (map (fn [v]
                  {:identity (:ClientVpnEndpointId v)
                   :properties v})))))

(def get-endpoint (ec2/get-by-identity list-endpoints))

(defn create-endpoint [context properties]
  (let [resp (-> (client context)
                 (gaws/invoke! {:op :CreateClientVpnEndpoint
                                :request properties}))]
    {:identity (:ClientVpnEndpointId resp)
     :properties (-> resp (dissoc :Status))}))

(defn delete-endpoint [context ident]
  (-> (client context)
      (gaws/invoke! {:op :DeleteClientVpnEndpoint
                     :request {:ClientVpnEndpointId ident}})))

(s/def ::ClientVpnEndpointId string?)
(s/def ::AssociationId string?)

(s/fdef network-association-id :args (s/cat :k (s/keys :req-un [::ClientVpnEndpointId ::AssociationId])) :ret ::gs/identity)
(defn network-association-id
  [props]
  (str (:ClientVpnEndpointId props) " " (:AssociationId props)))

(s/fdef create-network-association :args (s/cat :c ::gs/context :p ::gs/properties) :ret ::gs/provider-instance)
(defn create-network-association [context properties]
  (let [resp (-> (client context)
                 (gaws/invoke! {:op :AssociateClientVpnTargetNetwork
                                :request properties}))]
    {:identity (network-association-id resp)
     :properties (-> resp (dissoc :Status))}))

(defn list-network-associations [context]
  (let [c (client context)]
    (->> (list-endpoints context)
         (mapcat (fn [e]
                   (-> (gaws/invoke! c {:op :DescribeClientVpnTargetNetworks :request {:ClientVpnEndpointId (:identity e)}})
                       :ClientVpnTargetNetworks
                       (->>
                        (map (fn [assoc]
                               {:identity (network-association-id assoc)
                                :properties assoc})))))))))

(def get-network-association (ec2/get-by-identity list-network-associations))

(defn delete-network-association [context ident]
  (let [[_ endpoint-id assoc-id] (re-find #"([^ ]+) ([^ ]+)" ident)]
    (assert (and endpoint-id assoc-id))
    (-> (client context)
      (gaws/invoke! {:op :DisassociateClientVpnTargetNetwork
                     :request {:ClientVpnEndpointId endpoint-id
                               :AssociationId assoc-id}}))))

(s/def ::TargetNetworkCidr string?)

(s/fdef network-ingress-id :args (s/cat :p (s/keys :req-un [::ClientVpnEndpointId (or ::TargetNetworkCidr ::DestinationCidr)])) :ret string?)
(defn network-ingress-id
  [props]
  (str (:ClientVpnEndpointId props) " " (or (:TargetNetworkCidr props)
                                            (:DestinationCidr props))))

(s/fdef create-network-ingress :args (s/cat :c ::gs/context :p (s/keys :req-un [::ClientVpnEndpointId ::TargetNetworkCidr])) :ret ::gs/provider-instance)
(defn create-network-ingress [context properties]
  (-> (client context)
      (gaws/invoke! {:op :AuthorizeClientVpnIngress
                     :request properties})
      ((fn [resp]
         {:identity (network-ingress-id properties)}))))

(defn list-network-ingresses [context]
  (let [c (client context)]
    (->> (list-endpoints context)
         (mapcat (fn [e]
                   (-> (gaws/invoke! c {:op :DescribeClientVpnAuthorizationRules
                                        :request {:ClientVpnEndpointId (:identity e)}})
                       :AuthorizationRules
                       (->>
                        (map (fn [ar]
                               {:identity (network-ingress-id ar)
                                :properties ar})))))))))

(def get-network-ingress (ec2/get-by-identity list-network-ingresses))

(defn delete-network-ingress [context identity]
  (let [[_ endpoint-id cidr] (re-find #"([^ ]+) ([^ ]+)" identity)
        ingress (get-network-ingress context identity)]
    (assert (string? endpoint-id) (string? cidr))
    (-> (client context)
            (gaws/invoke! {:op :RevokeClientVpnIngress
                           :request (merge {:ClientVpnEndpointId endpoint-id
                                            :TargetNetworkCidr cidr}
                                           (if (-> ingress :properties :GroupId seq)
                                             {:GroupId (-> ingress :properties :GroupId seq)}
                                             {:RevokeAllGroups true}))}))))

(instrument-ns)

(defresource ::endpoint {:list list-endpoints
                         :get get-endpoint
                         :create create-endpoint
                         :delete delete-endpoint})

(defresource ::network-association {:list list-network-associations
                                    :create create-network-association
                                    :get get-network-association
                                    :delete delete-network-association})

(defresource ::network-ingress {:create create-network-ingress
                                :list list-network-ingresses
                                :get get-network-ingress
                                :delete delete-network-ingress})
